#include "TileObstructionView.hpp"
#include "../common/Tile.hpp"
#include "resource.h"


static int s_ViewID = 1000;


BEGIN_MESSAGE_MAP(CTileObstructionView, CScrollWindow)

  ON_WM_SIZE()
  ON_WM_PAINT()

  ON_WM_LBUTTONDOWN()
  ON_WM_LBUTTONUP()
  ON_WM_MOUSEMOVE()

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CTileObstructionView::CTileObstructionView()
: m_Tile(NULL)
, m_RenderPixel(16, 16, 32)
, m_PenDown(true)
, m_ScrollX(0)
, m_ScrollY(0)
, m_Clicked(false)
{
}

////////////////////////////////////////////////////////////////////////////////

CTileObstructionView::~CTileObstructionView()
{
}

////////////////////////////////////////////////////////////////////////////////

BOOL
CTileObstructionView::Create(sTile* tile, CWnd* parent)
{
  m_Tile = tile;

  CScrollWindow::Create(
    AfxRegisterWndClass(0, LoadCursor(NULL, MAKEINTRESOURCE(IDC_ARROW)), NULL, NULL),
    "",
    WS_CHILD | WS_VISIBLE | WS_HSCROLL | WS_VSCROLL,
    CRect(0, 0, 0, 0),
    parent,
    s_ViewID++);

  UpdateScrollBars();

  return TRUE;
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::SetDrawMode(bool pen_down)
{
  m_PenDown = pen_down;
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::Preset(int preset)
{
  switch (preset)
  {
    case TOV_PRESET_UNBLOCKED:  PresetUnblocked();  break;
    case TOV_PRESET_BLOCKED:    PresetBlocked();    break;
    case TOV_PRESET_UPPERRIGHT: PresetUpperRight(); break;
    case TOV_PRESET_LOWERRIGHT: PresetLowerRight(); break;
    case TOV_PRESET_LOWERLEFT:  PresetLowerLeft();  break;
    case TOV_PRESET_UPPERLEFT:  PresetUpperLeft();  break;
  }

  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::PresetUnblocked()
{
  for (int iy = 0; iy < m_Tile->GetHeight(); iy++)
    for (int ix = 0; ix < m_Tile->GetWidth(); ix++)
      m_Tile->SetObstructed(ix, iy, false);    
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::PresetBlocked()
{
  for (int iy = 0; iy < m_Tile->GetHeight(); iy++)
    for (int ix = 0; ix < m_Tile->GetWidth(); ix++)
      m_Tile->SetObstructed(ix, iy, true);    
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::PresetUpperRight()
{
  for (int iy = 0; iy < m_Tile->GetHeight(); iy++)
    for (int ix = 0; ix < m_Tile->GetWidth(); ix++)
      m_Tile->SetObstructed(ix, iy, ix > iy);
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::PresetLowerRight()
{
  for (int iy = 0; iy < m_Tile->GetHeight(); iy++)
    for (int ix = 0; ix < m_Tile->GetWidth(); ix++)
      m_Tile->SetObstructed(ix, iy, ix + iy >= (m_Tile->GetWidth() + m_Tile->GetHeight()) / 2);
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::PresetLowerLeft()
{
  for (int iy = 0; iy < m_Tile->GetHeight(); iy++)
    for (int ix = 0; ix < m_Tile->GetWidth(); ix++)
      m_Tile->SetObstructed(ix, iy, ix < iy);
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::PresetUpperLeft()
{
  for (int iy = 0; iy < m_Tile->GetHeight(); iy++)
    for (int ix = 0; ix < m_Tile->GetWidth(); ix++)
      m_Tile->SetObstructed(ix, iy, ix + iy + 1 < (m_Tile->GetWidth() + m_Tile->GetHeight()) / 2);
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::UpdateScrollBars()
{
  // get page size
  RECT client_rect;
  GetClientRect(&client_rect);
  int page_size_x = client_rect.right / 16;
  int page_size_y = client_rect.bottom / 16;

  SetHScrollRange(0, m_Tile->GetWidth() - 1, page_size_x);
  SetVScrollRange(0, m_Tile->GetHeight() - 1, page_size_y);
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::Click(int x, int y)
{
  int ox = x / 16 + m_ScrollX;
  int oy = y / 16 + m_ScrollY;
  if (ox >= 0 && oy >= 0 && ox < m_Tile->GetWidth() && oy < m_Tile->GetHeight())
  {
    m_Tile->SetObstructed(ox, oy, m_PenDown);
  
    // invalidate the rectangle if the state changed
    int px = (ox - m_ScrollX) * 16;
    int py = (oy - m_ScrollY) * 16;
    RECT r = { px, py, px + 16, py + 16 };
    InvalidateRect(&r);
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTileObstructionView::OnSize(UINT type, int cx, int cy)
{
  Invalidate();
  UpdateScrollBars();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTileObstructionView::OnPaint()
{
  CPaintDC dc_(this);
  HDC dc = dc_.m_hDC;

  RECT client_rect;
  GetClientRect(&client_rect);

  int num_pixels_x = client_rect.right / 16;
  int num_pixels_y = client_rect.bottom / 16;

  // we need this to draw the obstructed pixels
  HICON icon = AfxGetApp()->LoadIcon(IDI_OBSTRUCTED);

  // draw each pixel in the tile
  for (int iy = 0; iy < num_pixels_y + 1; iy++)
    for (int ix = 0; ix < num_pixels_x + 1; ix++)
    {
      // visibility check
      RECT rect = { ix * 16, iy * 16, ix * 16 + 16, iy * 16 + 16 }; 
      if (!RectVisible(dc, &rect))
        continue;

      RGBA color = rgbaBlack;
      bool obstructed = false;
      if (ix < m_Tile->GetWidth() && iy < m_Tile->GetHeight())
      {
        // draw the tile pixel color, not black
        color = m_Tile->GetPixel(ix + m_ScrollX, iy + m_ScrollY);
        obstructed = m_Tile->IsObstructed(ix + m_ScrollX, iy + m_ScrollY);
      }

      // fill the DIB we use to draw
      BGRA* render_pixels = (BGRA*)m_RenderPixel.GetPixels();
      for (int i = 0; i < 16 * 16; i++)
      {
        render_pixels[i].red   = color.red;
        render_pixels[i].green = color.green;
        render_pixels[i].blue  = color.blue;
      }

      if (obstructed)
        DrawIconEx(m_RenderPixel.GetDC(), 0, 0, icon, 16, 16, 0, NULL, DI_NORMAL);

      // render!
      BitBlt(dc, rect.left, rect.top, 16, 16, m_RenderPixel.GetDC(), 0, 0, SRCCOPY);

    }

}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTileObstructionView::OnLButtonDown(UINT flags, CPoint point)
{
  Click(point.x, point.y);
  SetCapture();
  m_Clicked = true;
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTileObstructionView::OnLButtonUp(UINT flags, CPoint point)
{
  ReleaseCapture();
  m_Clicked = false;
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTileObstructionView::OnMouseMove(UINT flags, CPoint point)
{
  if (m_Clicked)
    Click(point.x, point.y);
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::OnHScrollChanged(int x)
{
  m_ScrollX = x;
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

void
CTileObstructionView::OnVScrollChanged(int y)
{
  m_ScrollY = y;
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////
