#include "SwatchPalette.hpp"
#include "StateServer.hpp"
#include "configfile.h"
#include "resource.h"
#include "common_palettes.h" 


#define SWATCH_TILE_SIZE 10

BEGIN_MESSAGE_MAP(CSwatchPalette, CMiniFrameWnd)

  ON_WM_CLOSE()
  ON_WM_SIZE()
  ON_WM_PAINT()
  ON_WM_VSCROLL()
  ON_WM_LBUTTONDOWN()
  ON_WM_MOUSEMOVE()
  ON_WM_RBUTTONDOWN()

  ON_COMMAND(ID_SWATCH_FILE_LOAD,             OnFileLoad)
  ON_COMMAND(ID_SWATCH_FILE_SAVE,             OnFileSave)
  ON_COMMAND(ID_SWATCH_DEFAULT_DOS_PALETTE,   OnDefaultDOSPalette)
  ON_COMMAND(ID_SWATCH_DEFAULT_VERGE_PALETTE, OnDefaultVergePalette)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CSwatchPalette::CSwatchPalette(CWnd* parent, CSwatchPaletteHandler* handler, CStatusBar* status_bar)
: m_Created(false)
, m_TopRow(0)
, m_SelectedColor(0)
, m_NumColors(0)
, m_Color(NULL)
{
  m_Handler   = handler;
  m_StatusBar = status_bar;

  int left   = GetStateServer()->GetInt("SwatchPalette:left",   -1);
  int top    = GetStateServer()->GetInt("SwatchPalette:top",    -1);
  int right  = GetStateServer()->GetInt("SwatchPalette:right",  -1);
  int bottom = GetStateServer()->GetInt("SwatchPalette:bottom", -1);

  CRect start_rect(left, top, right, bottom);

  if (left == -1 && top == -1 && right == -1 && bottom == -1)
    start_rect = CRect(0, 0, 80, 80);

  Create(
    AfxRegisterWndClass(0, LoadCursor(NULL, MAKEINTRESOURCE(IDC_ARROW)), NULL, NULL),
    "swatch",
    WS_POPUP | WS_CAPTION | WS_SIZEBOX | MFS_SYNCACTIVE | MFS_THICKFRAME | MFS_BLOCKSYSMENU,
    start_rect,
    parent);

  m_Menu.LoadMenu(IDR_SWATCH);
  SetMenu(&m_Menu);

  // construct the palette array
  m_NumColors = GetStateServer()->GetInt("SwatchPalette:numcolors", -1);
  if (m_NumColors == -1)
    Generate_Colors();
  else
  {
    char SwatchString[80];
    char ColorString[80];
    int r,g,b,a;
    int i;

    m_Color = new RGBA[m_NumColors];
    for (i=0; i<m_NumColors; i++)
    {
      sprintf(SwatchString, "SwatchColor:%i", i);
      GetStateServer()->GetString(SwatchString, ColorString, 80, "");

      if (ColorString == "")
        sprintf(ColorString, "255 255 255 255");

      sscanf(ColorString, "%i%i%i%i", &r, &g, &b, &a);
      m_Color[i].red = r;
      m_Color[i].green = g;
      m_Color[i].blue = b;
      m_Color[i].alpha = a;
    }
    //MessageBox("Success!");
  }

  //Generate_Colors();
  m_Created = true;
  
  RECT rect;
  GetClientRect(&rect);
  OnSize(0, rect.right, rect.bottom);
}

////////////////////////////////////////////////////////////////////////////////

CSwatchPalette::~CSwatchPalette()
{

  if (m_Color)
  {
    delete m_Color;
    m_Color = NULL;
  }

}

////////////////////////////////////////////////////////////////////////////////

void
CSwatchPalette::UpdateColor(RGB color)
{
  m_CVcolor.red = color.red;
  m_CVcolor.green = color.green;
  m_CVcolor.blue = color.blue;
}

////////////////////////////////////////////////////////////////////////////////

void
CSwatchPalette::UpdateAlpha(byte alpha)
{
  m_CVcolor.alpha = alpha;
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSwatchPalette::OnClose()
{
  RECT rect;
  GetWindowRect(&rect);
  GetStateServer()->SetInt("SwatchPalette:left",   rect.left);
  GetStateServer()->SetInt("SwatchPalette:top",    rect.top);
  GetStateServer()->SetInt("SwatchPalette:right",  rect.right);
  GetStateServer()->SetInt("SwatchPalette:bottom", rect.bottom);
  GetStateServer()->SetInt("SwatchPalette:numcolors", m_NumColors);

  int i=0;
  char SwatchString[80];
  char ColorString[80];

  for (i=0; i<m_NumColors; i++)
  {
    sprintf(SwatchString, "SwatchColor:%i", i);
    sprintf(ColorString, "%i %i %i %i", m_Color[i].red, m_Color[i].green, m_Color[i].blue, m_Color[i].alpha);
    //MessageBox(SwatchString);

    GetStateServer()->SetString(SwatchString, ColorString);
  }
  
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSwatchPalette::OnSize(UINT type, int cx, int cy)
{
  if (cx > 0)
  {
    // if the current top row is greater than the total number of rows minus the page size
    if (m_TopRow > GetNumRows() - GetPageSize())
    {
      // move the top row up
      m_TopRow = GetNumRows() - GetPageSize();
      if (m_TopRow < 0)
        m_TopRow = 0;
      Invalidate();
    }
  }

  // reflect the changes
  UpdateScrollBar();
  Invalidate();

  CWnd::OnSize(type, cx, cy);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSwatchPalette::OnPaint()
{
  CPaintDC dc(this);
  CBrush brush;
  RECT rect;

  GetClientRect(&rect);
  
  for (int iy = 0; iy < rect.bottom / SWATCH_TILE_SIZE + 1; iy++)
    for (int ix = 0; ix < rect.right / SWATCH_TILE_SIZE + 1; ix++)
    {
      int num_colors_x = rect.right / SWATCH_TILE_SIZE;
      int ic = (iy + m_TopRow) * num_colors_x + ix;

      RECT Color_Location = {
        ix * SWATCH_TILE_SIZE,
        iy * SWATCH_TILE_SIZE,
        (ix + 1) * SWATCH_TILE_SIZE,
        (iy + 1) * SWATCH_TILE_SIZE
      };

      if (ix < num_colors_x && ic < m_NumColors)
      {
        brush.CreateSolidBrush(RGB(m_Color[ic].red, m_Color[ic].green, m_Color[ic].blue));
        dc.FillRect(&Color_Location, &brush);
        brush.DeleteObject();
      }
      else
      {
        // draw black rectangle
        dc.FillRect(&Color_Location, CBrush::FromHandle((HBRUSH)GetStockObject(BLACK_BRUSH)));
      }
    }
  
  //COLORREF cr;
  //cr = RGB(m_Color[0].red, m_Color[0].green, m_Color[0].blue);
  //brush.CreateSolidBrush(RGB(m_CVcolor.red, m_CVcolor.green, m_CVcolor.blue));
  //dc.FillRect(CRect(0,0,16,16), &brush);  

  brush.DeleteObject();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSwatchPalette::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar)
{
  switch (nSBCode)
  {
    case SB_LINEDOWN:   m_TopRow++;                break;
    case SB_LINEUP:     m_TopRow--;                break;
    case SB_PAGEDOWN:   m_TopRow += GetPageSize(); break;
    case SB_PAGEUP:     m_TopRow -= GetPageSize(); break;
    case SB_THUMBTRACK: m_TopRow = (int)nPos;       break;
  }

  // validate the values
  if (m_TopRow > GetNumRows() - GetPageSize())
    m_TopRow = GetNumRows() - GetPageSize();
  if (m_TopRow < 0)
    m_TopRow = 0;

  UpdateScrollBar();
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSwatchPalette::OnLButtonDown(UINT nFlags, CPoint point)
{
  RECT rect;
  GetClientRect(&rect);
  
  int num_colors_x = rect.right / SWATCH_TILE_SIZE;
  int ix = point.x / SWATCH_TILE_SIZE;
  int iy = point.y / SWATCH_TILE_SIZE;
  int ic = (iy + m_TopRow) * num_colors_x + ix;

  if (ic < m_NumColors)
    m_Handler->SP_ColorSelected(m_Color[ic]);
  else
    AddColor();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSwatchPalette::OnMouseMove(UINT nFlags, CPoint point)
{
  RECT rect;

  GetClientRect(&rect);
  int num_colors_x = rect.right / SWATCH_TILE_SIZE;
  int ix = point.x / SWATCH_TILE_SIZE;
  int iy = point.y / SWATCH_TILE_SIZE;
  int ic = (m_TopRow + iy) * num_colors_x + ix;

  if (ic < m_NumColors)
  {
    CString string;
    string.Format("Swatch (%i/%i): R%i, G%i, B%i, A%i", ic, m_NumColors, m_Color[ic].red, m_Color[ic].green, m_Color[ic].blue, m_Color[ic].alpha);
    
    m_StatusBar->SetWindowText(string);
  }
  else
    m_StatusBar->SetWindowText("");
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSwatchPalette::OnRButtonDown(UINT nFlags, CPoint point)
{
  RECT rect;
  GetClientRect(&rect);
  
  int num_colors_x = rect.right / SWATCH_TILE_SIZE;
  int ix = point.x / SWATCH_TILE_SIZE;
  int iy = point.y / SWATCH_TILE_SIZE;
  int ic = (iy + m_TopRow) * num_colors_x + ix;

  if (ic < m_NumColors)
    RemoveColor(ic);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSwatchPalette::OnFileLoad()
{
  CFileDialog Dialog(
    TRUE, "sswatch", NULL,
    OFN_FILEMUSTEXIST | OFN_HIDEREADONLY,
    "Sphere Swatch Files (*.sswatch)|*.sswatch|" \
    "All Files (*.*)|*.*||");

  if (Dialog.DoModal() == IDOK)
  {
    CONFIG file;
    int ver;
    int NumColors;
    RGBA* TempColor;

    CreateConfig(&file);
    LoadConfig(&file, Dialog.GetPathName());
    
    ReadConfigInt(&file, "Info", "version", &ver, -1);
    if (ver != 1)
    {
      DestroyConfig(&file);
      return;
    }

    ReadConfigInt(&file, "Data", "SwatchColor:numcolors", &NumColors, -1);
    if (NumColors != -1)
    {
      TempColor = new RGBA[NumColors];

      for (int i=0; i<NumColors; i++)
      {
        int r,g,b,a;
        char SwatchString[80];
        char ColorString[80];

        sprintf(SwatchString, "SwatchColor:%i", i);
        ReadConfigString(&file, "Data", SwatchString, ColorString, 80, "");
        if (ColorString == "")
          sprintf(ColorString, "255 255 255 255");

        sscanf(ColorString, "%i%i%i%i", &r,&g, &b, &a);
        TempColor[i].red = r;
        TempColor[i].green = g;
        TempColor[i].blue = b;
        TempColor[i].alpha = a;
      }

      m_NumColors = NumColors;
      delete[] m_Color;
      m_Color = TempColor;
    }

    DestroyConfig(&file);

    UpdateScrollBar();
    Invalidate();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSwatchPalette::OnFileSave()
{
    CFileDialog Dialog(
    FALSE, "sswatch", NULL,
    OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT,
    "Sphere Swatch Files (*.sswatch)|*.sswatch|" \
    "All Files (*.*)|*.*||");

  if (Dialog.DoModal() == IDOK)
  {
    CONFIG file;
    int ver;

    CreateConfig(&file);

    ver = 1;
    WriteConfigInt(&file, "Info", "version", ver);

    WriteConfigInt(&file, "Data", "SwatchColor:numcolors", m_NumColors);

    for (int i=0; i<m_NumColors; i++)
    {
      char SwatchString[80];
      char ColorString[80];

      sprintf(SwatchString, "SwatchColor:%i", i);
      sprintf(ColorString, "%i %i %i %i", m_Color[i].red, m_Color[i].green, m_Color[i].blue, m_Color[i].alpha);

      WriteConfigString(&file, "Data", SwatchString, ColorString);
    }

    SaveConfig(&file, Dialog.GetPathName());
    DestroyConfig(&file);
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSwatchPalette::OnDefaultDOSPalette()
{
  delete[] m_Color;
  m_NumColors = 256;
  m_Color = new RGBA[m_NumColors];
  for (int i=0; i<m_NumColors; i++)
  {
    m_Color[i].red = dos_palette[i].red;
    m_Color[i].green = dos_palette[i].green;
    m_Color[i].blue = dos_palette[i].blue;
    m_Color[i].alpha = 255;
  }

  UpdateScrollBar();
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSwatchPalette::OnDefaultVergePalette()
{
  delete[] m_Color;
  m_NumColors = 256;
  m_Color = new RGBA[m_NumColors];
  for (int i=0; i<m_NumColors; i++)
  {
    m_Color[i].red = verge_palette[i].red;
    m_Color[i].green = verge_palette[i].green;
    m_Color[i].blue = verge_palette[i].blue;
    m_Color[i].alpha = 255;
  }

  UpdateScrollBar();
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////

void
CSwatchPalette::Generate_Colors()
{
  delete[] m_Color;
  m_NumColors = 256;
  m_Color = new RGBA[m_NumColors];
  for (int i=0; i<m_NumColors; i++)
  {
    m_Color[i].red = verge_palette[i].red;
    m_Color[i].green = verge_palette[i].green;
    m_Color[i].blue = verge_palette[i].blue;
    m_Color[i].alpha = 255;
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CSwatchPalette::UpdateScrollBar()
{
  SCROLLINFO si;
  si.cbSize = sizeof(si);
  si.fMask  = SIF_ALL;
  si.nMin   = 0;

  if (GetPageSize() < GetNumRows())
  {
    si.nMax   = GetNumRows() - 1;
    si.nPage  = GetPageSize();
    si.nPos   = m_TopRow;
  }
  else
  {
    si.nMax   = 0xFFFF;
    si.nPage  = 0xFFFE;
    si.nPos   = 0;
  }

  SetScrollInfo(SB_VERT, &si);
}

////////////////////////////////////////////////////////////////////////////////

int
CSwatchPalette::GetPageSize()
{
  RECT ClientRect;
  GetClientRect(&ClientRect);
  return ClientRect.bottom / SWATCH_TILE_SIZE;
}

////////////////////////////////////////////////////////////////////////////////

int
CSwatchPalette::GetNumRows()
{
  RECT client_rect;
  GetClientRect(&client_rect);
  int num_colors_x = client_rect.right / SWATCH_TILE_SIZE;

  if (num_colors_x == 0)
    return -1;
  else
    return (m_NumColors + num_colors_x - 1) / num_colors_x;
}

////////////////////////////////////////////////////////////////////////////////

void
CSwatchPalette::AddColor()
{
  RGBA* TempColor;

  TempColor = new RGBA[m_NumColors + 1];
  memcpy(TempColor, m_Color, m_NumColors * sizeof(RGBA));
  TempColor[m_NumColors] = m_CVcolor;

  m_NumColors++;
  delete[] m_Color;
  m_Color = TempColor;

  UpdateScrollBar();
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

void
CSwatchPalette::RemoveColor(int ColorRef)
{
  RGBA* TempColor;
  int j = 0;
 
  TempColor = new RGBA[m_NumColors - 1];
  for (int i=0; i<m_NumColors; i++)
    if (i != ColorRef)
    {
      TempColor[j] = m_Color[i];
      j++;
    }

  m_NumColors--;
  delete[] m_Color;
  m_Color = TempColor;

  UpdateScrollBar();
  Invalidate();
}