#include "SpritePreviewPalette.hpp"
#include "StateServer.hpp"

#define ID_SPRITE_PLAY  100012
#define ID_SPRITE_STOP  100013
#define ID_SPRITE_TIMER 100014


BEGIN_MESSAGE_MAP(CSpritePreviewPalette, CMiniFrameWnd)

  ON_WM_CLOSE()
  ON_WM_SIZE()
  ON_WM_PAINT()
  ON_WM_TIMER()

  ON_COMMAND(ID_SPRITE_PLAY, OnPlay)
  ON_COMMAND(ID_SPRITE_STOP, OnStop)
END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CSpritePreviewPalette::CSpritePreviewPalette(CWnd* parent, sSpriteset* spriteset)
: m_Created(false)
, m_Direction(0)
, m_FrameNum(0)
, m_FrameDelay(0)
, m_DelayCounter(0)
{
  m_Spriteset = spriteset;
  m_DrawBitmap = new CDIBSection(spriteset->GetFrameWidth(), spriteset->GetFrameHeight(), 32);
  
  int left   = GetStateServer()->GetInt("SpritePreviewPalette:left",   -1);
  int top    = GetStateServer()->GetInt("SpritePreviewPalette:top",    -1);
  int right  = GetStateServer()->GetInt("SpritePreviewPalette:right",  -1);
  int bottom = GetStateServer()->GetInt("SpritePreviewPalette:bottom", -1);

  CRect start_rect(left, top, right, bottom);

  if (left == -1 && top == -1 && right == -1 && bottom == -1)
    start_rect = CRect(0, 0, 80, 80);

  Create(
    AfxRegisterWndClass(0, LoadCursor(NULL, MAKEINTRESOURCE(IDC_ARROW)), NULL, NULL),
    "preview",
    WS_POPUP | WS_CAPTION | WS_VISIBLE | WS_SIZEBOX | MFS_SYNCACTIVE | MFS_THICKFRAME | MFS_BLOCKSYSMENU,
    start_rect,
    parent);

  CFont* pFont = CFont::FromHandle((HFONT)GetStockObject(DEFAULT_GUI_FONT));
  m_PlayButton.Create("Play", WS_CHILD | WS_VISIBLE, CRect(0,0,10,10), this, ID_SPRITE_PLAY);
  m_PlayButton.SetFont(pFont);
  m_StopButton.Create("Stop", WS_CHILD | WS_VISIBLE, CRect(10,10,10,10), this, ID_SPRITE_STOP);
  m_StopButton.SetFont(pFont);
  m_StopButton.EnableWindow(false);
  m_Created = true;

  RECT rect;
  GetClientRect(&rect);
  OnSize(0, rect.right, rect.bottom);
}

////////////////////////////////////////////////////////////////////////////////

CSpritePreviewPalette::~CSpritePreviewPalette()
{
  delete m_DrawBitmap;
}

////////////////////////////////////////////////////////////////////////////////

void
CSpritePreviewPalette::SetDirection(int direction)
{
  if (m_Direction != direction)
  {
    m_Direction = direction;
    m_FrameNum = 0;
    m_FrameDelay = m_Spriteset->GetFrame(m_Direction, 0).GetDelay();
    m_DelayCounter = 0;

    Invalidate();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSpritePreviewPalette::OnPlay()
{
  SetTimer(ID_SPRITE_TIMER, 17, NULL);
  
  m_PlayButton.EnableWindow(false);
  m_StopButton.EnableWindow(true);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSpritePreviewPalette::OnStop()
{
  KillTimer(ID_SPRITE_TIMER);

  m_PlayButton.EnableWindow(true);
  m_StopButton.EnableWindow(false);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSpritePreviewPalette::OnClose()
{
  RECT rect;
  GetWindowRect(&rect);
  GetStateServer()->SetInt("SpritePreviewPalette:left",   rect.left);
  GetStateServer()->SetInt("SpritePreviewPalette:top",    rect.top);
  GetStateServer()->SetInt("SpritePreviewPalette:right",  rect.right);
  GetStateServer()->SetInt("SpritePreviewPalette:bottom", rect.bottom);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSpritePreviewPalette::OnSize(UINT type, int cx, int cy)
{
  if (m_PlayButton.m_hWnd)
  {
    m_PlayButton.MoveWindow(CRect(0, cy - 30, cx / 2, cy));
    m_StopButton.MoveWindow(CRect(cx / 2, cy - 30, cx, cy));
  }

  int AR_x, AR_y;
  int PrevAR = m_AspectRatio;
  AR_x = cx / m_Spriteset->GetFrameWidth();
  AR_y = (cy - 30) / m_Spriteset->GetFrameHeight();
  m_AspectRatio = (AR_x < AR_y ? AR_x : AR_y);
  if (!m_AspectRatio) m_AspectRatio = 1;

  if (PrevAR != m_AspectRatio)
    Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSpritePreviewPalette::OnPaint()
{
  CPaintDC dc(this);

  RECT rect;
  GetClientRect(&rect);

  CBrush backbrush;
  backbrush.CreateSysColorBrush(NULL);
  //dc.FillRect(&rect, &backbrush);
  if ((rect.right - rect.left) > (m_Spriteset->GetFrameWidth() * m_AspectRatio))
  {
    RECT window = rect;
    window.bottom -= 30;

    // left
    window.left = 0;
    window.right = ((rect.right - rect.left) - (m_Spriteset->GetFrameWidth() * m_AspectRatio)) / 2;
    dc.FillRect(&window, &backbrush);

    // right
    window.left = ((rect.right - rect.left) + (m_Spriteset->GetFrameWidth() * m_AspectRatio)) / 2;
    window.right = rect.right - rect.left;
    dc.FillRect(&window, &backbrush);
  }
  if ((rect.bottom - rect.top - 30) > (m_Spriteset->GetFrameHeight() * m_AspectRatio))
  {
    RECT window = rect;
    window.left = ((rect.right - rect.left) - (m_Spriteset->GetFrameWidth() * m_AspectRatio)) / 2;
    window.right = ((rect.right - rect.left) + (m_Spriteset->GetFrameWidth() * m_AspectRatio)) / 2;

    // top
    window.top = 0;
    window.bottom = ((rect.bottom - rect.top - 30) - (m_Spriteset->GetFrameHeight() * m_AspectRatio)) / 2;
    dc.FillRect(&window, &backbrush);

    // bottom
    window.top = ((rect.bottom - rect.top - 30) + (m_Spriteset->GetFrameHeight() * m_AspectRatio)) / 2;
    window.bottom = rect.bottom - rect.top - 30;
    dc.FillRect(&window, &backbrush);
  }
  backbrush.DeleteObject();

  int bmp_width = m_Spriteset->GetFrameWidth();
  for (int j=0; j<m_Spriteset->GetFrameHeight(); j++)
    for (int i=0; i<m_Spriteset->GetFrameWidth(); i++)
    {
      BGRA* dest = (BGRA*)m_DrawBitmap->GetPixels();
      RGBA* src = m_Spriteset->GetFrame(m_Direction,m_FrameNum).GetPixels();
      int alpha = src[j * bmp_width + i].alpha;

      dest[j * bmp_width + i].red = (src[j * bmp_width + i].red * alpha + src[j * bmp_width + i].blue * (255 - alpha)) / 256;
      dest[j * bmp_width + i].green = (src[j * bmp_width + i].green * alpha + src[j * bmp_width + i].green * (255 - alpha)) / 256;
      dest[j * bmp_width + i].blue = (src[j * bmp_width + i].blue * alpha + src[j * bmp_width + i].blue * (255 - alpha)) / 256;
    }

  //dc.BitBlt(0,0, rect.right - rect.left, rect.bottom - rect.top - 30, CDC::FromHandle(m_DrawBitmap->GetDC()), 0, 0, SRCCOPY);
  //dc.StretchBlt(0, 0, rect.right - rect.left, rect.bottom - rect.top - 30, CDC::FromHandle(m_DrawBitmap->GetDC()), 0, 0, m_DrawBitmap->GetWidth(), m_DrawBitmap->GetHeight(), SRCCOPY);
  dc.StretchBlt(
    ((rect.right - rect.left) -  (m_Spriteset->GetFrameWidth() * m_AspectRatio)) / 2, 
    ((rect.bottom - rect.top) - (m_Spriteset->GetFrameHeight() * m_AspectRatio) - 30) / 2, 
    m_Spriteset->GetFrameWidth() * m_AspectRatio, 
    m_Spriteset->GetFrameHeight() * m_AspectRatio, 
    CDC::FromHandle(m_DrawBitmap->GetDC()), 
    0, 0, m_DrawBitmap->GetWidth(), m_DrawBitmap->GetHeight(), SRCCOPY);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CSpritePreviewPalette::OnTimer(UINT nIDEvent)
{
  m_DelayCounter++;

  if (m_DelayCounter >= m_FrameDelay)
  {
    m_FrameNum++;
    if (m_FrameNum >= m_Spriteset->GetNumFrames(m_Direction))
      m_FrameNum = 0;

    m_FrameDelay = m_Spriteset->GetFrame(m_Direction, m_FrameNum).GetDelay();
    if (m_FrameDelay <= 0) m_FrameDelay = 1;
    m_DelayCounter = 0;

    Invalidate();
  }
}