#include "ScriptWindow.hpp"
#include "FileDialogs.hpp"
#include "Project.hpp"
#include "Scripting.hpp"
#include "MemoryBuffer.hpp"
#include "FileBuffer.hpp"
#include "FileSystem.hpp"
#include "x++.hpp"
#include "resource.h"


const int ID_EDIT = 900;


BEGIN_MESSAGE_MAP(CScriptWindow, CSaveableDocumentWindow)

  ON_WM_SIZE()
  ON_WM_SETFOCUS()

  ON_COMMAND(ID_SCRIPT_CHECKSYNTAX, OnScriptCheckSyntax)
  ON_COMMAND(ID_SCRIPT_FIND,        OnScriptFind)
  ON_COMMAND(ID_SCRIPT_REPLACE,     OnScriptReplace)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CScriptWindow::CScriptWindow(CStatusBar* status_bar)
: CSaveableDocumentWindow("", IDR_SCRIPT)
, m_Created(false)
, m_StatusBar(status_bar)
{
  SetSaved(false);
  SetModified(false);
  Create();
}

////////////////////////////////////////////////////////////////////////////////

CScriptWindow::CScriptWindow(CStatusBar* status_bar, const char* filename)
: CSaveableDocumentWindow(filename, IDR_SCRIPT)
, m_Created(false)
, m_StatusBar(status_bar)
{
  SetSaved(true);
  SetModified(false);
  Create();
  LoadScript(filename);
}

////////////////////////////////////////////////////////////////////////////////

CScriptWindow::~CScriptWindow()
{
}

////////////////////////////////////////////////////////////////////////////////

void
CScriptWindow::Create()
{
  // create the child window
  CSaveableDocumentWindow::Create(
    AfxRegisterWndClass(0, LoadCursor(NULL, IDC_ARROW), NULL, AfxGetApp()->LoadIcon(IDI_SCRIPT)));

  // creates the script view
  m_Edit.Create(m_StatusBar, this, this);

  m_Created = true;

  // update the size of the view
  RECT Rect;
  GetClientRect(&Rect);
  OnSize(0, Rect.right - Rect.left, Rect.bottom - Rect.top);

  // give the view focus
  m_Edit.SetFocus();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg bool
CScriptWindow::LoadScript(const char* filename)
{
  // get the file size
  int file_size = FileSize(GetDocumentPath());
  if (file_size == -1)
    return false;

  // open the file
  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  // allocate a temporary storage buffer and read it
  char* buffer = new char[file_size + 1];
  fread(buffer, sizeof(char), file_size, file);
  
  // null-terminate the string
  buffer[file_size] = 0;

  // put the buffer into the edit control
  m_Edit.SetWindowText(buffer);
  m_Edit.FormatAll();
  m_Edit.SetSel(0, 0);

  // delete the buffer and close the file
  delete[] buffer;
  fclose(file);

  SetModified(false);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CScriptWindow::OnSize(UINT type, int cx, int cy)
{
  if (m_Created)
    m_Edit.MoveWindow(0, 0, cx, cy);

  CSaveableDocumentWindow::OnSize(type, cx, cy);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CScriptWindow::OnSetFocus(CWnd* old)
{
  if (m_Created)
    m_Edit.SetFocus();
}

////////////////////////////////////////////////////////////////////////////////

void
CScriptWindow::SV_ScriptChanged()
{
  if (!IsModified())
    SetModified(true);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CScriptWindow::OnScriptCheckSyntax()
{
  OnFileSave();
  if (IsModified())  // if the user cancelled, don't compile
    return;

  char old_directory[MAX_PATH];
  GetCurrentDirectory(MAX_PATH, old_directory);

  // set the directory to where the script is
  char script_path[MAX_PATH];
  strcpy(script_path, GetDocumentPath());
  if (strrchr(script_path, '\\'))
    *strrchr(script_path, '\\') = 0;
  SetCurrentDirectory(script_path);

  // verify the script
  sCompileError error;
  if (!VerifyScript(GetDocumentPath(), error))
  {
    // show the error message
    MessageBox(error.m_Message.c_str(), "Check syntax", MB_OK);
    if (error.m_Token.length() > 0)
      m_Edit.SetSel(error.m_TokenStart, error.m_TokenStart + error.m_Token.length());
  }
  else
    MessageBox("Script is valid");
  
  SetCurrentDirectory(old_directory);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CScriptWindow::OnScriptFind()
{
  m_Edit.ScriptFindWord();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CScriptWindow::OnScriptReplace()
{
  m_Edit.ScriptFindReplaceWord();
}

////////////////////////////////////////////////////////////////////////////////

bool
CScriptWindow::GetSavePath(char* path)
{
  CScriptFileDialog Dialog(FDM_SAVE);
  if (Dialog.DoModal() != IDOK)
    return false;

  strcpy(path, Dialog.GetPathName());
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CScriptWindow::SaveDocument(const char* path)
{
  FILE* file = fopen(path, "wb");
  if (file == NULL)
    return false;

  CString text;
  m_Edit.GetWindowText(text);

  fwrite((const char*)text, 1, text.GetLength(), file);

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////
