#include "ProjectWindow.hpp"
#include "MainWindow.hpp"
#include "Project.hpp"
#include "FileDialogs.hpp"
#include "Scripting.hpp"
#include "x++.hpp"
#include "types.h"
#include "resource.h"


const int TreeID = 9865;

const int tiGameSettings = 100;
const int tiMaps         = 101;
const int tiSpritesets   = 102;
const int tiMusic        = 103;
const int tiSounds       = 104;
const int tiScripts      = 105;
const int tiImages       = 106;
const int tiAnimations   = 107;
const int tiWindowStyles = 108;
const int tiFonts        = 109;


BEGIN_MESSAGE_MAP(CProjectWindow, CMDIChildWnd)

  ON_WM_SYSCOMMAND()
  ON_WM_SIZE()
  ON_WM_SETFOCUS()

  ON_COMMAND(ID_PROJECTGROUP_INSERT, OnProjectGroupInsert)

  ON_COMMAND(ID_PROJECTITEM_OPEN,   OnProjectItemOpen)
  ON_COMMAND(ID_PROJECTITEM_DELETE, OnProjectItemDelete)

  ON_NOTIFY(TVN_KEYDOWN, TreeID, OnKeyDown)
  ON_NOTIFY(NM_DBLCLK,   TreeID, OnDoubleClick)
  ON_NOTIFY(NM_RCLICK,   TreeID, OnRightClick)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CProjectWindow::CProjectWindow(CMainWindow* main_window, CProject* project)
: m_MainWindow(main_window)
, m_Project(project)
{
}

////////////////////////////////////////////////////////////////////////////////

BOOL
CProjectWindow::Create()
{
  char szProjectName[512];
  sprintf(szProjectName, "Project [%s]", m_Project->GetGameSubDirectory());

  // create the window
  CMDIChildWnd::Create(
    AfxRegisterWndClass(CS_NOCLOSE, NULL, NULL, AfxGetApp()->LoadIcon(IDI_PROJECT)),
    szProjectName,
    WS_CHILD | WS_VISIBLE | WS_OVERLAPPEDWINDOW);

  // create widgets
  m_TreeControl.Create(
    WS_VISIBLE | WS_CHILD | TVS_SHOWSELALWAYS | TVS_HASLINES | TVS_LINESATROOT | TVS_HASBUTTONS,
    CRect(0, 0, 0, 0),
    this,
    TreeID);
  m_TreeControl.SetFocus();

  // make sure they are in the right place
  RECT rect;
  GetClientRect(&rect);
  OnSize(0, rect.right, rect.bottom);

  Update();

  return TRUE;
}

////////////////////////////////////////////////////////////////////////////////

BOOL
CProjectWindow::DestroyWindow()
{
  // remove the old image list
  delete m_TreeControl.SetImageList(NULL, 0);

  m_TreeControl.DestroyWindow();
  return CMDIChildWnd::DestroyWindow();
}

////////////////////////////////////////////////////////////////////////////////

void
CProjectWindow::Update()
{
  m_Project->RefreshItems();

  m_TreeControl.DeleteAllItems();

#define INSERT_ROOT(text, item) \
  m_TreeControl.InsertItem(TVIF_PARAM | TVIF_TEXT, text, 0, 0, 0, 0, item, TVI_ROOT, TVI_LAST);

                              INSERT_ROOT("Game Settings", tiGameSettings);
  HTREEITEM htiMaps         = INSERT_ROOT("Maps",          tiMaps);
  HTREEITEM htiSpritesets   = INSERT_ROOT("Spritesets",    tiSpritesets);
  HTREEITEM htiScripts      = INSERT_ROOT("Scripts",       tiScripts);
  HTREEITEM htiSounds       = INSERT_ROOT("Sounds",        tiSounds);
  HTREEITEM htiMusic        = INSERT_ROOT("Music",         tiMusic);
  HTREEITEM htiFonts        = INSERT_ROOT("Fonts",         tiFonts);
  HTREEITEM htiWindowStyles = INSERT_ROOT("Window Styles", tiWindowStyles);
  HTREEITEM htiImages       = INSERT_ROOT("Images",        tiImages);
  HTREEITEM htiAnimations   = INSERT_ROOT("Animations",    tiAnimations);

  #define INSERT_ITEMS(hti, grouptype)                          \
    for (int i = 0; i < m_Project->GetItemCount(grouptype); i++)   \
      m_TreeControl.InsertItem(TVIF_TEXT, m_Project->GetItem(grouptype, i), 0, 0, 0, 0, 0, hti, TVI_SORT);

  INSERT_ITEMS(htiMaps,         GT_MAPS);
  INSERT_ITEMS(htiSpritesets,   GT_SPRITESETS);
  INSERT_ITEMS(htiScripts,      GT_SCRIPTS);
  INSERT_ITEMS(htiSounds,       GT_SOUNDS);
  INSERT_ITEMS(htiMusic,        GT_MUSIC);
  INSERT_ITEMS(htiFonts,        GT_FONTS);
  INSERT_ITEMS(htiWindowStyles, GT_WINDOWSTYLES);
  INSERT_ITEMS(htiImages,       GT_IMAGES);
  INSERT_ITEMS(htiAnimations,   GT_ANIMATIONS);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CProjectWindow::OnSysCommand(UINT id, LPARAM param)
{
  // do not close it damnit!
  // and oh, Ctrl-F4 calls close MDI window, if ya don't know.

  if (id != SC_CLOSE)
    CWnd::OnSysCommand(id, param);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CProjectWindow::OnSize(UINT type, int cx, int cy)
{
  if (m_TreeControl.m_hWnd)
    m_TreeControl.MoveWindow(CRect(0, 0, cx, cy), TRUE);

  CMDIChildWnd::OnSize(type, cx, cy);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CProjectWindow::OnSetFocus(CWnd* old_window)
{
  if (m_TreeControl.m_hWnd)
    m_TreeControl.SetFocus();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CProjectWindow::OnProjectGroupInsert()
{
  HTREEITEM item = m_TreeControl.GetSelectedItem();
  switch (m_TreeControl.GetItemData(item))
  {
    case tiMaps:         ::PostMessage(m_MainWindow->m_hWnd, WM_COMMAND, MAKEWPARAM(ID_PROJECT_INSERT_MAP,         0), 0);  break;
    case tiSpritesets:   ::PostMessage(m_MainWindow->m_hWnd, WM_COMMAND, MAKEWPARAM(ID_PROJECT_INSERT_SPRITESET,   0), 0);  break;
    case tiScripts:      ::PostMessage(m_MainWindow->m_hWnd, WM_COMMAND, MAKEWPARAM(ID_PROJECT_INSERT_SCRIPT,      0), 0);  break;
    case tiSounds:       ::PostMessage(m_MainWindow->m_hWnd, WM_COMMAND, MAKEWPARAM(ID_PROJECT_INSERT_SOUND,       0), 0);  break;
    case tiMusic:        ::PostMessage(m_MainWindow->m_hWnd, WM_COMMAND, MAKEWPARAM(ID_PROJECT_INSERT_MUSIC,       0), 0);  break;
    case tiFonts:        ::PostMessage(m_MainWindow->m_hWnd, WM_COMMAND, MAKEWPARAM(ID_PROJECT_INSERT_FONT,        0), 0);  break;
    case tiWindowStyles: ::PostMessage(m_MainWindow->m_hWnd, WM_COMMAND, MAKEWPARAM(ID_PROJECT_INSERT_WINDOWSTYLE, 0), 0);  break;
    case tiImages:       ::PostMessage(m_MainWindow->m_hWnd, WM_COMMAND, MAKEWPARAM(ID_PROJECT_INSERT_IMAGE,       0), 0);  break;
    case tiAnimations:   ::PostMessage(m_MainWindow->m_hWnd, WM_COMMAND, MAKEWPARAM(ID_PROJECT_INSERT_ANIMATION,   0), 0);  break;
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CProjectWindow::OnProjectItemOpen()
{
  OnDoubleClick(NULL, NULL);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CProjectWindow::OnProjectItemDelete()
{
  if (MessageBox("This will permanently delete the file.\nAre you sure you want to continue?", "Delete Project Item", MB_YESNO) == IDNO)
    return;

  HTREEITEM item = m_TreeControl.GetSelectedItem();
  if (item == NULL)
    return;

  // check to see if user clicked on an item in a folder
  HTREEITEM parent = m_TreeControl.GetParentItem(item);
  if (parent == NULL)
    return;

  int GroupType;
  switch (m_TreeControl.GetItemData(parent))
  {
    case tiMaps:         GroupType = GT_MAPS;         break;
    case tiSpritesets:   GroupType = GT_SPRITESETS;   break;
    case tiScripts:      GroupType = GT_SCRIPTS;      break;
    case tiSounds:       GroupType = GT_SOUNDS;       break;
    case tiMusic:        GroupType = GT_MUSIC;        break;
    case tiFonts:        GroupType = GT_FONTS;        break;
    case tiWindowStyles: GroupType = GT_WINDOWSTYLES; break;
    case tiImages:       GroupType = GT_IMAGES;       break;
    case tiAnimations:   GroupType = GT_ANIMATIONS;   break;
  }

  // build file path
  char filename[MAX_PATH];
  strcpy(filename, m_Project->GetDirectory());
  strcat(filename, "\\");
  strcat(filename, m_Project->GetGroupDirectory(GroupType));
  strcat(filename, "\\");
  strcat(filename, m_TreeControl.GetItemText(item));

  if (!DeleteFile(filename))
    MessageBox("Error: Could not delete file");

  m_Project->RefreshItems();
  Update();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CProjectWindow::OnKeyDown(NMHDR* notify, LRESULT* result)
{
  NMTVKEYDOWN* key = (NMTVKEYDOWN*)notify;

  switch (key->wVKey)
  {
    case VK_RETURN: 
      OnDoubleClick(NULL, NULL);
      break;

    case VK_INSERT:
      OnProjectGroupInsert();
      break;
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CProjectWindow::OnDoubleClick(NMHDR* /*notify*/, LRESULT* /*result*/)
{
  HTREEITEM item = m_TreeControl.GetSelectedItem();
  if (item == NULL)
    return;

  switch (m_TreeControl.GetItemData(item))
  {
    case tiGameSettings:
      // open game window
      m_MainWindow->OpenGameSettings();
      break;

    default:
    {
      // check to see if user clicked on an item in a folder
      HTREEITEM parent = m_TreeControl.GetParentItem(item);
      if (parent == NULL)
        return;

      int GroupType;
      switch (m_TreeControl.GetItemData(parent))
      {
        case tiMaps:         GroupType = GT_MAPS;         break;
        case tiSpritesets:   GroupType = GT_SPRITESETS;   break;
        case tiScripts:      GroupType = GT_SCRIPTS;      break;
        case tiSounds:       GroupType = GT_SOUNDS;       break;
        case tiMusic:        GroupType = GT_MUSIC;        break;
        case tiFonts:        GroupType = GT_FONTS;        break;
        case tiWindowStyles: GroupType = GT_WINDOWSTYLES; break;
        case tiImages:       GroupType = GT_IMAGES;       break;
        case tiAnimations:   GroupType = GT_ANIMATIONS;   break;
      }

      // build file path
      char szFilename[MAX_PATH];
      strcpy(szFilename, m_Project->GetDirectory());
      strcat(szFilename, "\\");
      strcat(szFilename, m_Project->GetGroupDirectory(GroupType));
      strcat(szFilename, "\\");
      strcat(szFilename, m_TreeControl.GetItemText(item));

      m_MainWindow->OpenDocumentWindow(GroupType, szFilename);
      break;
    }
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CProjectWindow::OnRightClick(NMHDR* notify, LRESULT* result)
{
  SelectItemAtCursor();
  HTREEITEM item = m_TreeControl.GetSelectedItem();
  if (item == NULL)
    return;

  // if the user right-clicked on a group, open the "group" popup menu
  switch (m_TreeControl.GetItemData(item))
  {
    case tiMaps:
    case tiSpritesets:
    case tiScripts:
    case tiSounds:
    case tiMusic:
    case tiFonts:
    case tiWindowStyles:
    case tiImages:
    case tiAnimations:
    {
      CMenu Menu;
      Menu.LoadMenu(IDR_PROJECTGROUP);

      POINT Point;
      GetCursorPos(&Point);
      Menu.GetSubMenu(0)->TrackPopupMenu(TPM_LEFTALIGN | TPM_TOPALIGN | TPM_RIGHTBUTTON, Point.x, Point.y, this);
      
      Menu.DestroyMenu();
      return;
    }
  }

  HTREEITEM parent = m_TreeControl.GetParentItem(item);
  if (parent == NULL)
    return;

  CMenu menu;
  menu.LoadMenu(IDR_PROJECTITEM);

  // get the mouse coordinates
  POINT Point;
  GetCursorPos(&Point);

  menu.GetSubMenu(0)->TrackPopupMenu(TPM_LEFTALIGN | TPM_TOPALIGN | TPM_RIGHTBUTTON, Point.x, Point.y, this);
  
  menu.DestroyMenu();
}

////////////////////////////////////////////////////////////////////////////////

void
CProjectWindow::SelectItemAtCursor()
{
  // select the item the user right-clicked on
  POINT Point;
  GetCursorPos(&Point);
  ::ScreenToClient(m_TreeControl.m_hWnd, &Point);

  UINT uFlags;
  HTREEITEM item = m_TreeControl.HitTest(Point, &uFlags);

  // if we're on the item label, select it
  if (uFlags & TVHT_ONITEM)
    m_TreeControl.SelectItem(item);
  else
    m_TreeControl.SelectItem(NULL);
}


////////////////////////////////////////////////////////////////////////////////
