#include <windows.h>
#include <stdio.h>
#include <string.h>
#include "Project.hpp"
#include "configfile.h"
#include "types.h"
#include "x++.hpp"


////////////////////////////////////////////////////////////////////////////////

CProject::CProject()
: m_Directory(newstr(""))
, m_Filename(newstr(""))

, m_GameTitle(newstr(""))
, m_GameScript(newstr(""))

, m_ScreenWidth(320)
, m_ScreenHeight(240)
{
  for (int i = 0; i < 10; i++)
  {
    m_Groups[i].num_entries = 0;
    m_Groups[i].entries = NULL;
  }
}

////////////////////////////////////////////////////////////////////////////////

CProject::~CProject()
{
  Destroy();
}

////////////////////////////////////////////////////////////////////////////////

bool
CProject::Create(const char* games_directory, const char* project_name)
{
  Destroy();

  // create the project
  if (SetCurrentDirectory(games_directory) == FALSE)
    return false;
  
  // if creating the directory failed, it may already exist
  CreateDirectory(project_name, NULL);
  
  // wait to see if SetCurrentDirectory() fails
  if (SetCurrentDirectory(project_name) == FALSE)
    return false;

  // set the project directory
  char path[MAX_PATH];
  if (GetCurrentDirectory(MAX_PATH, path) == FALSE)
    return false;
  m_Directory = newstr(path);
    
  // set the project filename
  m_Filename = new char[strlen(path) + strlen(project_name) + 20];
  sprintf(m_Filename, "%s\\game.inf", path);

  // set default values in project
  m_GameTitle = newstr("");
  m_GameScript = newstr("");

  m_ScreenWidth = 320;
  m_ScreenHeight = 240;

  RefreshItems();
  return Save();
}

////////////////////////////////////////////////////////////////////////////////

bool
CProject::Open(const char* filename)
{
  Destroy();

  // set the game directory
  m_Directory = newstr(filename);
  *strrchr(m_Directory, '\\') = 0;

  // set the game filename
  m_Filename = newstr(m_Directory + strlen(m_Directory) + 1);

  // load the game.inf
  CONFIG config;
  LoadConfig(&config, m_Filename);

  // game title
  char title[520];
  ReadConfigString(&config, "", "name", title, 512, "");
  m_GameTitle = newstr(title);

  // script
  char script[520];
  ReadConfigString(&config, "", "script", script, 512, "");
  m_GameScript = newstr(script);

  // screen dimensions
  ReadConfigInt(&config, "", "screen_width",  &m_ScreenWidth, 320);
  ReadConfigInt(&config, "", "screen_height", &m_ScreenHeight, 240);

  DestroyConfig(&config);

  RefreshItems();

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CProject::Save() const
{
  SetCurrentDirectory(m_Directory);
  
  CONFIG config;
  LoadConfig(&config, m_Filename);

  // title
  WriteConfigString(&config, "", "name", m_GameTitle);

  // script
  WriteConfigString(&config, "", "script", m_GameScript);

  // screen dimensions
  WriteConfigInt(&config, "", "screen_width",  m_ScreenWidth);
  WriteConfigInt(&config, "", "screen_height", m_ScreenHeight);

  SaveConfig(&config, m_Filename);
  DestroyConfig(&config);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetDirectory() const
{
  return m_Directory;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetGameSubDirectory() const
{
  if (strrchr(m_Directory, '\\'))
    return strrchr(m_Directory, '\\') + 1;
  else
    return "";
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetGameTitle() const
{
  return m_GameTitle;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetGameScript() const
{
  return m_GameScript;
}

////////////////////////////////////////////////////////////////////////////////

int
CProject::GetScreenWidth() const
{
  return m_ScreenWidth;
}

////////////////////////////////////////////////////////////////////////////////

int
CProject::GetScreenHeight() const
{
  return m_ScreenHeight;
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::SetGameTitle(const char* game_title)
{
  delete[] m_GameTitle;
  m_GameTitle = newstr(game_title);
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::SetGameScript(const char* game_script)
{
  delete[] m_GameScript;
  m_GameScript = newstr(game_script);
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::SetScreenWidth(int width)
{
  m_ScreenWidth = width;
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::SetScreenHeight(int height)
{
  m_ScreenHeight = height;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetGroupDirectory(int grouptype)
{
  switch (grouptype)
  {
    case GT_MAPS:         return "maps";
    case GT_SPRITESETS:   return "spritesets";
    case GT_SCRIPTS:      return "scripts";
    case GT_SOUNDS:       return "sounds";
    case GT_MUSIC:        return "music";
    case GT_FONTS:        return "fonts";
    case GT_WINDOWSTYLES: return "windowstyles";
    case GT_IMAGES:       return "images";
    case GT_ANIMATIONS:   return "anim";
    default:              return NULL;
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::RefreshItems()
{
  // empty the old lists
  for (int i = 0; i < NUM_GROUP_TYPES; i++)
  {
    for (int j = 0; j < m_Groups[i].num_entries; j++)
      delete[] m_Groups[i].entries[j];

    m_Groups[i].num_entries = 0;
    m_Groups[i].entries = (char**)realloc(m_Groups[i].entries, 0);
  }

  // store the old directory
  char old_directory[MAX_PATH];
  GetCurrentDirectory(MAX_PATH, old_directory);

  for (int i = 0; i < NUM_GROUP_TYPES; i++)
  {
    SetCurrentDirectory(m_Directory);
    
    if (!SetCurrentDirectory(GetGroupDirectory(i)))
      continue;

    // go through each filter and add it to the project
    for (int j = 0; j < GetNumExtensions(i); j++)
    {
      const char* extension = GetExtension(i, j);
      char* filter = new char[strlen(extension) + 2];
      filter[0] = '*';
      strcpy(filter + 1, extension);

      WIN32_FIND_DATA ffd;
      HANDLE h = FindFirstFile(filter, &ffd);
      delete[] filter;
      if (h == INVALID_HANDLE_VALUE)
        continue;

      do {

        if (!(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
          AddItem(i, ffd.cFileName);

      } while (FindNextFile(h, &ffd));

      FindClose(h);
    }

  }

  // restore the old directory
  SetCurrentDirectory(old_directory);
}

////////////////////////////////////////////////////////////////////////////////

int
CProject::GetItemCount(int group_type) const
{
  return m_Groups[group_type].num_entries;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetItem(int group_type, int i) const
{
  return m_Groups[group_type].entries[i];
}

////////////////////////////////////////////////////////////////////////////////

bool
CProject::HasItem(int group_type, const char* item) const
{
  for (int i = 0; i < GetItemCount(group_type); i++)
    if (strcmp(item, GetItem(group_type, i)) == 0)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::AddItem(int grouptype, const char* filename)
{
  Group* group = m_Groups + grouptype;

  // make sure it's not in the group already
  for (int i = 0; i < group->num_entries; i++)
    if (strcmp(filename, group->entries[i]) == 0)
      return;

  group->entries = (char**)realloc(group->entries, sizeof(char*) * (group->num_entries + 1));
  group->entries[group->num_entries] = newstr(filename);
  group->num_entries++;
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::Destroy()
{
  delete[] m_Directory;
  m_Directory = NULL;

  delete[] m_Filename;
  m_Filename = NULL;

  delete[] m_GameTitle;
  m_GameTitle = NULL;

  m_GameScript = 0;
  m_ScreenWidth = 0;
  m_ScreenHeight = 0;

  for (int i = 0; i < NUM_GROUP_TYPES; i++)
  {
    for (int j = 0; j < m_Groups[i].num_entries; j++)
      delete[] m_Groups[i].entries[j];

    m_Groups[i].num_entries = 0;
    m_Groups[i].entries = (char**)realloc(m_Groups[i].entries, 0);
  }
}

////////////////////////////////////////////////////////////////////////////////
