#include "MapView.hpp"
#include "EntityWarpDialog.hpp"
#include "EntityPersonDialog.hpp"
#include "EntityTriggerDialog.hpp"
#include "EntityDoodadDialog.hpp"
#include "menu.h"
#include "x++.hpp"
#include "resource.h"


static int s_MapViewID = 2000;


BEGIN_MESSAGE_MAP(CMapView, CWnd)

  ON_WM_PAINT()
  ON_WM_SIZE()
  ON_WM_HSCROLL()
  ON_WM_VSCROLL()
  ON_WM_LBUTTONDOWN()
  ON_WM_MOUSEMOVE()
  ON_WM_LBUTTONUP()
  ON_WM_RBUTTONUP()
  ON_WM_TIMER()

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CMapView::CMapView()
: m_Map(NULL)

, m_BlitTile(NULL)
, m_ZoomSize(1)

, m_CurrentX(0)
, m_CurrentY(0)
, m_SelectedTile(0)
, m_SelectedLayer(0)

, m_Clicked(false)
{
}

////////////////////////////////////////////////////////////////////////////////

CMapView::~CMapView()
{
  // destroy the blit DIB
  delete m_BlitTile;
}

////////////////////////////////////////////////////////////////////////////////

BOOL
CMapView::Create(CStatusBar* status_bar, CMapViewHandler* handler, CWnd* parent, sMap* map)
{
  m_Handler = handler;
  m_Map = map;
  m_StatusBar = status_bar;

  m_BlitTile = new CDIBSection(
    map->GetTileset().GetTileWidth(),
    map->GetTileset().GetTileHeight(),
    32);

  // create the window object
  BOOL retval = CWnd::Create(
    AfxRegisterWndClass(0, LoadCursor(NULL, IDC_ARROW), NULL, NULL),
    "",
    WS_CHILD | WS_VISIBLE | WS_HSCROLL | WS_VSCROLL,
    CRect(0, 0, 0, 0),
    parent,
    s_MapViewID++);

  UpdateScrollBars();

  return retval;
}

////////////////////////////////////////////////////////////////////////////////

int
CMapView::GetZoomSize()
{
  return m_ZoomSize;
}

////////////////////////////////////////////////////////////////////////////////

void
CMapView::SetZoomSize(int size)
{
  m_ZoomSize = size;

  delete m_BlitTile;
  m_BlitTile = new CDIBSection(
    m_Map->GetTileset().GetTileWidth() * size,
    m_Map->GetTileset().GetTileHeight() * size,
    32);
  Invalidate();
  UpdateScrollBars();
}

////////////////////////////////////////////////////////////////////////////////

void
CMapView::TilesetChanged()
{
  delete m_BlitTile;
  m_BlitTile = new CDIBSection(
    m_Map->GetTileset().GetTileWidth(),
    m_Map->GetTileset().GetTileHeight(),
    32);

  Invalidate();
  UpdateScrollBars();
}

////////////////////////////////////////////////////////////////////////////////

void
CMapView::SelectTile(int tile)
{
  m_SelectedTile = tile;
}

////////////////////////////////////////////////////////////////////////////////

void
CMapView::SelectLayer(int layer)
{
  m_SelectedLayer = layer;
}

////////////////////////////////////////////////////////////////////////////////

void
CMapView::UpdateScrollBars()
{
  SCROLLINFO si;
  si.cbSize = sizeof(si);
  si.fMask  = SIF_ALL;

  // update the horizontal scroll bar
  if (GetPageSizeX() <= GetTotalTilesX())
  {
    si.nMin  = 0;
    si.nMax  = GetTotalTilesX();
    si.nPage = GetPageSizeX();
    si.nPos  = m_CurrentX;
  }
  else // if the page size is bigger than the total size
  {
    si.nMin  = 0;
    si.nMax  = 0xFFFF;
    si.nPage = 0xFFFE;
    si.nPos  = 0;
  }
  SetScrollInfo(SB_HORZ, &si);

  // update the vertical scroll bar
  if (GetPageSizeY() <= GetTotalTilesY())
  {
    si.nMin  = 0;
    si.nMax  = GetTotalTilesY();
    si.nPage = GetPageSizeY();
    si.nPos  = m_CurrentY;
  }
  else
  {
    si.nMin  = 0;
    si.nMax  = 0xFFFF;
    si.nPage = 0xFFFE;
    si.nPos  = 0;
  }
  SetScrollInfo(SB_VERT, &si);
}

////////////////////////////////////////////////////////////////////////////////

int
CMapView::GetPageSizeX()
{
  RECT ClientRect;
  GetClientRect(&ClientRect);
  return ClientRect.right / m_Map->GetTileset().GetTileWidth() / m_ZoomSize;
}

////////////////////////////////////////////////////////////////////////////////

int
CMapView::GetPageSizeY()
{
  RECT ClientRect;
  GetClientRect(&ClientRect);
  return ClientRect.bottom / m_Map->GetTileset().GetTileHeight() / m_ZoomSize;
}

////////////////////////////////////////////////////////////////////////////////

int
CMapView::GetTotalTilesX()
{
  int max_x = 0;
  for (int i = 0; i < m_Map->GetNumLayers(); i++)
    if (m_Map->GetLayer(i).GetWidth() - 1 > max_x)
      max_x = m_Map->GetLayer(i).GetWidth() - 1;
  return max_x;
}

////////////////////////////////////////////////////////////////////////////////

int
CMapView::GetTotalTilesY()
{
  int max_y = 0;
  for (int i = 0; i < m_Map->GetNumLayers(); i++)
    if (m_Map->GetLayer(i).GetHeight() - 1 > max_y)
      max_y = m_Map->GetLayer(i).GetHeight() - 1;
  return max_y;
}

////////////////////////////////////////////////////////////////////////////////

void
CMapView::ValidateScrollValues()
{
  int iOldCurrentX = m_CurrentX;
  int iOldCurrentY = m_CurrentY;

  // horizontal
  if (m_CurrentX < 0)
    m_CurrentX = 0;
  if (m_CurrentX > GetTotalTilesX() - GetPageSizeX() + 1)
  {
    m_CurrentX = GetTotalTilesX() - GetPageSizeX() + 1;
    if (m_CurrentX < 0)
      m_CurrentX = 0;
  }

  // vertical
  if (m_CurrentY < 0)
    m_CurrentY = 0;
  if (m_CurrentY > GetTotalTilesY() - GetPageSizeY() + 1)
  {
    m_CurrentY = GetTotalTilesY() - GetPageSizeY() + 1;
    if (m_CurrentY < 0)
      m_CurrentY = 0;
  }

  // if the CurrentX or CurrentY have changed, update the window
  if (iOldCurrentX != m_CurrentX ||
      iOldCurrentY != m_CurrentY)
    Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

void
CMapView::Click(int x, int y)
{
  int tx = m_CurrentX + (x / m_Map->GetTileset().GetTileWidth() / m_ZoomSize);
  int ty = m_CurrentY + (y / m_Map->GetTileset().GetTileHeight() / m_ZoomSize);

  sLayer& Layer = m_Map->GetLayer(m_SelectedLayer);

  // change the tile
  if (tx >= 0 &&
      ty >= 0 &&
      tx < Layer.GetWidth() &&
      ty < Layer.GetHeight())
  {
    int oldtile = Layer.GetTile(tx, ty);
    Layer.SetTile(tx, ty, m_SelectedTile);

    // if the tile has changed, invalidate it
    if (oldtile != m_SelectedTile)
    {
      int tile_width  = m_Map->GetTileset().GetTileWidth();
      int tile_height = m_Map->GetTileset().GetTileHeight();
      
      RECT Rect =
      {
        (tx - m_CurrentX) * tile_width * m_ZoomSize,
        (ty - m_CurrentY) * tile_height * m_ZoomSize,
        (tx - m_CurrentX) * tile_width  * m_ZoomSize + tile_width * m_ZoomSize,
        (ty - m_CurrentY) * tile_height  * m_ZoomSize + tile_height * m_ZoomSize,
      };
      InvalidateRect(&Rect);

      // tell the parent window that the map changed
      m_Handler->MV_MapChanged();
    }
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CMapView::SelectTileUnderPoint(CPoint Point)
{
  int tx = m_CurrentX + (Point.x / m_Map->GetTileset().GetTileWidth() / m_ZoomSize);
  int ty = m_CurrentY + (Point.y / m_Map->GetTileset().GetTileHeight() / m_ZoomSize);

  // change the tile
  if (tx >= 0 &&
      ty >= 0 &&
      tx < m_Map->GetLayer(m_SelectedLayer).GetWidth() &&
      ty < m_Map->GetLayer(m_SelectedLayer).GetHeight())
  {
    int tile = m_Map->GetLayer(m_SelectedLayer).GetTile(tx, ty);
    SelectTile(tile);

    // change the current tile
    m_Handler->MV_SelectedTileChanged(tile);
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CMapView::DrawTile(CDC& dc, const RECT& rect, int tx, int ty)
{
  int tile_width  = m_Map->GetTileset().GetTileWidth();
  int tile_height = m_Map->GetTileset().GetTileHeight();

  // clear the DIB
  memset(m_BlitTile->GetPixels(), 0,  m_ZoomSize * m_ZoomSize * tile_width * tile_height * 4);

  for (int i = 0; i < m_Map->GetNumLayers(); i++)
  {
    const sLayer& layer = m_Map->GetLayer(i);   
    if (tx >= 0 &&
        ty >= 0 &&
        tx < layer.GetWidth() &&
        ty < layer.GetHeight())
    {
      int tile = layer.GetTile(tx, ty);
      const RGBA* src = m_Map->GetTileset().GetTile(tile).GetPixels();      
      BGRA* dest = (BGRA*)m_BlitTile->GetPixels();

      int counter = 0;
      for (int j=0; j<tile_height; j++)
      {
        for (int k=0; k<tile_width; k++)
          for (int l=0; l<m_ZoomSize; l++)
          {
            int alpha = src[j * tile_width + k].alpha;
            dest[counter].red   = (alpha * src[j * tile_width + k].red   + (255 - alpha) * dest[counter].red)   / 256;
            dest[counter].green = (alpha * src[j * tile_width + k].green + (255 - alpha) * dest[counter].green) / 256;
            dest[counter].blue  = (alpha * src[j * tile_width + k].blue  + (255 - alpha) * dest[counter].blue)  / 256;
            counter++;
          }

        for (int k=1; k<m_ZoomSize; k++)
        {
          memcpy(dest + counter, dest + (counter - tile_width * m_ZoomSize), tile_width * m_ZoomSize * sizeof(RGBA));
          counter += tile_width * m_ZoomSize;
        }
      }
    }
  }

  dc.BitBlt(rect.left, rect.top, tile_width * m_ZoomSize, tile_height * m_ZoomSize,
            CDC::FromHandle(m_BlitTile->GetDC()), 0, 0, SRCCOPY);

  if (tx == m_Map->GetStartX() / tile_width &&
      ty == m_Map->GetStartY() / tile_height)
  {
    // save the DC's state so there aren't any ill effects
    dc.SaveDC();
    dc.SetBkMode(TRANSPARENT);

    RECT r = rect;
    OffsetRect(&r, 1, 1);

    // draw text backdrop
    dc.SetTextColor(0x000000);
    dc.DrawText("ST", &r, DT_CENTER | DT_VCENTER);

    OffsetRect(&r, -1, -1);

    // draw white text
    dc.SetTextColor(0xFFFFFF);
    dc.DrawText("ST", &r, DT_CENTER | DT_VCENTER);

    dc.RestoreDC(-1);
  }

  for (int i = 0; i < m_Map->GetNumEntities(); i++)
  {
    sEntity& entity = m_Map->GetEntity(i);
    if (tx == entity.m_X / tile_width && ty == entity.m_Y / tile_height)
    {
      HICON icon;
      switch (entity.GetEntityType())
      {
        case sEntity::WARP:    icon = AfxGetApp()->LoadIcon(IDI_ENTITY_WARP); break;
        case sEntity::PERSON:  icon = AfxGetApp()->LoadIcon(IDI_ENTITY_PERSON); break;
        case sEntity::TRIGGER: icon = AfxGetApp()->LoadIcon(IDI_ENTITY_TRIGGER); break;
        case sEntity::DOODAD:  icon = AfxGetApp()->LoadIcon(IDI_ENTITY_DOODAD); break;
      }

      int tw = m_Map->GetTileset().GetTileWidth()  * m_ZoomSize;
      int th = m_Map->GetTileset().GetTileHeight() * m_ZoomSize;
      DrawIconEx(dc.m_hDC, rect.left, rect.top, icon, tw, th, 0, NULL, DI_NORMAL);
    }
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMapView::OnPaint()
{
  CPaintDC dc(this);

  int NumTilesX = GetPageSizeX() + 1;
  int NumTilesY = GetPageSizeY() + 1;

  // draw all tiles visible in the client window
  for (int ix = 0; ix < NumTilesX; ix++)
    for (int iy = 0; iy < NumTilesY; iy++)
    {
      // visibility check
      int tile_width  = m_Map->GetTileset().GetTileWidth();
      int tile_height = m_Map->GetTileset().GetTileHeight();
      RECT Rect = {
        ix * tile_width * m_ZoomSize,
        iy * tile_height * m_ZoomSize,
        ix * tile_width * m_ZoomSize + tile_width * m_ZoomSize,
        iy * tile_height * m_ZoomSize + tile_height * m_ZoomSize,
      };
      if (dc.RectVisible(&Rect))
      {

        int tx = ix + m_CurrentX;
        int ty = iy + m_CurrentY;
        DrawTile(dc, Rect, tx, ty);

      }
    }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMapView::OnSize(UINT uType, int cx, int cy)
{
  ValidateScrollValues();
  UpdateScrollBars();

  CWnd::OnSize(uType, cx, cy);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMapView::OnHScroll(UINT uSBCode, UINT uPos, CScrollBar* pScrollBar)
{
  int old_x = m_CurrentX;

  switch (uSBCode)
  {
    case SB_LINEDOWN:   m_CurrentX++; break;
    case SB_LINEUP:     m_CurrentX--; break;
    case SB_PAGEDOWN:   m_CurrentX += GetPageSizeX(); break;
    case SB_PAGEUP:     m_CurrentX -= GetPageSizeX(); break;
    case SB_THUMBTRACK: m_CurrentX = (int)uPos; break;
  }

  ValidateScrollValues();
  UpdateScrollBars();

  // do the scrolling thing
  int new_x = m_CurrentX;
  CDC* dc_ = GetDC();
  HDC dc = dc_->m_hDC;

  HRGN region = CreateRectRgn(0, 0, 0, 0);
  int factor = m_ZoomSize * m_Map->GetTileset().GetTileWidth();
  ScrollDC(dc, (old_x - new_x) * factor, 0, NULL, NULL, region, NULL);
  ::InvalidateRgn(m_hWnd, region, FALSE);
  DeleteObject(region);

  ReleaseDC(dc_);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMapView::OnVScroll(UINT uSBCode, UINT uPos, CScrollBar* pScrollBar)
{
  int old_y = m_CurrentY;

  switch (uSBCode)
  {
    case SB_LINEDOWN:   m_CurrentY++; break;
    case SB_LINEUP:     m_CurrentY--; break;
    case SB_PAGEDOWN:   m_CurrentY += GetPageSizeY(); break;
    case SB_PAGEUP:     m_CurrentY -= GetPageSizeY(); break;
    case SB_THUMBTRACK: m_CurrentY = (int)uPos; break;
  }

  ValidateScrollValues();
  UpdateScrollBars();

  // do the scrolling thing
  int new_y = m_CurrentY;
  CDC* dc_ = GetDC();
  HDC dc = dc_->m_hDC;

  HRGN region = CreateRectRgn(0, 0, 0, 0);
  int factor = m_ZoomSize * m_Map->GetTileset().GetTileHeight();
  ScrollDC(dc, 0, (old_y - new_y) * factor, NULL, NULL, region, NULL);
  ::InvalidateRgn(m_hWnd, region, FALSE);
  DeleteObject(region);

  ReleaseDC(dc_);

}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMapView::OnLButtonDown(UINT uFlags, CPoint Point)
{
  // if the shift key is down, select the current tile
  if (uFlags & MK_SHIFT)
  {
    SelectTileUnderPoint(Point);
  }
  else
  {
    Click(Point.x, Point.y);

    // grab all mouse events until the user releases the button
    SetCapture();
    m_Clicked = true;
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMapView::OnMouseMove(UINT uFlags, CPoint Point)
{
  if (m_Clicked)
    Click(Point.x, Point.y);

  // calculate the map cooridnates
  int x = (Point.x / m_Map->GetTileset().GetTileWidth() + m_CurrentX) / m_ZoomSize;
  int y = (Point.y / m_Map->GetTileset().GetTileHeight() + m_CurrentY) / m_ZoomSize;
  if (x < GetTotalTilesX() && y < GetTotalTilesY())
  {
    CString Position;
    Position.Format("Map (%i,%i)", x, y);
    m_StatusBar->SetWindowText(Position);
  }
  else
    m_StatusBar->SetWindowText("");
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMapView::OnLButtonUp(UINT uFlags, CPoint Point)
{
  m_Clicked = false;
  ReleaseCapture();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMapView::OnRButtonUp(UINT uFlags, CPoint Point)
{
  // get a handle to the menu
  HMENU _menu = LoadMenu(AfxGetApp()->m_hInstance, MAKEINTRESOURCE(IDR_MAPVIEW));
  HMENU menu = GetSubMenu(_menu, 0);

  int tile_width = m_Map->GetTileset().GetTileWidth();
  int tile_height = m_Map->GetTileset().GetTileHeight();

  // tile coordinates
  int tx = (Point.x / m_Map->GetTileset().GetTileWidth()) / m_ZoomSize + m_CurrentX;
  int ty = (Point.y / m_Map->GetTileset().GetTileHeight()) / m_ZoomSize + m_CurrentY;

  // validate the menu items
  bool on_entity = false;
  for (int i = 0; i < m_Map->GetNumEntities(); i++)
    if (m_Map->GetEntity(i).m_X / m_Map->GetTileset().GetTileWidth() == tx &&
        m_Map->GetEntity(i).m_Y / m_Map->GetTileset().GetTileHeight() == ty)
      on_entity = true;

  if (on_entity == false)
  {
    DisableItem(menu, ID_MAPVIEW_DELETEENTITY);
    DisableItem(menu, ID_MAPVIEW_EDITENTITY);
  }
  else
  {
    DisableItem(menu, ID_MAPVIEW_INSERTENTITY_WARP);
    DisableItem(menu, ID_MAPVIEW_INSERTENTITY_PERSON);
    DisableItem(menu, ID_MAPVIEW_INSERTENTITY_TRIGGER);
    DisableItem(menu, ID_MAPVIEW_INSERTENTITY_DOODAD);
  }

  switch (GetZoomSize())
  {
    case 1: SetMenuCheck(menu, ID_MAPVIEW_ZOOM_1X, TRUE); break;
    case 2: SetMenuCheck(menu, ID_MAPVIEW_ZOOM_2X, TRUE); break;
    case 4: SetMenuCheck(menu, ID_MAPVIEW_ZOOM_4X, TRUE); break;
    case 8: SetMenuCheck(menu, ID_MAPVIEW_ZOOM_8X, TRUE); break;
  }

  // show the popup menu
  CPoint Screen = Point;
  ClientToScreen(&Screen);
  BOOL retval = TrackPopupMenu(
    menu,
    TPM_LEFTALIGN | TPM_TOPALIGN | TPM_RETURNCMD | TPM_RIGHTBUTTON,
    Screen.x,
    Screen.y,
    0,
    m_hWnd,
    NULL);

  // convert tile coordinates to pixel coordinates
  int px = tx * tile_width + tile_width / 2;
  int py = ty * tile_height + tile_height / 2;

  // execute command
  switch (retval)
  {
    case ID_MAPVIEW_SELECTTILE:
    {
      int tile = m_Map->GetLayer(m_SelectedLayer).GetTile(tx, ty);
      m_SelectedTile = tile;
      m_Handler->MV_SelectedTileChanged(tile);
      break;
    }

    case ID_MAPVIEW_SETENTRYPOINT:
    {
      m_Map->SetStartX(px);
      m_Map->SetStartY(py);

      Invalidate();
      m_Handler->MV_MapChanged();
      break;
    }

    case ID_MAPVIEW_INSERTENTITY_WARP:
    {
      sWarpEntity warp;
      warp.m_X = px;
      warp.m_Y = py;
      warp.m_Layer = m_SelectedLayer;

      CEntityWarpDialog dialog(warp, m_Map->GetTileset());
      if (dialog.DoModal() == IDOK)
      {
        // insert it into the map
        m_Map->AddEntity(new sWarpEntity(warp));
        Invalidate();
        m_Handler->MV_MapChanged();
      }
      break;
    }

    case ID_MAPVIEW_INSERTENTITY_PERSON:
    {
      sPersonEntity person;
      person.m_X = px;
      person.m_Y = py;
      person.m_Layer = m_SelectedLayer;

      CEntityPersonDialog dialog(person);
      if (dialog.DoModal() == IDOK)
      {
        // insert it into the map
        m_Map->AddEntity(new sPersonEntity(person));
        Invalidate();
        m_Handler->MV_MapChanged();
      }
      break;
    }

    case ID_MAPVIEW_INSERTENTITY_TRIGGER:
    {
      sTriggerEntity trigger;
      trigger.m_X = px;
      trigger.m_Y = py;
      trigger.m_Layer = m_SelectedLayer;

      CEntityTriggerDialog dialog(trigger);
      if (dialog.DoModal() == IDOK)
      {
        // insert it into the map
        m_Map->AddEntity(new sTriggerEntity(trigger));
        Invalidate();
        m_Handler->MV_MapChanged();
      }
      break;
    }

    case ID_MAPVIEW_INSERTENTITY_DOODAD:
    {
      sDoodadEntity doodad;
      doodad.m_X = px;
      doodad.m_Y = py;
      doodad.m_Layer = m_SelectedLayer;

      CEntityDoodadDialog dialog(doodad);
      if (dialog.DoModal() == IDOK)
      {
        // insert it into the map
        m_Map->AddEntity(new sDoodadEntity(doodad));
        Invalidate();
        m_Handler->MV_MapChanged();
      }
      break;
    }

    case ID_MAPVIEW_DELETEENTITY:
    {
      int entity = m_Map->FindEntity(px, py, m_SelectedLayer);
      if (entity != -1)
      {
        m_Map->DeleteEntity(entity);

        Invalidate();
        m_Handler->MV_MapChanged();
      }
      break;
    }

    case ID_MAPVIEW_EDITENTITY:
    {
      int entity = m_Map->FindEntity(px, py, m_SelectedLayer);
      if (entity != -1)
      {
        sEntity& e = m_Map->GetEntity(entity);
        switch (e.GetEntityType())
        {
          case sEntity::WARP:
          {
            CEntityWarpDialog dialog((sWarpEntity&)e, m_Map->GetTileset());
            if (dialog.DoModal() == IDOK)
              m_Handler->MV_MapChanged();
            break;
          }

          case sEntity::PERSON:
          {
            CEntityPersonDialog dialog((sPersonEntity&)e);
            if (dialog.DoModal() == IDOK)
              m_Handler->MV_MapChanged();
            break;
          }
          
          case sEntity::TRIGGER:
          {
            CEntityTriggerDialog dialog((sTriggerEntity&)e);
            if (dialog.DoModal() == IDOK)
              m_Handler->MV_MapChanged();
            break;
          }

          case sEntity::DOODAD:
          {
            CEntityDoodadDialog dialog((sDoodadEntity&)e);
            if (dialog.DoModal() == IDOK)
              m_Handler->MV_MapChanged();
            break;
          }
        }
      }
      break;
    }

    case ID_MAPVIEW_ZOOM_1X:
      SetZoomSize(1);
      break;

    case ID_MAPVIEW_ZOOM_2X:
      SetZoomSize(2);
      break;

    case ID_MAPVIEW_ZOOM_4X:
      SetZoomSize(4);
      break;

    case ID_MAPVIEW_ZOOM_8X:
      SetZoomSize(8);
      break;
  }
}

////////////////////////////////////////////////////////////////////////////////
