#include "ImageWindow.hpp"
#include "FileDialogs.hpp"
#include "resource.h"


#define ID_ALPHASLIDER 900


BEGIN_MESSAGE_MAP(CImageWindow, CSaveableDocumentWindow)

  ON_WM_SIZE()

  ON_COMMAND(ID_IMAGE_VIEWATORIGINALSIZE, OnImageViewOriginalSize)
  ON_COMMAND(ID_IMAGE_SLIDEUP,    OnImageSlideUp)
  ON_COMMAND(ID_IMAGE_SLIDERIGHT, OnImageSlideRight)
  ON_COMMAND(ID_IMAGE_SLIDEDOWN,  OnImageSlideDown)
  ON_COMMAND(ID_IMAGE_SLIDELEFT,  OnImageSlideLeft)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CImageWindow::CImageWindow()
: CSaveableDocumentWindow("", IDR_IMAGE)
, m_Created(false)
{
  m_Image.Create(16, 16);

  // create the window
  Create(AfxRegisterWndClass(0, NULL, (HBRUSH)(COLOR_WINDOW + 1), AfxGetApp()->LoadIcon(IDI_IMAGE)));

  m_ImageView.Create(this, this);
  m_PaletteView.Create(this, this);
  m_ColorView.Create(this, this);
  m_AlphaView.Create(this, this);

  m_Created = true;

  // make sure everything is in the right position
  RECT ClientRect;
  GetClientRect(&ClientRect);
  OnSize(0, ClientRect.right, ClientRect.bottom);

  UpdateImageView();

  SetSaved(false);
  SetModified(false);
}

////////////////////////////////////////////////////////////////////////////////

CImageWindow::CImageWindow(const char* szImage)
: CSaveableDocumentWindow(szImage, IDR_IMAGE)
, m_Created(false)
{
  if (!m_Image.Load(szImage))
  {
    AfxGetApp()->m_pMainWnd->MessageBox("Error: Could not load image");
    delete this;
    return;
  }

  // create the window
  Create(AfxRegisterWndClass(0, NULL, (HBRUSH)(COLOR_WINDOW + 1), AfxGetApp()->LoadIcon(IDI_IMAGE)));

  m_ImageView.Create(this, this);
  m_PaletteView.Create(this, this);
  m_ColorView.Create(this, this);
  m_AlphaView.Create(this, this);

  m_Created = true;

  // make sure everything is in the right position
  RECT ClientRect;
  GetClientRect(&ClientRect);
  OnSize(0, ClientRect.right, ClientRect.bottom);

  UpdateImageView();

  SetSaved(true);
  SetModified(false);
}

////////////////////////////////////////////////////////////////////////////////

CImageWindow::~CImageWindow()
{
  m_ImageView.DestroyWindow();
  m_PaletteView.DestroyWindow();
  m_ColorView.DestroyWindow();
  m_AlphaView.DestroyWindow();
}

////////////////////////////////////////////////////////////////////////////////

void
CImageWindow::UpdateImageView()
{
  m_ImageView.SetImage(
    m_Image.GetWidth(),
    m_Image.GetHeight(),
    m_Image.GetPixels());
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageWindow::OnSize(UINT type, int cx, int cy)
{
  const int palette_width = 40;
  const int color_height  = 40;
  const int alpha_width   = 32;

  if (m_Created)
  {
    m_ImageView.MoveWindow(0, 0, cx - palette_width - alpha_width, cy, FALSE);
    m_PaletteView.MoveWindow(cx - palette_width - alpha_width, 0, palette_width, cy - color_height, FALSE);
    m_ColorView.MoveWindow(cx - palette_width - alpha_width, cy - color_height, palette_width, color_height, FALSE);
    m_AlphaView.MoveWindow(cx - alpha_width, 0, alpha_width, cy, FALSE);

    m_ImageView.Invalidate();
    m_PaletteView.Invalidate();
    m_ColorView.Invalidate();
    m_AlphaView.Invalidate();
  }

  CSaveableDocumentWindow::OnSize(type, cx, cy);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageWindow::OnImageViewOriginalSize()
{
  CSaveableDocumentWindow::SetWindowPos(&wndTop, 0, 0, m_Image.GetWidth() + 72 + 12, m_Image.GetHeight() + 31, SWP_NOMOVE | SWP_NOOWNERZORDER);
  OnSize(0, m_Image.GetWidth() + 72, m_Image.GetHeight());
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageWindow::OnImageSlideUp()
{
  m_Image.Translate(0, -1);
  UpdateImageView();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageWindow::OnImageSlideRight()
{
  m_Image.Translate(1, 0);
  UpdateImageView();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageWindow::OnImageSlideDown()
{
  m_Image.Translate(0, 1);
  UpdateImageView();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageWindow::OnImageSlideLeft()
{
  m_Image.Translate(-1, 0);
  UpdateImageView();
}

////////////////////////////////////////////////////////////////////////////////

bool
CImageWindow::GetSavePath(char* path)
{
  CImageFileDialog Dialog(FDM_SAVE);
  if (Dialog.DoModal() != IDOK)
    return false;

  strcpy(path, Dialog.GetPathName());
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CImageWindow::SaveDocument(const char* path)
{
  return m_Image.Save(path);
}

////////////////////////////////////////////////////////////////////////////////

void
CImageWindow::IV_ImageChanged()
{
  memcpy(m_Image.GetPixels(), m_ImageView.GetPixels(), m_Image.GetWidth() * m_Image.GetHeight() * sizeof(RGBA));
  SetModified(true);
}

////////////////////////////////////////////////////////////////////////////////

void
CImageWindow::IV_ColorChanged(RGBA color)
{
  RGB rgb = { color.red, color.green, color.blue };
  m_ColorView.SetColor(rgb);
  m_AlphaView.SetAlpha(color.alpha);
}

////////////////////////////////////////////////////////////////////////////////

void
CImageWindow::PV_ColorChanged(RGB color)
{
  m_ColorView.SetColor(color);

  RGBA rgba = { color.red, color.green, color.blue, m_ImageView.GetColor().alpha };
  m_ImageView.SetColor(rgba);
}

////////////////////////////////////////////////////////////////////////////////

void
CImageWindow::CV_ColorChanged(RGB color)
{
  RGBA rgba = { color.red, color.green, color.blue, m_ImageView.GetColor().alpha };
  m_ImageView.SetColor(rgba);
}

////////////////////////////////////////////////////////////////////////////////

void
CImageWindow::AV_AlphaChanged(byte alpha)
{
  RGBA rgba = m_ImageView.GetColor();
  rgba.alpha = alpha;
  m_ImageView.SetColor(rgba);
}

////////////////////////////////////////////////////////////////////////////////
