#include "ImageView.hpp"
#include "x++.hpp"


static int s_ImageViewID = 9000;

static UINT s_ClipboardFormat;


BEGIN_MESSAGE_MAP(CImageView, CWnd)

  ON_WM_PAINT()
  ON_WM_SIZE()

  ON_WM_LBUTTONDOWN()
  ON_WM_LBUTTONUP()
  ON_WM_RBUTTONDOWN()
  ON_WM_MOUSEMOVE()

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CImageView::CImageView()
: m_Width(0)
, m_Height(0)
, m_Pixels(NULL)
, m_Color(rgbaBlack)
, m_MouseDown(false)
, m_NumUndoImages(0)
, m_UndoImages(NULL)
{
  s_ClipboardFormat = RegisterClipboardFormat("FlatImage32");
}

////////////////////////////////////////////////////////////////////////////////

CImageView::~CImageView()
{
}

////////////////////////////////////////////////////////////////////////////////

BOOL
CImageView::Create(CImageViewHandler* handler, CWnd* parent_window)
{
  m_Handler = handler;

  return CWnd::Create(
    AfxRegisterWndClass(0, LoadCursor(NULL, MAKEINTRESOURCE(IDC_ARROW)), NULL, LoadIcon(NULL, IDI_APPLICATION)),
    "ImageView",
    WS_CHILD | WS_VISIBLE,
    CRect(0, 0, 0, 0),
    parent_window,
    s_ImageViewID++);
}

////////////////////////////////////////////////////////////////////////////////

bool
CImageView::SetImage(const CImage32& image)
{
  return SetImage(image.GetWidth(), image.GetHeight(), image.GetPixels());
}

////////////////////////////////////////////////////////////////////////////////

bool
CImageView::SetImage(int width, int height, const RGBA* pixels)
{
  ResetUndoStates();

  m_Width  = width;
  m_Height = height;
  delete[] m_Pixels;
  m_Pixels = new RGBA[width * height];
  memcpy(m_Pixels, pixels, width * height * sizeof(RGBA));

  Invalidate();
  return true;
}

////////////////////////////////////////////////////////////////////////////////

int
CImageView::GetWidth() const
{
  return m_Width;
}

////////////////////////////////////////////////////////////////////////////////

int
CImageView::GetHeight() const
{
  return m_Height;
}

////////////////////////////////////////////////////////////////////////////////

RGBA*
CImageView::GetPixels()
{
  return m_Pixels;
}

////////////////////////////////////////////////////////////////////////////////

const RGBA*
CImageView::GetPixels() const
{
  return m_Pixels;
}

////////////////////////////////////////////////////////////////////////////////

void
CImageView::SetColor(RGBA color)
{
  m_Color = color;
}

////////////////////////////////////////////////////////////////////////////////

RGBA
CImageView::GetColor() const
{
  return m_Color;
}

////////////////////////////////////////////////////////////////////////////////

void
CImageView::FillRGB()
{
  AddUndoState();

  for (int i = 0; i < m_Width * m_Height; i++)
  {
    m_Pixels[i].red   = m_Color.red;
    m_Pixels[i].green = m_Color.green;
    m_Pixels[i].blue  = m_Color.blue;
  }

  Invalidate();
  m_Handler->IV_ImageChanged();
}

////////////////////////////////////////////////////////////////////////////////

void
CImageView::FillAlpha()
{
  AddUndoState();

  for (int i = 0; i < m_Width * m_Height; i++)
    m_Pixels[i].alpha = m_Color.alpha;

  Invalidate();
  m_Handler->IV_ImageChanged();
}

////////////////////////////////////////////////////////////////////////////////

bool
CImageView::Copy()
{
  if (OpenClipboard() == FALSE)
    return false;

  // clear the previous contents of the clipboard
  EmptyClipboard();

  // ADD FLAT 32

  // copy the image as a flat 32-bit color image
  HGLOBAL memory = GlobalAlloc(GHND, 4 + 4 + m_Width * m_Height * 4);
  dword* ptr = (dword*)GlobalLock(memory);

  *ptr++ = m_Width;
  *ptr++ = m_Height;
  memcpy(ptr, m_Pixels, m_Width * m_Height * sizeof(RGBA));

  // put the image on the clipboard
  GlobalUnlock(memory);
  SetClipboardData(s_ClipboardFormat, memory);

  // ADD DDB

  // create a pixel array to initialize the bitmap
  BGRA* pixels = new BGRA[m_Width * m_Height];
  for (int iy = 0; iy < m_Height; iy++)
    for (int ix = 0; ix < m_Width; ix++)
    {
      pixels[iy * m_Width + ix].red   = m_Pixels[iy * m_Width + ix].red;
      pixels[iy * m_Width + ix].green = m_Pixels[iy * m_Width + ix].green;
      pixels[iy * m_Width + ix].blue  = m_Pixels[iy * m_Width + ix].blue;
    }

  // create the bitmap
  HBITMAP bitmap = CreateBitmap(m_Width, m_Height, 1, 32, pixels);

  // put the bitmap in the clipboard
  SetClipboardData(CF_BITMAP, bitmap);
  delete[] pixels;
  DeleteObject(bitmap);

  CloseClipboard();

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CImageView::Paste()
{
  if (OpenClipboard() == FALSE)
    return false;

  AddUndoState();

  // see if the flat image is in the clipboard
  HGLOBAL memory = (HGLOBAL)GetClipboardData(s_ClipboardFormat);
  if (memory != NULL)
  {
    // get the height and pixels from the clipboard
    dword* ptr = (dword*)GlobalLock(memory);
    int width = *ptr++;
    int height = *ptr++;
    RGBA* pixels = new RGBA[width * height];
    memcpy(pixels, ptr, width * height * sizeof(RGBA));

    // put them into the current view
    for (int iy = 0; iy < m_Height; iy++)
      for (int ix = 0; ix < m_Width; ix++)
      {
        if (ix < width && iy < height)
          m_Pixels[iy * m_Width + ix] = pixels[iy * width + ix];
        else
          m_Pixels[iy * m_Width + ix] = rgbaBlack;
      }

    delete[] pixels;

    // things have changed
    Invalidate();
    m_Handler->IV_ImageChanged();

    return true;
  }

  // grab a bitmap out of the clipboard
  HBITMAP bitmap = (HBITMAP)GetClipboardData(CF_BITMAP);
  if (bitmap != NULL)
  {
    BITMAP b;
    GetObject(bitmap, sizeof(b), &b);

    HDC dc = CreateCompatibleDC(NULL);
    SelectObject(dc, bitmap);

    for (int iy = 0; iy < m_Height; iy++)
      for (int ix = 0; ix < m_Width; ix++)
      {
        COLORREF pixel = GetPixel(dc, ix, iy);
        if (pixel == CLR_INVALID)
          pixel = RGB(0, 0, 0);
        m_Pixels[iy * m_Width + ix].red   = GetRValue(pixel);
        m_Pixels[iy * m_Width + ix].green = GetGValue(pixel);
        m_Pixels[iy * m_Width + ix].blue  = GetBValue(pixel);
        m_Pixels[iy * m_Width + ix].alpha = 255;
      }

    DeleteDC(dc);

    CloseClipboard();

    // things have changed
    Invalidate();
    m_Handler->IV_ImageChanged();

    return true;
  }

  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool
CImageView::CanUndo() const
{
  return m_NumUndoImages > 0;
}

////////////////////////////////////////////////////////////////////////////////

void
CImageView::Undo()
{
  Image* i = m_UndoImages + m_NumUndoImages - 1;
  memcpy(m_Pixels, i->pixels, m_Width * m_Height * sizeof(RGBA));
  delete[] i->pixels;

  Image* new_images = new Image[m_NumUndoImages - 1];
  for (int i = 0; i < m_NumUndoImages - 1; i++)
    new_images[i] = m_UndoImages[i];

  m_NumUndoImages--;
  delete[] m_UndoImages;
  m_UndoImages = new_images;

  Invalidate();
  m_Handler->IV_ImageChanged();
}

////////////////////////////////////////////////////////////////////////////////

void
CImageView::Click(int x, int y)
{
  if (m_Width == 0 || m_Height == 0)
    return;

  // convert pixel coordinates to image coordinates

  // get client rectangle
  RECT ClientRect;
  GetClientRect(&ClientRect);

  // calculate size of pixel squares
  int hsize = ClientRect.right / m_Width;
  int vsize = ClientRect.bottom / m_Height;
  int size = min(hsize, vsize);
  if (size < 1)
    size = 1;

  int totalx = size * m_Width;
  int totaly = size * m_Height;
  int offsetx = (ClientRect.right - totalx) / 2;
  int offsety = (ClientRect.bottom - totaly) / 2;

  x -= offsetx;
  y -= offsety;

  x /= size;
  y /= size;

  // bounds check
  if (x < 0 ||
      y < 0 ||
      x >= m_Width ||
      y >= m_Height)
    return;

  // now that we have image coordinates, we can update the image
  if (memcmp(m_Pixels + y * m_Width + x, &m_Color, sizeof(RGBA)) != 0)
  {
    m_Pixels[y * m_Width + x] = m_Color;

    // invalidate the image
    Invalidate();

    m_Handler->IV_ImageChanged();
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CImageView::GetColor(RGBA* color, int x, int y)
{
  // convert pixel coordinates to image coordinates

  // get client rectangle
  RECT ClientRect;
  GetClientRect(&ClientRect);

  // calculate size of pixel squares
  int hsize = ClientRect.right / m_Width;
  int vsize = ClientRect.bottom / m_Height;
  int size = min(hsize, vsize);
  if (size < 1)
    size = 1;

  int totalx = size * m_Width;
  int totaly = size * m_Height;
  int offsetx = (ClientRect.right - totalx) / 2;
  int offsety = (ClientRect.bottom - totaly) / 2;

  x -= offsetx;
  y -= offsety;

  x /= size;
  y /= size;

  // bounds check
  if (x < 0 ||
      y < 0 ||
      x >= m_Width ||
      y >= m_Height)
    return;

  // now that we have image coordinates, we can update the image
  if (memcmp(m_Pixels + y * m_Width + x, &m_Color, sizeof(RGBA)) != 0)
  {
    m_Color = m_Pixels[y * m_Width + x];
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CImageView::AddUndoState()
{
  Image* new_images = new Image[m_NumUndoImages + 1];
  for (int i = 0; i < m_NumUndoImages; i++)
    new_images[i] = m_UndoImages[i];

  new_images[m_NumUndoImages].width = m_Width;
  new_images[m_NumUndoImages].height = m_Height;
  new_images[m_NumUndoImages].pixels = new RGBA[m_Width * m_Height];
  memcpy(new_images[m_NumUndoImages].pixels, m_Pixels, m_Width * m_Height * sizeof(RGBA));

  m_NumUndoImages++;
  delete[] m_UndoImages;
  m_UndoImages = new_images;
}

////////////////////////////////////////////////////////////////////////////////

void
CImageView::ResetUndoStates()
{
  for (int i = 0; i < m_NumUndoImages; i++)
    delete[] m_UndoImages[i].pixels;

  delete[] m_UndoImages;
  m_UndoImages = NULL;
  m_NumUndoImages = 0;
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageView::OnPaint()
{
  CPaintDC _dc(this);
  HDC dc = _dc.m_hDC;

  if (m_Width == 0 || m_Height == 0 || m_Pixels == NULL)
    return;

  // get client rectangle
  RECT ClientRect;
  GetClientRect(&ClientRect);

  // calculate size of pixel squares
  int hsize = ClientRect.right / m_Width;
  int vsize = ClientRect.bottom / m_Height;
  int size = min(hsize, vsize);
  if (size < 1)
    size = 1;

  int totalx = size * m_Width;
  int totaly = size * m_Height;
  int offsetx = (ClientRect.right - totalx) / 2;
  int offsety = (ClientRect.bottom - totaly) / 2;

  // draw black rectangles in the empty parts
  HBRUSH black_brush = (HBRUSH)GetStockObject(BLACK_BRUSH);
  RECT Rect;

  // top
  SetRect(&Rect, 0, 0, ClientRect.right, offsety - 1);
  FillRect(dc, &Rect, black_brush);

  // bottom
  SetRect(&Rect, 0, offsety + totaly + 1, ClientRect.right, ClientRect.bottom);
  FillRect(dc, &Rect, black_brush);

  // left
  SetRect(&Rect, 0, offsety - 1, offsetx - 1, offsety + totaly + 1);
  FillRect(dc, &Rect, black_brush);

  // right
  SetRect(&Rect, offsetx + totalx + 1, offsety - 1, ClientRect.right, offsety + totaly + 1);
  FillRect(dc, &Rect, black_brush);

  // draw the image
  for (int ix = 0; ix < m_Width; ix++)
    for (int iy = 0; iy < m_Height; iy++)
    {
      RGBA color = m_Pixels[iy * m_Width + ix];

      // opaque
      if (color.alpha == 255)
      {
        HBRUSH brush = CreateSolidBrush(RGB(color.red, color.green, color.blue));

        RECT Rect = { ix * size, iy * size, ix * size + size, iy * size + size };
        OffsetRect(&Rect, offsetx, offsety);
        FillRect(dc, &Rect, brush);

        DeleteObject(brush);
      }
      // translucent
      else
      {
        // calculate background grid colors
        RGB Color1 = rgbWhite;
        RGB Color2 = rgbLightGrey;

        Color1.red   = (color.red   * color.alpha + Color1.red   * (256 - color.alpha)) / 256;
        Color1.green = (color.green * color.alpha + Color1.green * (256 - color.alpha)) / 256;
        Color1.blue  = (color.blue  * color.alpha + Color1.blue  * (256 - color.alpha)) / 256;

        Color2.red   = (color.red   * color.alpha + Color2.red   * (256 - color.alpha)) / 256;
        Color2.green = (color.green * color.alpha + Color2.green * (256 - color.alpha)) / 256;
        Color2.blue  = (color.blue  * color.alpha + Color2.blue  * (256 - color.alpha)) / 256;

        HBRUSH Brush1 = CreateSolidBrush(RGB(Color1.red, Color1.green, Color1.blue));
        HBRUSH Brush2 = CreateSolidBrush(RGB(Color2.red, Color2.green, Color2.blue));

        RECT Rect;

        // draw rectangles

        // upper left
        SetRect(&Rect,
          ix * size,
          iy * size,
          ix * size + size / 2,
          iy * size + size / 2);
        OffsetRect(&Rect, offsetx, offsety);
        FillRect(dc, &Rect, Brush1);

        // upper right
        SetRect(&Rect,
          ix * size + size / 2,
          iy * size,
          ix * size + size,
          iy * size + size / 2);
        OffsetRect(&Rect, offsetx, offsety);
        FillRect(dc, &Rect, Brush2);

        // lower left
        SetRect(&Rect,
          ix * size,
          iy * size + size / 2,
          ix * size + size / 2,
          iy * size + size);
        OffsetRect(&Rect, offsetx, offsety);
        FillRect(dc, &Rect, Brush2);

        // lower right
        SetRect(&Rect,
          ix * size + size / 2,
          iy * size + size / 2,
          ix * size + size,
          iy * size + size);
        OffsetRect(&Rect, offsetx, offsety);
        FillRect(dc, &Rect, Brush1);
       
        DeleteObject(Brush1);
        DeleteObject(Brush2);
      }

    }

  // draw a white rectangle around the image
  SetRect(&Rect, offsetx - 1, offsety - 1, offsetx + totalx + 1, offsety + totaly + 1);

  HPEN   white_pen = CreatePen(PS_SOLID, 1, RGB(0xFF, 0xFF, 0xFF));
  HBRUSH old_brush = (HBRUSH)SelectObject(dc, GetStockObject(NULL_BRUSH));
  HPEN   old_pen   = (HPEN)  SelectObject(dc, white_pen);
  
  Rectangle(dc, Rect.left, Rect.top, Rect.right, Rect.bottom);

  SelectObject(dc, old_pen);
  SelectObject(dc, old_brush);
  DeleteObject(white_pen);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageView::OnSize(UINT type, int cx, int cy)
{
  Invalidate();
  CWnd::OnSize(type, cx, cy);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageView::OnLButtonDown(UINT flags, CPoint point)
{
  AddUndoState();

  Click(point.x, point.y);

  m_MouseDown = true;
  SetCapture();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageView::OnLButtonUp(UINT flags, CPoint point)
{
  m_MouseDown = false;
  ReleaseCapture();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageView::OnRButtonDown(UINT flags, CPoint point)
{
  GetColor(&m_Color, point.x, point.y);
  m_Handler->IV_ColorChanged(m_Color);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CImageView::OnMouseMove(UINT flags, CPoint point)
{
  if (!m_MouseDown)
    return;
  
  Click(point.x, point.y);
}

////////////////////////////////////////////////////////////////////////////////
