#include "DocumentWindow.hpp"
#include "WindowCommands.hpp"
#include "WindowAttributes.hpp"
#include "resource.h"


BEGIN_MESSAGE_MAP(CDocumentWindow, CMDIChildWnd)

  ON_WM_CREATE()
  ON_WM_CLOSE()
  ON_WM_SIZING()
  ON_WM_MDIACTIVATE()

  ON_UPDATE_COMMAND_UI(ID_FILE_SAVE,       OnUpdateSaveableCommand)
  ON_UPDATE_COMMAND_UI(ID_FILE_SAVEAS,     OnUpdateSaveableCommand)
  ON_UPDATE_COMMAND_UI(ID_FILE_SAVECOPYAS, OnUpdateSaveableCommand)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CDocumentWindow::CDocumentWindow(const char* document_path, int menu_resource, const CSize& min_size)
: m_MenuResource(menu_resource)
, m_MinSize(min_size)
, m_NumPalettes(0)
, m_Palettes(NULL)
{
  strcpy(m_DocumentPath, document_path);
  strcpy(m_Caption, GetDocumentTitle());
}

////////////////////////////////////////////////////////////////////////////////

CDocumentWindow::~CDocumentWindow()
{
  for (int i = 0; i < m_NumPalettes; i++)
  {
    m_Palettes[i]->SendMessage(WM_CLOSE);
    m_Palettes[i]->DestroyWindow();
  }
  realloc(m_Palettes, 0);

  // this destructor can be called from a constructor, and the main window would
  // try to remove the window pointer before it was added
  AfxGetApp()->m_pMainWnd->PostMessage(WM_DW_CLOSING, 0, (LPARAM)this);
}

////////////////////////////////////////////////////////////////////////////////

bool
CDocumentWindow::Close()
{
  return true;
}

////////////////////////////////////////////////////////////////////////////////

BOOL
CDocumentWindow::Create(LPCTSTR class_name)
{
  BOOL result = CMDIChildWnd::Create(class_name, "");
  UpdateWindowCaption();
  return result;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CDocumentWindow::GetDefaultWindowClass()
{
  return AfxRegisterWndClass(0, ::LoadCursor(NULL, MAKEINTRESOURCE(IDC_ARROW)), NULL, NULL);
}

////////////////////////////////////////////////////////////////////////////////

void
CDocumentWindow::SetDocumentPath(const char* path)
{
  strcpy(m_DocumentPath, path);
}

////////////////////////////////////////////////////////////////////////////////

const char*
CDocumentWindow::GetDocumentPath() const
{
  return m_DocumentPath;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CDocumentWindow::GetDocumentTitle() const
{
  if (strrchr(m_DocumentPath, '\\') == NULL)
    return m_DocumentPath;
  else
    return strrchr(m_DocumentPath, '\\') + 1;
}

////////////////////////////////////////////////////////////////////////////////

void
CDocumentWindow::SetCaption(const char* caption)
{
  strcpy(m_Caption, caption);
  UpdateWindowCaption();
}

////////////////////////////////////////////////////////////////////////////////

const char*
CDocumentWindow::GetCaption()
{
  return m_Caption;
}

////////////////////////////////////////////////////////////////////////////////

void
CDocumentWindow::UpdateWindowCaption()
{
  SetWindowText(m_Caption);
}

////////////////////////////////////////////////////////////////////////////////

bool
CDocumentWindow::IsSaveable() const
{
  return false;
}

////////////////////////////////////////////////////////////////////////////////

void
CDocumentWindow::AttachPalette(CWnd* palette)
{
  m_Palettes = (CWnd**)realloc(m_Palettes, sizeof(CWnd*) * (m_NumPalettes + 1));
  m_Palettes[m_NumPalettes] = palette;
  m_NumPalettes++;
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CDocumentWindow::OnCreate(LPCREATESTRUCT cs)
{
  SetWindowLong(m_hWnd, GWL_USERDATA, IsSaveable() ? WA_SAVEABLE : 0);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CDocumentWindow::OnClose()
{
  if (Close())
    CMDIChildWnd::OnClose();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CDocumentWindow::OnSizing(UINT side, LPRECT rect)
{
  CMDIChildWnd::OnSizing(side, rect);

  if (m_MinSize == CSize(0, 0))
    return;

  int width = rect->right - rect->left;
  int height = rect->bottom - rect->top;

  // check sides; if they are too small, resize them

  if (side == WMSZ_TOPRIGHT ||
      side == WMSZ_RIGHT ||
      side == WMSZ_BOTTOMRIGHT)
  {
    if (width < m_MinSize.cx)
      rect->right = rect->left + m_MinSize.cx;
  }

  if (side == WMSZ_TOPLEFT ||
      side == WMSZ_LEFT ||
      side == WMSZ_BOTTOMLEFT)
  {
    if (width < m_MinSize.cx)
      rect->left = rect->right - m_MinSize.cx;
  }

  if (side == WMSZ_BOTTOMLEFT ||
      side == WMSZ_BOTTOM ||
      side == WMSZ_BOTTOMRIGHT)
  {
    if (height < m_MinSize.cy)
      rect->bottom = rect->top + m_MinSize.cy;
  }

  if (side == WMSZ_TOPLEFT ||
      side == WMSZ_TOP ||
      side == WMSZ_TOPRIGHT)
  {
    if (height < m_MinSize.cy)
      rect->top = rect->bottom - m_MinSize.cy;
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CDocumentWindow::OnMDIActivate(BOOL activate, CWnd* active_window, CWnd* inactive_window)
{
  if (activate)
  {
    // set the child menu resource
    AfxGetApp()->m_pMainWnd->SendMessage(WM_SET_CHILD_MENU, m_MenuResource);

    // show the palettes
    for (int i = 0; i < m_NumPalettes; i++)
      m_Palettes[i]->ShowWindow(SW_SHOW);
  }
  else
  {
    // clear the child menu
    AfxGetApp()->m_pMainWnd->SendMessage(WM_CLEAR_CHILD_MENU);

    // hide the palettes
    for (int i = 0; i < m_NumPalettes; i++)
      m_Palettes[i]->ShowWindow(SW_HIDE);
  }
  
  CMDIChildWnd::OnMDIActivate(activate, active_window, inactive_window);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CDocumentWindow::OnUpdateSaveableCommand(CCmdUI* cmdui)
{
  cmdui->Enable(IsSaveable() == true);
}

////////////////////////////////////////////////////////////////////////////////
