#include <windows.h>
#include <prsht.h>
#include <stdio.h>
#include "../engine/win32/sphere_config.h"
#include "x++.hpp"
#include "resource.h"


#define DRIVER_DIRECTORY "system/video"


struct DRIVERLIST
{
  int    num_drivers;
  char** drivers;
};


struct DRIVERINFO
{
  char* name;
  char* author;
  char* date;
  char* version;
  char* description;
};


void BuildDriverList();
void DestroyDriverList();
void LoadConfiguration();
void SaveConfiguration();
void ExecuteDialog();

bool        IsValidDriver(const char* filename);
void        InitializeDriverList();
bool        AddDriver(const char* filename);

BOOL CALLBACK VideoDialogProc(HWND window, UINT message, WPARAM wparam, LPARAM lparam);
BOOL CALLBACK AudioDialogProc(HWND window, UINT message, WPARAM wparam, LPARAM lparam);
BOOL CALLBACK InputDialogProc(HWND window, UINT message, WPARAM wparam, LPARAM lparam);

bool ConfigureDriver(HWND window, const char* driver);
bool GetDriverInfo(const char* drivername, DRIVERINFO* driverinfo);
bool FreeDriverInfo(DRIVERINFO* driverinfo);


HINSTANCE    MainInstance;
char         ConfigFile[MAX_PATH];
SPHERECONFIG Config;
DRIVERLIST   DriverList;


////////////////////////////////////////////////////////////////////////////////

int WINAPI WinMain(HINSTANCE instance, HINSTANCE, LPSTR, int)
{
  MainInstance = instance;

  BuildDriverList();
  if (DriverList.num_drivers == 0)
  {
    MessageBox(
      NULL,
      "No video drivers found in <sphere>/system/video/.\n"
      "This probably means Sphere was installed incorrectly.\n"
      "Sphere Configuration cannot continue.",
      "Sphere Configuration",
      MB_OK);
    return 1;
  }

  LoadConfiguration();
  ExecuteDialog();
  SaveConfiguration();

  DestroyDriverList();
  return 0;
}

////////////////////////////////////////////////////////////////////////////////

void BuildDriverList()
{
  InitializeDriverList();

  // enter system/video directory
  char old_directory[MAX_PATH];
  GetCurrentDirectory(MAX_PATH, old_directory);
  SetCurrentDirectory(DRIVER_DIRECTORY);

  // enumerate DLLs
  WIN32_FIND_DATA ffd;
  HANDLE handle = FindFirstFile("*.dll", &ffd);
  if (handle != INVALID_HANDLE_VALUE)
  {
    do
    {
      // if it's a valid driver, add it to the list
      if (IsValidDriver(ffd.cFileName))
        AddDriver(ffd.cFileName);

    } while (FindNextFile(handle, &ffd));

    FindClose(handle);
  }

  SetCurrentDirectory(old_directory);
}

////////////////////////////////////////////////////////////////////////////////

void DestroyDriverList()
{
  for (int i = 0; i < DriverList.num_drivers; i++)
  {
    delete[] DriverList.drivers[i];
  }
  delete[] DriverList.drivers;
}

////////////////////////////////////////////////////////////////////////////////

void LoadConfiguration()
{  
  // create the config filename
#ifdef NDEBUG
  GetModuleFileName(MainInstance, ConfigFile, MAX_PATH);
  if (*strrchr(ConfigFile, '\\'))
  {
    *strrchr(ConfigFile, '\\') = 0;
    strcat(ConfigFile, "\\engine.ini");
  }
  else
    strcpy(ConfigFile, "engine.ini");
#else
  GetCurrentDirectory(MAX_PATH, ConfigFile);
  strcat(ConfigFile, "\\engine.ini");
#endif

  // load the configuration
  LoadSphereConfig(&Config, ConfigFile);
}

////////////////////////////////////////////////////////////////////////////////

void SaveConfiguration()
{
  // must be called after LoadConfiguration()
  // save the configuration
  SaveSphereConfig(&Config, ConfigFile);
}

////////////////////////////////////////////////////////////////////////////////

void ExecuteDialog()
{
  PROPSHEETPAGE Pages[3];

  // default values
  for (int i = 0; i < arraysize(Pages); i++)
  {
    Pages[i].dwSize    = sizeof(Pages[i]);
    Pages[i].dwFlags   = PSP_DEFAULT;
    Pages[i].hInstance = MainInstance;
    Pages[i].hIcon     = NULL;
  }
  
  // video page
  Pages[0].pszTemplate = MAKEINTRESOURCE(IDD_VIDEO_PAGE);
  Pages[0].pfnDlgProc  = VideoDialogProc;

  // audio page
  Pages[1].pszTemplate = MAKEINTRESOURCE(IDD_AUDIO_PAGE);
  Pages[1].pfnDlgProc  = AudioDialogProc;

  // input page
  Pages[2].pszTemplate = MAKEINTRESOURCE(IDD_INPUT_PAGE);
  Pages[2].pfnDlgProc  = InputDialogProc;

  // create the dialog box
  PROPSHEETHEADER psh;
  memset(&psh, 0, sizeof(psh));
  psh.dwSize      = sizeof(psh);
  psh.dwFlags     = PSH_NOAPPLYNOW | PSH_PROPSHEETPAGE;
  psh.hwndParent  = NULL;
  psh.hInstance   = MainInstance;
  psh.hIcon       = NULL;
  psh.pszCaption  = "Sphere Configuration";
  psh.nPages      = arraysize(Pages);
  psh.nStartPage  = 0;
  psh.ppsp        = Pages;

  PropertySheet(&psh);
}

////////////////////////////////////////////////////////////////////////////////

BOOL CALLBACK VideoDialogProc(HWND window, UINT message, WPARAM wparam, LPARAM lparam)
{
  switch (message)
  {
    case WM_INITDIALOG:
    {
      for (int i = 0; i < DriverList.num_drivers; i++)
        SendDlgItemMessage(window, IDC_DRIVERLIST, LB_ADDSTRING, 0, (LPARAM)DriverList.drivers[i]);

      SendDlgItemMessage(window, IDC_DRIVERLIST, LB_SELECTSTRING, 0, (LPARAM)Config.videodriver);

      // update the driver information
      SendMessage(window, WM_COMMAND, MAKEWPARAM(IDC_DRIVERLIST, LBN_SELCHANGE), 0);

      SetFocus(GetDlgItem(window, IDC_DRIVERLIST));
      return FALSE;
    }

    ////////////////////////////////////////////////////////////////////////////

    case WM_COMMAND:
      // driver configure
      if (LOWORD(wparam) == IDC_CONFIGURE_DRIVER)
      {
        char driver[MAX_PATH];
        int sel = SendDlgItemMessage(window, IDC_DRIVERLIST, LB_GETCURSEL, 0, 0);
        SendDlgItemMessage(window, IDC_DRIVERLIST, LB_GETTEXT, sel, (LPARAM)driver);
        ConfigureDriver(window, driver);
        return TRUE;
      }

      // new driver has been selected
      if (LOWORD(wparam) == IDC_DRIVERLIST && HIWORD(wparam) == LBN_SELCHANGE)
      {
        char drivername[MAX_PATH];
        int sel = SendDlgItemMessage(window, IDC_DRIVERLIST, LB_GETCURSEL, 0, 0);
        if (sel == -1)
          return TRUE;

        SendDlgItemMessage(window, IDC_DRIVERLIST, LB_GETTEXT, sel, (LPARAM)drivername);

        // retrieve the driver's information and put it in the controls
        DRIVERINFO driverinfo;
        GetDriverInfo(drivername, &driverinfo);

        SetDlgItemText(window, IDC_NAME,        driverinfo.name);
        SetDlgItemText(window, IDC_AUTHOR,      driverinfo.author);
        SetDlgItemText(window, IDC_DATE,        driverinfo.date);
        SetDlgItemText(window, IDC_VERSION,     driverinfo.version);
        SetDlgItemText(window, IDC_DESCRIPTION, driverinfo.description);

        FreeDriverInfo(&driverinfo);

        return TRUE;
      }

      return FALSE;

    ////////////////////////////////////////////////////////////////////////////

    case WM_NOTIFY:
    {
      PSHNOTIFY* psn = (PSHNOTIFY*)lparam;
      if (psn->hdr.code == PSN_APPLY)
      {
        int sel = SendDlgItemMessage(window, IDC_DRIVERLIST, LB_GETCURSEL, 0, 0);
        SendDlgItemMessage(window, IDC_DRIVERLIST, LB_GETTEXT, sel, (LPARAM)Config.videodriver);
        return TRUE;
      }

      return FALSE;
    }

    ////////////////////////////////////////////////////////////////////////////

    default:
      return FALSE;
  }
}

////////////////////////////////////////////////////////////////////////////////

BOOL CALLBACK AudioDialogProc(HWND window, UINT message, WPARAM wparam, LPARAM lparam)
{
  switch (message)
  {
    case WM_INITDIALOG:
      // output driver
      SendDlgItemMessage(window, IDC_OUTPUT_DRIVER, CB_ADDSTRING, 0, (LPARAM)"No sound");
      SendDlgItemMessage(window, IDC_OUTPUT_DRIVER, CB_ADDSTRING, 0, (LPARAM)"Windows Multimedia");
      SendDlgItemMessage(window, IDC_OUTPUT_DRIVER, CB_ADDSTRING, 0, (LPARAM)"DirectSound");
      SendDlgItemMessage(window, IDC_OUTPUT_DRIVER, CB_SETCURSEL, Config.output_driver, 0);

      // mixers
      SendDlgItemMessage(window, IDC_MIXER, CB_ADDSTRING, 0, (LPARAM)"Autodetect");
      SendDlgItemMessage(window, IDC_MIXER, CB_ADDSTRING, 0, (LPARAM)"Standard");
      SendDlgItemMessage(window, IDC_MIXER, CB_ADDSTRING, 0, (LPARAM)"Pentium MMX");
      SendDlgItemMessage(window, IDC_MIXER, CB_ADDSTRING, 0, (LPARAM)"PPro/P5/P6 MMX");
      SendDlgItemMessage(window, IDC_MIXER, CB_ADDSTRING, 0, (LPARAM)"Interpolation - Autodetect");
      SendDlgItemMessage(window, IDC_MIXER, CB_ADDSTRING, 0, (LPARAM)"Interpolation - FPU");
      SendDlgItemMessage(window, IDC_MIXER, CB_ADDSTRING, 0, (LPARAM)"Interpolation - MMX P5");
      SendDlgItemMessage(window, IDC_MIXER, CB_ADDSTRING, 0, (LPARAM)"Interpolation - MMX P6");
      SendDlgItemMessage(window, IDC_MIXER, CB_SETCURSEL, Config.mixer, 0);

      if (Config.stereo)
        CheckDlgButton(window, IDC_STEREO, BST_CHECKED);

      switch (Config.bit_rate)
      {
        case 8000:  CheckDlgButton(window, IDC_OUTPUT_RATE_8000,  BST_CHECKED); break;
        case 11025: CheckDlgButton(window, IDC_OUTPUT_RATE_11025, BST_CHECKED); break;
        case 22050: CheckDlgButton(window, IDC_OUTPUT_RATE_22050, BST_CHECKED); break;
        case 44100: CheckDlgButton(window, IDC_OUTPUT_RATE_44100, BST_CHECKED); break;
      }

      SetFocus(GetDlgItem(window, IDC_OUTPUT_DRIVER));
      return FALSE;

    ////////////////////////////////////////////////////////////////////////////

    case WM_NOTIFY:
    {
      PSHNOTIFY* psn = (PSHNOTIFY*)lparam;
      if (psn->hdr.code == PSN_APPLY)
      {
        Config.output_driver = SendDlgItemMessage(window, IDC_OUTPUT_DRIVER, CB_GETCURSEL, 0, 0);
        Config.mixer         = SendDlgItemMessage(window, IDC_MIXER,         CB_GETCURSEL, 0, 0);

        if (IsDlgButtonChecked(window, IDC_OUTPUT_RATE_44100) == BST_CHECKED)
          Config.bit_rate = 44100;
        else if (IsDlgButtonChecked(window, IDC_OUTPUT_RATE_22050) == BST_CHECKED)
          Config.bit_rate = 22050;
        else if (IsDlgButtonChecked(window, IDC_OUTPUT_RATE_11025) == BST_CHECKED)
          Config.bit_rate = 11025;
        else if (IsDlgButtonChecked(window, IDC_OUTPUT_RATE_8000) == BST_CHECKED)
          Config.bit_rate = 8000;

        Config.stereo = (IsDlgButtonChecked(window, IDC_STEREO) == BST_CHECKED);

        return TRUE;
      }

      return FALSE;
    }

    ////////////////////////////////////////////////////////////////////////////

    default:
      return FALSE;
  }
}

////////////////////////////////////////////////////////////////////////////////

BOOL CALLBACK InputDialogProc(HWND window, UINT message, WPARAM wparam, LPARAM lparam)
{
  window;
  message;
  wparam;
  lparam;

  switch (message)
  {
    case WM_COMMAND:
      return false;

    default:
      return FALSE;
  }
}

////////////////////////////////////////////////////////////////////////////////

bool IsValidDriver(const char* filename)
{
  HINSTANCE module = LoadLibrary(filename);
  if (module == NULL)
    return false;

  if (GetProcAddress(module, "GetDriverInfo")        == NULL ||
      GetProcAddress(module, "ConfigureDriver")      == NULL ||
      GetProcAddress(module, "InitVideoDriver")      == NULL ||
      GetProcAddress(module, "CloseVideoDriver")     == NULL ||
      GetProcAddress(module, "FlipScreen")           == NULL ||
      GetProcAddress(module, "SetClippingRectangle") == NULL ||
      GetProcAddress(module, "GetClippingRectangle") == NULL ||
      GetProcAddress(module, "ApplyColorMask")       == NULL ||
      GetProcAddress(module, "CreateImage")          == NULL ||
      GetProcAddress(module, "GrabImage")            == NULL ||
      GetProcAddress(module, "DestroyImage")         == NULL ||
      GetProcAddress(module, "BlitImage")            == NULL ||
      GetProcAddress(module, "GetImageWidth")        == NULL ||
      GetProcAddress(module, "GetImageHeight")       == NULL ||
      GetProcAddress(module, "LockImage")            == NULL ||
      GetProcAddress(module, "UnlockImage")          == NULL ||
      GetProcAddress(module, "DirectBlit")           == NULL ||
      GetProcAddress(module, "DirectGrab")           == NULL)
  {
    FreeLibrary(module);
    return false;
  }

  FreeLibrary(module);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

void InitializeDriverList()
{
  DriverList.num_drivers = 0;
  DriverList.drivers     = NULL;
}

////////////////////////////////////////////////////////////////////////////////

bool AddDriver(const char* filename)
{
  // add driver to list
  resize(DriverList.drivers, DriverList.num_drivers, DriverList.num_drivers + 1);
  DriverList.drivers[DriverList.num_drivers] = newstr(filename);
  DriverList.num_drivers++;

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool ConfigureDriver(HWND window, const char* driver)
{
  char old_directory[MAX_PATH];
  GetCurrentDirectory(MAX_PATH, old_directory);
  SetCurrentDirectory(DRIVER_DIRECTORY);

  // load the driver
  HINSTANCE module = LoadLibrary(driver);
  if (driver == NULL)
  {
    SetCurrentDirectory(old_directory);
    return false;
  }

  // get the configuration function
  bool (__stdcall *configure_driver)(HWND parent) = (bool (_stdcall *)(HWND))GetProcAddress(module, "ConfigureDriver");
  if (configure_driver == NULL)
  {
    FreeLibrary(module);
    SetCurrentDirectory(old_directory);
    return false;
  }

  configure_driver(window);

  // clean up and go home
  FreeLibrary(module);
  SetCurrentDirectory(old_directory);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool GetDriverInfo(const char* drivername, DRIVERINFO* driverinfo)
{
  // store old directory
  char old_directory[MAX_PATH];
  GetCurrentDirectory(MAX_PATH, old_directory);
  SetCurrentDirectory(DRIVER_DIRECTORY);

  // load the driver
  HMODULE module = LoadLibrary(drivername);
  if (module == NULL)
  {
    SetCurrentDirectory(old_directory);
    return false;
  }

  // get the GetDriverInfo function
  void (__stdcall* get_driver_info)(DRIVERINFO* driverinfo);
  get_driver_info = (void (__stdcall*)(DRIVERINFO* driverinfo))GetProcAddress(module, "GetDriverInfo");
  if (get_driver_info == NULL)
  {
    FreeLibrary(module);
    SetCurrentDirectory(old_directory);
    return false;
  }

  // we must duplicate the information because once the driver is unloaded all of the memory is gone
  DRIVERINFO di;
  get_driver_info(&di);

  driverinfo->name        = newstr(di.name);
  driverinfo->author      = newstr(di.author);
  driverinfo->date        = newstr(di.date);
  driverinfo->version     = newstr(di.version);
  driverinfo->description = newstr(di.description);

  // clean up and go home
  FreeLibrary(module);
  SetCurrentDirectory(old_directory);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool FreeDriverInfo(DRIVERINFO* driverinfo)
{
  delete[] driverinfo->name;
  delete[] driverinfo->author;
  delete[] driverinfo->date;
  delete[] driverinfo->version;
  delete[] driverinfo->description;
  return true;
}

////////////////////////////////////////////////////////////////////////////////
