#include <string.h>
#include "SphereScript.hpp"
#include "SS_Internal.hpp"
#include "SS_Compiler.hpp"
#include "SS_Assembler.hpp"
#include "MemoryBuffer.hpp"
#include "FileBuffer.hpp"
#include "x++.hpp"


////////////////////////////////////////////////////////////////////////////////

bool sCompileScript(const char* outfile, const char* infile, sCompileError& error, sTiming* timing)
{
  // verify that the extension is .ss on the infile
  if (strcmp_ci(infile + strlen(infile) - 3, ".ss") != 0)
  {
    error.SetMessage("Only .ss files can be compiled");
    return false;
  }

  CFileBuffer out(outfile, false);
  CFileBuffer in(infile);
  CMemoryBuffer assembly;
  
  try
  {
    clock_t begin = clock();
    sCompiler().Execute(&in, &assembly);
    clock_t middle = clock();
    sAssembler().Execute(&assembly, &out);
    clock_t end = clock();

    if (timing)
    {
      timing->compile = middle - begin;
      timing->assemble = end - middle;
    }
  }
  catch (const sScriptException& e)
  {
    error.SetMessage(e.message.c_str());
    error.SetToken(e.token.t.c_str());
    error.SetTokenStart(e.token.location);
    error.SetTokenLine(e.token.line);
    return false;
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool SS_IsKeyword(const char* token)
{
  const char* keywords[] =
  {
    "type",
    "void",
    "int",
    "bool",
    "string",
    "float",
    "true",
    "false",
    "if",
    "else",
    "while",
    "do",
    "until",
    "switch",
    "case",
    "for",
    "return",
    "implement",
    "use",
  };

  for (int i = 0; i < arraysize(keywords); i++)
    if (strcmp(token, keywords[i]) == 0)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool SS_IsOperator(const char* token)
{
  const char* operators[] =
  {
    // not really operators, but they should be highlighted
    ",",
    ";",

    "(",
    ")",
    "!", // unary
    "+", // unary
    "-", // unary
    "=",
    "*=",
    "/=",
    "%=",
    "+=",
    "-=",
    "&",
    "|",
    "==",
    "!=",
    "<",
    ">",
    "<=",
    ">=",
    "*",
    "/",
    "%",

// already in list
//    "+",
//    "-",
  };

  for (int i = 0; i < arraysize(operators); i++)
    if (strcmp(token, operators[i]) == 0)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool SS_IsSystemType(const char* token)
{
  for (int i = 0; i < g_NumSystemTypes; i++)
    if (strcmp(token, g_SystemTypes[i].name) == 0)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool SS_IsSystemFunction(const char* token)
{
  for (int i = 0; i < g_NumSystemFunctions; i++)
    if (strcmp(token, g_SystemFunctions[i].name) == 0)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////
