#include <stdio.h>
#include "SS_Linker.hpp"
#include "SS_Tokenizer.hpp"
#include "SphereScript.hpp"


////////////////////////////////////////////////////////////////////////////////

void
sLibraryHeader::Read(const char* name)
{
  UserTypes.clear();
  Functions.clear();

  char full_name[520];
  sprintf(full_name, "%s.ssl", name);
  FILE* file = fopen(full_name, "rb");
  if (file == NULL)
    throw sScriptException("Could not open library header");

  // load text from the file
  int text_size;
  fseek(file, 0, SEEK_END);
  text_size = ftell(file);
  char* text = new char[text_size + 1];
  fseek(file, 0, SEEK_SET);
  fread(text, 1, text_size, file);
  text[text_size] = 0;

  // build the t
  Parse(text);

  delete[] text;
  fclose(file);
}

////////////////////////////////////////////////////////////////////////////////

int
sLibraryHeader::GetNumUserTypes() const
{
  return UserTypes.size();
}

////////////////////////////////////////////////////////////////////////////////

sLibraryUserType
sLibraryHeader::GetUserType(int i) const
{
  return UserTypes[i];
}

////////////////////////////////////////////////////////////////////////////////

int
sLibraryHeader::GetNumFunctions() const
{
  return Functions.size();
}

////////////////////////////////////////////////////////////////////////////////

sLibraryFunction
sLibraryHeader::GetFunction(int i) const
{
  return Functions[i];
}

////////////////////////////////////////////////////////////////////////////////

void
sLibraryHeader::Parse(const char* text)
{
  sTokenize(Tokens, text);
  CurrentToken = 0;

  cHeader();
}

////////////////////////////////////////////////////////////////////////////////

bool
sLibraryHeader::IsAlpha(char c)
{
  return (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z');
}

////////////////////////////////////////////////////////////////////////////////

bool
sLibraryHeader::IsDigit(char c)
{
  return (c >= '0' && c <= '9');
}

////////////////////////////////////////////////////////////////////////////////

bool
sLibraryHeader::IsIdentifier(sString s)
{
  if (s.length() == 0)
    return false;

  if (IsAlpha(s[0]) == false && s[0] != '_')
    return false;

  for (int i = 1; i < s.length(); i++)
  {
    if (IsAlpha(s[i]) == false &&
        IsDigit(s[i]) == false &&
        s[i] != '_')
      return false;
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sLibraryHeader::IsDataType(sString s)
{
  for (int i = 0; i < UserTypes.size(); i++)
    if (s == UserTypes[i].name)
      return true;
  if (SS_IsSystemType(s.c_str()))
    return true;
  return s == "int" ||
         s == "string" ||
         s == "float" ||
         s == "bool";
}

////////////////////////////////////////////////////////////////////////////////

bool
sLibraryHeader::IsReturnType(sString s)
{
  return s == "void" ||
         IsDataType(s);
}

////////////////////////////////////////////////////////////////////////////////

sString
sLibraryHeader::cIdentifier()
{
  if (!IsIdentifier(Token()))
    throw sScriptException("Not a valid identifier");
  
  sString s = Token();
  Next();
  return s;
}

////////////////////////////////////////////////////////////////////////////////

sString
sLibraryHeader::cDataType()
{
  if (!IsDataType(Token()))
    throw sScriptException("Not a valid data type");

  sString s = Token();
  Next();
  return s;
}

////////////////////////////////////////////////////////////////////////////////

sString
sLibraryHeader::cReturnType()
{
  if (!IsReturnType(Token()))
    throw sScriptException("Not a valid return type");

  sString s = Token();
  Next();
  return s;
}

////////////////////////////////////////////////////////////////////////////////

void
sLibraryHeader::cHeader()
{
  while (Token() != END_TOKENS_MARKER)
  {
    if (Token() == "type")
      cUserType();
    else
      cFunction();
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sLibraryHeader::cUserType()
{
  sLibraryUserType user_type;

  Match("type");
  user_type.name = cIdentifier();
  Match("{");

  while (Token() != "}")
  {
    sString data_type = cDataType();
    do {
      sField field;
      field.type = data_type;
      field.name = cIdentifier();
      user_type.fields.push_back(field);
      if (Token() != ";")
        Match(",");
    } while (Token() != ";");
    Match(";");
  }
  Match("}");

  // validate user type
  for (int i = 0; i < user_type.fields.size(); i++)
    for (int j = i + 1; j < user_type.fields.size(); j++)
    {
      if (user_type.fields[i].name == user_type.fields[j].name)
        throw sScriptException("Duplicate field names in user type '" + user_type.name + "'");
    }

  UserTypes.push_back(user_type);
}

////////////////////////////////////////////////////////////////////////////////

void
sLibraryHeader::cFunction()
{
  sLibraryFunction function;
  function.return_type = cReturnType();
  function.name = cIdentifier();

  Match("(");
  while (Token() != ")")
  {
    sParameter parameter;
    parameter.type = cDataType();
    parameter.name = cIdentifier();
    function.parameters.push_back(parameter);

    if (Token() != ")")
      Match(",");
  }
  Match(")");
  Match(";");

  // validate function
  for (int i = 0; i < function.parameters.size(); i++)
    for (int j = i + 1; j < function.parameters.size(); j++)
    {
      if (function.parameters[i].name == function.parameters[j].name)
        throw sScriptException("Function has duplicate parameters");
    }

  Functions.push_back(function);
}

////////////////////////////////////////////////////////////////////////////////

void
sLibraryHeader::Next()
{
  CurrentToken++;
}

////////////////////////////////////////////////////////////////////////////////

sString
sLibraryHeader::Token()
{
  if (CurrentToken < 0 || CurrentToken >= Tokens.size())
    throw sScriptException("Token outside of library");
  return Tokens[CurrentToken].t;
}

////////////////////////////////////////////////////////////////////////////////

void
sLibraryHeader::Match(sString token)
{
  if (Token() != token)
    throw sScriptException("Expected: " + token); 
  Next(); 
}

////////////////////////////////////////////////////////////////////////////////
