#include <stdio.h>
#include <stdlib.h>
#include "SS_Compiler.hpp"
#include "SS_Tokenizer.hpp"
#include "SS_Linker.hpp"



////////////////////////////////////////////////////////////////////////////////

sCompiler::sCompiler()
{
}

////////////////////////////////////////////////////////////////////////////////

sCompiler::~sCompiler()
{
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::Execute(CBuffer* in, CBuffer* out)
{
  inbuffer = in;
  outbuffer = out;

  // reset all of the variables
  Pass = NO_PASS;
  Tokens.clear();
  CurrentToken = 0;
  CurrentFunction = "";
  SystemFunctions.clear();
  UserTypes.clear();
  ArrayTypes.clear();
  GlobalVariables.clear();
  Functions.clear();
  Parameters.clear();
  LocalVariableLevels.clear();
  StringLiterals.clear();
  RequiredFunctions.clear();
  ExternalFunctions.clear();
  LabelIndex = 0;
  
  InitializeSystemTypes();
  InitializeSystemFunctions();

  Tokenize();
  FirstPass();
  SecondPass();
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::InitializeSystemTypes()
{
  for (int i = 0; i < g_NumSystemTypes; i++)
  {
    sUserType t;
    t.name = g_SystemTypes[i].name;

    // parse field list and add it to type
    char* new_field_list = newstr(g_SystemTypes[i].field_list);

    char* token = strtok(new_field_list, " ,");
    while (token != NULL)
    {
      sVariable field;
      field.data_type = StringToType(token);

      token = strtok(NULL, " ,");
      field.name      = token;

      token = strtok(NULL, " ,");

      // add the field to the user type field list
      t.fields.push_back(field);
    }

    delete[] new_field_list;

    UserTypes.push_back(t);
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::InitializeSystemFunctions()
{
  for (int i = 0; i < g_NumSystemFunctions; i++)
  {
    sFunction f;
    f.name        = g_SystemFunctions[i].name;
    f.return_type = g_SystemFunctions[i].return_type;

    // add parameters to list

    // create modifiable parameter list (so strtok() can mess it up)
    char* new_parameter_list = newstr(g_SystemFunctions[i].parameter_list);
    
    // parse parameter list
    char* token = strtok(new_parameter_list, ",");
    while (token != NULL)
    {
      f.parameters.push_back(StringToType(token));
      token = strtok(NULL, ",");
    }
    
    delete[] new_parameter_list;

    // add function
    SystemFunctions.push_back(f);
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::Tokenize()
{
  // allocate a temporary character buffer for fast access
  char* intext = new char[inbuffer->Size() + 1];
  inbuffer->Read(inbuffer->Size(), (byte*)intext);
  intext[inbuffer->Size()] = 0;

  sTokenize(Tokens, intext);

  // we no longer need the temporary character buffer
  delete[] intext;

  // replace constants in script with values
  for (int i = 0; i < Tokens.size(); i++)
  {
    for (int j = 0; j < g_NumSystemConstants; j++)
      if (Tokens[i].t == g_SystemConstants[j].name)
      {
        Tokens[i].t = g_SystemConstants[j].value;
        break;
      }
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::FirstPass()
{
  Pass = FIRST_PASS;
  cProgram();

  if (RequiredFunctions.size() != 0)
    Error("Script did not implement all required functions");
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::SecondPass()
{
  Pass = SECOND_PASS;
  cProgram();
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsAlpha(char c)
{
  return (c >= 'a' && c <= 'z') ||
         (c >= 'A' && c <= 'Z');
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsNumber(char c)
{
  return c >= '0' && c <= '9';
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsValidIdentifier(sString s)
{
  if (s.length() == 0)
    return false;

  if (IsAlpha(s[0]) == false && s[0] != '_')
    return false;

  for (int i = 1; i < s.length(); i++)
  {
    if (IsAlpha(s[i]) == false &&
        IsNumber(s[i]) == false &&
        s[i] != '_')
      return false;
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsKeyword(sString s)
{
  return SS_IsKeyword(s.c_str());
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsNativeType(sString s)
{
  return s == "int" ||
         s == "bool" ||
         s == "string" ||
         s == "float";
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsUserType(sString s)
{
  for (int i = 0; i < UserTypes.size(); i++)
    if (UserTypes[i].name == s)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsDataType(sString s)
{
  return IsNativeType(s) ||
         IsUserType(s);
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsReturnType(sString s)
{
  return s == "void" ||
         IsDataType(s);
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsGlobalIdentifier(sString s)
{
  return IsSystemFunction(s) ||
         IsUserType(s) ||
         IsGlobalVariable(s) ||
         IsFunction(s);
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsTakenIdentifier(sString s)
{
  return IsGlobalIdentifier(s) ||
         IsParameter(s) ||
         IsLocalVariable(s);
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsFunction(sString s)
{
  for (int i = 0; i < Functions.size(); i++)
    if (s == Functions[i].name)
      return true;
  for (int i = 0; i < ExternalFunctions.size(); i++)
    if (s == ExternalFunctions[i].name)
      return true;
  return IsSystemFunction(s);
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsSystemFunction(sString s)
{
  for (int i = 0; i < SystemFunctions.size(); i++)
    if (s == SystemFunctions[i].name)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsExternalFunction(sString s)
{
  for (int i = 0; i < ExternalFunctions.size(); i++)
    if (s == ExternalFunctions[i].name)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsLValue(sString s)
{
  return IsGlobalVariable(s) ||
         IsParameter(s) ||
         IsLocalVariable(s);
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsGlobalVariable(sString s)
{
  for (int i = 0; i < GlobalVariables.size(); i++)
    if (s == GlobalVariables[i].name)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsParameter(sString s)
{
  for (int i = 0; i < Parameters.size(); i++)
    if (s == Parameters[i].name)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsLocalVariable(sString s)
{
  for (int i = 0; i < LocalVariableLevels.size(); i++)
    for (int j = 0; j < LocalVariableLevels[i].size(); j++)
      if (LocalVariableLevels[i][j].name == s)
        return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsAssignmentOperator(sString s)
{
  return s == "=" ||
         s == "+=" ||
         s == "-=" ||
         s == "*=" ||
         s == "/=" ||
         s == "%=";
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsLogicOperator(sString s)
{
  return s == "&" ||
         s == "|";
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsComparisonOperator(sString s)
{
  return s == "==" ||
         s == "!=" ||
         s == "<"  ||
         s == ">"  ||
         s == "<=" ||
         s == ">=";
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsMultiplicationOperator(sString s)
{
  return s == "*" ||
         s == "/" ||
         s == "%";
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsAdditionOperator(sString s)
{
  return s == "+" ||
         s == "-";
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsBooleanConstant(sString s)
{
  return s == "true" ||
         s == "false";
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsNumericConstant(sString s)
{
  bool decimal = false;
  for (int i = 0; i < s.length(); i++)
  {
    if (s[i] == '.')
    {
      if (decimal)
        return false;
      decimal = true;
    }
    else if (IsNumber(s[i]) == false)
      return false;
  }
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsPositiveInteger(sString s)
{
  for (int i = 0; i < s.length(); i++)
    if (IsNumber(s[i]) == false)
      return false;

  if (atoi(s.c_str()) > 0)
    return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

sString
sCompiler::cDataType()
{
  if (Token() == END_TOKENS_MARKER)
    Error("Expected: data type");

  if (IsDataType(Token()))
    return TokenInc();
  Error("Not a valid data type");
  return "";
}

////////////////////////////////////////////////////////////////////////////////

sString
sCompiler::cReturnType()
{
  if (Token() == END_TOKENS_MARKER)
    Error("Expected: return type");

  if (IsReturnType(Token()))
    return TokenInc();
  Error("Not a valid return type");
  return "";
}

////////////////////////////////////////////////////////////////////////////////

sString
sCompiler::cIdentifier()
{
  // SPECIAL CASE -- make a nice error message
  if (Token() == END_TOKENS_MARKER)
    Error("Expected: identifier");

  if (IsKeyword(Token()))
    Error("Identifier cannot be reserved word");

  if (!IsValidIdentifier(Token()))
    Error("Not a valid identifier");

  return TokenInc();
}

////////////////////////////////////////////////////////////////////////////////

sVariable
sCompiler::cParameter()
{
  sVariable v;
  v.data_type = StringToType(cDataType());
  v.name      = cIdentifier();
  return v;
}

////////////////////////////////////////////////////////////////////////////////

sString
sCompiler::cMangledVariable(ssType& t)
{
  // make sure we're operating on the correct data
  if (!IsLValue(Token()))
    Error("Not an l-value");

  // get the initial variable name
  sString mangled_name = "_" + Token();
  t = TypeOf(TokenInc());

  // access each field
  while (Token() == "." || Token() == "[")
  {
    // user type access
    if (Token() == ".")
    {
      if (!IsUserType(t))
        Error("Variable must be a user type to use . operator");

      Match(".");
      mangled_name += ":";
      mangled_name += itos(OffsetOfField(t, Token()));
      t = TypeOfField(t, Token());
      Next();
    }
    // array access
    else if (Token() == "[")
    {
      if (!IsArrayType(t))
        Error("Variable must be an array to use [] operator");

      Match("[");
      if (cExpression() != TYPE_INT)
        Error("Array indexes must be integers");

      Emit("mov r3 " + mangled_name);
      mangled_name = "r3:r2";
      Emit("mov r2 r0");

      Match("]");
      t = ArrayTypes[t - TYPE_ARRAY].sub_type;
    }
  }

  if (IsArrayType(t))
    Error("Array types are not directly useable");

  return mangled_name;
}

////////////////////////////////////////////////////////////////////////////////

int
sCompiler::cPositiveInteger()
{
  if (IsPositiveInteger(Token()))
    return atoi(TokenInc().c_str());
  Error("Expected: Integer");
  return 0;
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cProgram()
{
  CurrentToken = 0;

  // process library directives
  while (Token() == "use")
    cUseDirective();

  if (Token() == "implement")
    cImplementDirective();

  if (Pass == FIRST_PASS)
  {
    // first pass simply builds symbol tables
    while (Token() != END_TOKENS_MARKER)
      cGlobal();
  }
  else
  {
    // second pass emits all code

    EmitDependancies();
    EmitGlobalVariableDefinitions();

    while (Token() != END_TOKENS_MARKER)
      cGlobal();

    EmitInitializeFunction();
    EmitUninitializeFunction();
    EmitStringLiterals();
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cUseDirective()
{
  Match("use");
  sString name = cIdentifier();

  // only process use directives on first pass
  if (Pass == FIRST_PASS)
  {
    sLibraryHeader header;
    header.Read(name.c_str());

    for (int i = 0; i < header.GetNumUserTypes(); i++)
    {
      if (IsTakenIdentifier(header.GetUserType(i).name))
        throw sScriptException("Duplicate identifier in header");
      sLibraryUserType lut = header.GetUserType(i);
    
      // convert library user type to compiler user type
      sUserType user_type;
      user_type.name = lut.name;

      for (int j = 0; j < lut.fields.size(); j++)
      {
        sVariable field;
        field.name      = lut.fields[j].name;
        field.data_type = StringToType(lut.fields[j].type);
        user_type.fields.push_back(field);
      }

      UserTypes.push_back(user_type);
    }

    for (int i = 0; i < header.GetNumFunctions(); i++)
    {
      if (IsTakenIdentifier(header.GetFunction(i).name))
        throw sScriptException("Duplicate identifier in header");
    
      sLibraryFunction lf = header.GetFunction(i);
    
      sFunction function;
      function.name        = lf.name;
      function.return_type = StringToType(lf.return_type);
    
      for (int j = 0; j < lf.parameters.size(); j++)
      {
        ssType type = StringToType(lf.parameters[j].type);
        function.parameters.push_back(type);
      }

      ExternalFunctions.push_back(function);
    }

    AddDependancy(name);
  }

  Match(";");
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cImplementDirective()
{
  Match("implement");
  sString name = cIdentifier();

  // only process use directives on first pass
  if (Pass == FIRST_PASS)
  {
    sLibraryHeader header;
    header.Read(name.c_str());

    for (int i = 0; i < header.GetNumUserTypes(); i++)
    {
      if (IsTakenIdentifier(header.GetUserType(i).name))
        throw sScriptException("Duplicate identifier in header");
      sLibraryUserType lut = header.GetUserType(i);
    
      // convert library user type to compiler user type
      sUserType user_type;
      user_type.name = lut.name;

      for (int j = 0; j < lut.fields.size(); j++)
      {
        sVariable field;
        field.name      = lut.fields[j].name;
        field.data_type = StringToType(lut.fields[j].type);
        user_type.fields.push_back(field);
      }

      UserTypes.push_back(user_type);
    }

    for (int i = 0; i < header.GetNumFunctions(); i++)
    {
      if (IsTakenIdentifier(header.GetFunction(i).name))
        throw sScriptException("Duplicate identifier in header");
    
      sLibraryFunction lf = header.GetFunction(i);
    
      sFunction function;
      function.name        = lf.name;
      function.return_type = StringToType(lf.return_type);
    
      for (int j = 0; j < lf.parameters.size(); j++)
      {
        ssType type = StringToType(lf.parameters[j].type);
        function.parameters.push_back(type);
      }

      RequiredFunctions.push_back(function);
    }
  }

  Match(";");
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cGlobal()
{
  // check for user type
  if (Token() == "type")
  {
    cUserType();
    return;
  }

  // check for function
  if (CurrentToken < Tokens.size() - 2)
  {
    Next();
    Next();
    if (Token() == "(")  // we have a function
    {
      Prev();
      Prev();
      cFunction();
      return;
    }

    Prev();
    Prev();
  }

  // the only other possibility is a global variable
  cGlobalVariable();
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cUserType()
{
  if (Pass == FIRST_PASS)
  {
    sUserType ut;

    Match("type");
    ut.name = cIdentifier();
    if (IsGlobalIdentifier(ut.name))
    {
      Prev();
      Error("Identifier is already defined");
    }

    Match("{");

    // define the user type
    do
    {
      sVariable v;
      v.data_type = StringToType(cDataType());

      // use a stack to push on the array sizes because [] is right-to-left
      sStack<int> sizes;
      while (Token() == "[")
      {
        Match("[");
        sizes.push(cPositiveInteger());
        Match("]");
      }

      while (sizes.empty() == false)
      {
        sArrayType a;
        a.initial_size = sizes.top();
        sizes.pop();

        a.sub_type = v.data_type;
        v.data_type = AddArrayType(a);
      }

      // handle the identifiers
      do
      {
        v.name = cIdentifier();

        // check if identifier is already used
        for (int i = 0; i < ut.fields.size(); i++)
          if (v.name == ut.fields[i].name)
          {
            Prev();
            Error("Identifier is already used in user type");
          }

        // add field to user type
        ut.fields.push_back(v);

        if (Token() != ";")
          Match(",");
      }
      while (Token() != ";");
      Match(";");

    }
    while (Token() != "}");

    Match("}");

    UserTypes.push_back(ut);
  }
  else
  {
    Match("type");
    cIdentifier();
    Match("{");
    while (Token() != "}")
      Next();
    Match("}");
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cGlobalVariable()
{
  // only build the global variable table on the first pass
  if (Pass == FIRST_PASS)
  {
    sVariable v;
    v.data_type = StringToType(cDataType());

    // use a stack to push on the array sizes because [] is right-to-left
    sStack<int> sizes;
    while (Token() == "[")
    {
      Match("[");
      sizes.push(cPositiveInteger());
      Match("]");
    }

    while (sizes.empty() == false)
    {
      sArrayType a;
      a.initial_size = sizes.top();
      sizes.pop();

      a.sub_type = v.data_type;
      v.data_type = AddArrayType(a);
    }

    do
    {
      v.name      = cIdentifier();

      if (IsGlobalIdentifier(v.name))
      {
        Prev();
        Error("Identifier is already defined");
      }

      // go on to new variable
      if (Token() != ";")
        Match(",");

      GlobalVariables.push_back(v);
    }
    while (Token() != ";");
    Match(";");

  }
  else
  {
    cDataType();
    while (Token() == "[")
    {
      Match("[");
      cPositiveInteger();
      Match("]");
    }

    while (Token() != ";")
      Next();
    Match(";");
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cFunction()
{
  if (Pass == FIRST_PASS)
  {
    // add function declaration to function list
    sFunction function;
    function.return_type = StringToType(cReturnType());
    function.name        = cIdentifier();

    if (IsGlobalIdentifier(function.name))
    {
      Prev();
      Error("Identifier is already defined");
    }

    // store the data types in the parameter list
    Match("(");
    while (Token() != ")")
    {
      ssType type = StringToType(cDataType());
      cIdentifier();
      if (Token() != ")")
        Match(",");
      function.parameters.push_back(type);
    }
    Match(")");

    // see if we have a required function
    for (int i = 0; i < RequiredFunctions.size(); i++)
      if (RequiredFunctions[i].name == function.name)
      {
        // check the function to see if it fits the prototype
        if (RequiredFunctions[i] != function)
          Error("Function is not same as prototype in header");

        RequiredFunctions.remove(i);
      }

    cBlock();

    Functions.push_back(function);
  }
  else  // SECOND PASS
  {
    cReturnType();
    sString name = cIdentifier();

    // parameter list
    Match("(");
    while (Token() != ")")
    {
      sVariable v;
      v.data_type = StringToType(cDataType());
      v.name      = cIdentifier();

      if (IsTakenIdentifier(v.name))
      {
        Prev();
        Error("Identifier is already in use");
      }

      if (Token() != ")")
        Match(",");
      Parameters.push_back(v);
    }
    Match(")");

    // remember, we are currently compiling this function
    CurrentFunction = name;
    Emit("function _" + name);
    for (int i = 0; i < Parameters.size(); i++)
      Emit("parameter _" + Parameters[i].name);

    // generate the function code
    cBlock();

    ConstructVariable("r0", GetFunction(CurrentFunction).return_type);

    Emit("endfunction");

    // make sure the next function doesn't use these parameters
    Parameters.clear();
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cBlock()
{
  if (Pass == FIRST_PASS)
  {
    // on the first pass, just skip all blocks
    Match("{");
    int levels = 1;
    while (levels > 0)
    {
      if (Token() == "{")
        levels++;
      if (Token() == "}")
        levels--;
      Next();
    }
  }
  else
  {
    BeginScopeLevel();

    // actually compile code
    Match("{");
    while (Token() != "}")
      cStatement();
    Match("}");

    EndScopeLevel();
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cStatement()
{
  if (Token() == ";")            // null statement
    Match(";");
  else if (Token() == "{")       // compound statement
    cBlock();
  else if (IsDataType(Token()))  // declare local variable
    cLocalVariable();
  else if (Token() == "return")  // return
    cReturn();
  else if (Token() == "if")      // if
    cIf();
  else if (Token() == "for")     // for
    cFor();
  else if (Token() == "while")   // while
    cWhile();
  else if (Token() == "do")      // do
    cDo();
  else if (Token() == "switch")  // switch
    cSwitch();
  else                           // expression
  {
    // emit expression code, then clean up result
    DestructVariable("r0", cExpression());
    Match(";");
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cLocalVariable()
{
  // add local variable to current scope level
  sVariable v;
  v.data_type = StringToType(cDataType());

  // use a stack to push on the array sizes because [] is right-to-left
  sStack<int> sizes;
  while (Token() == "[")
  {
    Match("[");
    sizes.push(cPositiveInteger());
    Match("]");
  }

  while (sizes.empty() == false)
  {
    sArrayType a;
    a.initial_size = sizes.top();
    sizes.pop();

    a.sub_type = v.data_type;
    v.data_type = AddArrayType(a);
  }

  do
  {
    v.name      = cIdentifier();

    if (IsTakenIdentifier(v.name))
    {
      Prev();
      Error("Identifier already in use");
    }

    // put the local variable in this scope level
    LocalVariableLevels[LocalVariableLevels.size() - 1].push_back(v);

    // allocate the variable
    Emit("lc _" + v.name);
    ConstructVariable("_" + v.name, v.data_type);

    // handle initialization expression
    if (Token() == "=")
    {
      Match("=");
      MatchTypes(v.data_type, cExpression());
      MoveVariable(v.data_type, "_" + v.name, "r0");
      DestructVariable("r0", v.data_type);
    }

    // go on to new variable
    if (Token() != ";")
      Match(",");
  }
  while (Token() != ";");

  Match(";");
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cReturn()
{
  Match("return");
  if (GetFunction(CurrentFunction).return_type == TYPE_VOID)
    Match(";");
  else
  {
    if (GetFunction(CurrentFunction).return_type != cExpression())
      Error("Incompatible data types in expression");
    Match(";");
  }

  // destruct all locals
  for (int i = LocalVariableLevels.size() - 1; i >= 0; i--)
    for (int j = 0; j < LocalVariableLevels[i].size(); j++)
    {
      DestructVariable("_" + LocalVariableLevels[i][j].name, LocalVariableLevels[i][j].data_type);
    }

  Emit("ret");
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cIf()
{
  sString label1 = NewLabel();
  sString label2 = NewLabel();

  Match("if");
  Match("(");
  if (cExpression() != TYPE_BOOL)
    Error("Test expression must be of type bool");
  Match(")");

  Emit("not r0");
  Emit("jmpt r0 " + label2);

  BeginScopeLevel();
  cStatement();
  EndScopeLevel();

  Emit("jmp " + label1);

  PostLabel(label2);

  if (Token() == "else")
  {
    Match("else");

    BeginScopeLevel();
    cStatement();
    EndScopeLevel();
  }

  PostLabel(label1);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cFor()
{
  sString L1 = NewLabel();
  sString L2 = NewLabel();
  sString L3 = NewLabel();
  sString L4 = NewLabel();

  Match("for");
  Match("(");
        
  // initialize expression
  cExpression();
  Match(";");

  // test expression
  PostLabel(L1);
  if (cExpression() != TYPE_BOOL)
    Error("Test expression must be of type bool");
  Match(";");

  Emit("not r0");
  Emit("jmpt r0 " + L3);
  Emit("jmp " + L2);

  // update expression
  PostLabel(L4);
  cExpression();
  Match(")");
  Emit("jmp " + L1);

  // statement
  PostLabel(L2);

  BeginScopeLevel();
  cStatement();
  EndScopeLevel();

  Emit("jmp " + L4);

  // done
  PostLabel(L3);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cWhile()
{
  sString label1 = NewLabel();
  sString label2 = NewLabel();

  PostLabel(label1);

  Match("while");
  Match("(");
  if (cExpression() != TYPE_BOOL)
    Error("Test expression must be of type bool");
  Match(")");

  Emit("jmpf r0 " + label2);

  BeginScopeLevel();
  cStatement();
  EndScopeLevel();

  Emit("jmp " + label1);
  PostLabel(label2);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cDo()
{
  sString label = NewLabel();

  Match("do");
  PostLabel(label);

  BeginScopeLevel();
  cStatement();
  EndScopeLevel();

  bool not = false;
  if (Token() == "while")
    Match("while");
  else if (Token() == "until")
  {
    Match("until");
    not = true;
  }
  else
    Error("Expected: while or until");

  Match("(");
  if (cExpression() != TYPE_BOOL)
    Error("Test expression must be of type bool");  
  if (not)
    Emit("not r0");
  Emit("jmpt r0 " + label);

  Match(")");
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cSwitch()
{
  Match("switch");
  Match("(");
  ssType t = cExpression();
  if (t != TYPE_INT && t != TYPE_FLOAT && t != TYPE_STRING)
    Error("Switch expressions must be int, float, or string");
  Match(")");

  // this works with strings too!
  Emit("mov r2 r0");

  Match("{");

  sString endlabel = NewLabel();
  sString label    = NewLabel();

  while (Token() != "}")
  {
    Match("case");
    Match("(");
    if (cExpression() != t)
      Error("Case expression must be of same type as switch expression");
    Match(")");

    if (t == TYPE_INT)  
      Emit("cmpe r0 r2");
    else if (t == TYPE_STRING)
      Emit("scmpe r0 r2");
    else if (t == TYPE_FLOAT)
      Emit("fcmpe r0 r2");
    else
      Error("Invalid case type");

    Emit("not r0");
    Emit("jmpt r0 " + label);

    BeginScopeLevel();
    cStatement();
    EndScopeLevel();

    Emit("jmp " + endlabel);

    PostLabel(label);
    label = NewLabel();
  }

  PostLabel(endlabel);

  if (Token() == "other")
  {
    Match("other");
    Match(":");
    cStatement();
  }
      
  Match("}");

  DestructVariable("r2", t);
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::cExpression()
{
  int old_token = CurrentToken;

  // skip past the variable
  while (Token() == "." || IsValidIdentifier(Token()) || Token() == "[")
  {
    if (Token() == "[")
    {
      Match("[");
      int num_brackets = 1;
      while (num_brackets > 0)
      {
        if (Token() == "[")
          num_brackets++;
        else if (Token() == "]")
          num_brackets--;
        Next();
      }
    }
    else
      Next();
  }

  // test if it's an assignment operation
  if (IsAssignmentOperator(Token()))
  {
    CurrentToken = old_token;
    return cAssignment();
  }
  else
  {
    CurrentToken = old_token;
    return cLogicExpression();
  }
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::cAssignment()
{
  ssType var_type;
  sString dest = cMangledVariable(var_type);
  sString op = TokenInc();

  // store registers created in cMangledVariable
  Emit("push r2");
  Emit("push r3");

  ssType t = cExpression();
  MatchTypes(var_type, t);

  // r0 should be the only important register after cExpression

  Emit("pop r3");
  Emit("pop r2");

  if (t == TYPE_INT)
  {
    if      (op == "=")  Emit("mov " + dest + " r0");
    else if (op == "+=") Emit("add " + dest + " r0");
    else if (op == "-=") Emit("sub " + dest + " r0");
    else if (op == "*=") Emit("mul " + dest + " r0");
    else if (op == "/=") Emit("div " + dest + " r0");
    else if (op == "%=") Emit("mod " + dest + " r0");
    else Error("Unknown operator for data type");
  }
  else if (t == TYPE_STRING)
  {
    if      (op == "=")  Emit("smov " + dest + " r0");
    else if (op == "+=") Emit("sadd " + dest + " r0");
    else Error("Unknown operator for data type");
  }
  else if (t == TYPE_BOOL)
  {
    if (op == "=") Emit("mov " + dest + " r0");
    else Error("Unknown operator for data type");
  }
  else if (t == TYPE_FLOAT)
  {
    if      (op == "=")  Emit("mov "  + dest + " r0");
    else if (op == "+=") Emit("fadd " + dest + " r0");
    else if (op == "-=") Emit("fsub " + dest + " r0");
    else if (op == "*=") Emit("fmul " + dest + " r0");
    else if (op == "/=") Emit("fdiv " + dest + " r0");
    else Error("Unknown operator for data type");
  }
  else
  {
    MoveVariable(t, dest, "r0");
  }

  return t;
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::cLogicExpression()
{
  ssType t = cComparisonExpression();
  while (IsLogicOperator(Token()))
  {
    Emit("push r0");
    cLogic(t);
    t = TYPE_BOOL;
  }
  return t;
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cLogic(ssType t)
{
  sString op = TokenInc();
  if (t != TYPE_BOOL ||
      cComparisonExpression() != TYPE_BOOL)
    Error("Can only perform logic operations on boolean expressions");
  
  Emit("pop r1");
  if (op == "|")
    Emit("or r0 r1");
  else if (op == "&")
    Emit("and r0 r1");
  else
    Error("Unknown logic operator");
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::cComparisonExpression()
{
  ssType t = cAdditionExpression();
  while (IsComparisonOperator(Token()))
  {
    Emit("push r0");
    cCompare(t);
    t = TYPE_BOOL;
  }
  return t;
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cCompare(ssType t)
{
  sString op = TokenInc();
  MatchTypes(t, cAdditionExpression());
  
  Emit("pop r1");
  if (t == TYPE_INT)
  {
    if      (op == "==") { Emit("cmpe r0 r1"); }
    else if (op == "!=") { Emit("cmpe r0 r1"); Emit("not r0"); }
    else if (op == ">" ) { Emit("cmpg r1 r0"); Emit("mov r0 r1"); }
    else if (op == "<" ) { Emit("cmpl r1 r0"); Emit("mov r0 r1"); }
    else if (op == ">=") { Emit("cmpl r0 r1"); }
    else if (op == "<=") { Emit("cmpg r0 r1"); }
    else Error("Unknown operator for this type");
  }
  else if (t == TYPE_STRING)
  {
    if      (op == "==") { Emit("scmpe r1 r0"); Emit("sfree r0"); Emit("mov r0 r1"); }
    else if (op == "!=") { Emit("scmpe r1 r0"); Emit("sfree r0"); Emit("mov r0 r1"); Emit("not r0"); }
    else if (op == ">" ) { Emit("scmpg r1 r0"); Emit("sfree r0"); Emit("mov r0 r1"); }
    else if (op == "<" ) { Emit("scmpl r1 r0"); Emit("sfree r0"); Emit("mov r0 r1"); }
    else if (op == ">=") { Emit("scmpl r0 r1"); Emit("sfree r1"); }
    else if (op == "<=") { Emit("scmpg r0 r1"); Emit("sfree r1"); }
    else Error("Unknown operator for this type");
  }
  else if (t == TYPE_FLOAT)
  {
    if      (op == "==") { Emit("fcmpe r0 r1"); }
    else if (op == "!=") { Emit("fcmpe r0 r1"); Emit("not r0"); }
    else if (op == ">" ) { Emit("fcmpg r1 r0"); Emit("mov r0 r1"); }
    else if (op == "<" ) { Emit("fcmpl r1 r0"); Emit("mov r0 r1"); }
    else if (op == ">=") { Emit("fcmpl r0 r1"); }
    else if (op == "<=") { Emit("fcmpg r0 r1"); }
    else Error("Unknown operator for this type");
  }
  else if (t == TYPE_BOOL)
  {
    if      (op == "==") { Emit("cmpe r0 r1"); }
    else if (op == "!=") { Emit("cmpe r0 r1"); Emit("not r0"); }
    else Error("Unknown operator for this type");
  }
  else
    Error("Cannot compare data type");
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::cAdditionExpression()
{
  ssType t = cMultiplicationExpression();
  while (IsAdditionOperator(Token()))
  {
    Emit("push r0");
    cAdd(t);
  }
  return t;
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cAdd(ssType t)
{
  sString op = TokenInc();
  MatchTypes(t, cMultiplicationExpression());

  Emit("pop r1");
  if (t == TYPE_INT)
  {
    if      (op == "+") { Emit("add r0 r1"); }
    else if (op == "-") { Emit("sub r0 r1"); Emit("neg r0"); }
    else Error("Unknown operator for this type");
  }
  else if (t == TYPE_STRING)
  {
    if (op == "+") { Emit("sadd r1 r0"); Emit("smov r0 r1"); }
    else Error("Unknown operator for this type");
    Emit("sfree r1");
  }
  else if (t == TYPE_FLOAT)
  {
    if      (op == "+") { Emit("fadd r0 r1"); }
    else if (op == "-") { Emit("fsub r0 r1"); Emit("fneg r0"); }
    else Error("Unknown operator for this type");
  }
  else
    Error("Cannot add data type");
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::cMultiplicationExpression()
{
  ssType t = cTerm();
  while (IsMultiplicationOperator(Token()))
  {
    Emit("push r0");
    cMultiply(t);
  }
  return t;
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::cMultiply(ssType t)
{
  sString op = TokenInc();
  MatchTypes(t, cTerm());

  Emit("pop r1");
  if (t == TYPE_INT)
  {
    if      (op == "*") { Emit("mul r0 r1"); }
    else if (op == "/") { Emit("div r1 r0"); Emit("mov r0 r1"); }
    else if (op == "%") { Emit("mod r1 r0"); Emit("mov r0 r1"); }
    else Error("Unknown operator for this type");
  }
  else if (t == TYPE_FLOAT)
  {
    if      (op == "*") { Emit("fmul r0 r1"); }
    else if (op == "/") { Emit("fdiv r1 r0"); Emit("mov r0 r1"); }
    else Error("Unknown operator for this type");
  }
  else
    Error("Cannot multiply data type");
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::cTerm()
{
  // OPERATORS

  // parentheses
  if (Token() == "(")
  {
    Match("(");
    ssType t = cExpression();
    Match(")");
    return t;
  }

  // unary not
  if (Token() == "!")
  {
    Match("!");
    ssType t = cTerm();
    if (t != TYPE_BOOL)
      Error("Not operator (!) can only be used on boolean expressions");
    Emit("not r0");
    return t;
  }

  // unary plus
  if (Token() == "+")
  {
    Match("+");
    return cTerm();
  }

  // unary minus
  if (Token() == "-")
  {
    Match("-");
    ssType t = cTerm();
    if (t != TYPE_INT && t != TYPE_FLOAT)
      Error("Unary minus operator (-) can only be used on int and float expressions");
    if (t == TYPE_INT)   Emit("neg r0");
    if (t == TYPE_FLOAT) Emit("fneg r0");
    return t;
  }

  
  // VALUES

  // string
  if (Token()[0] == '"')
  {
    return cStringLiteral();
  }

  // function call
  if (IsFunction(Token()))
  {
    sFunction& f = GetFunction(Token());
    Next();  // skip function name
    Match("(");

    // push on the parameters
    if (f.parameters.size() > 0)
    {
      MatchTypes(f.parameters[0], cExpression());
      Emit("push r0");
    }

    for (int i = 1; i < f.parameters.size(); i++)
    {
      Match(",");
      MatchTypes(f.parameters[i], cExpression());
      Emit("push r0");
    }

    // call the function
    Match(")");
    
    if (IsSystemFunction(f.name))
      Emit("syscall _" + f.name);
    else if (IsExternalFunction(f.name))
    {
      sStringLiteral sl;
      sl.label = "f_" + itos(StringLiterals.size());
      sl.value = "\"_" + f.name + '"';
      StringLiterals.push_back(sl);

      Emit("extcall " + sl.label);
    }
    else
      Emit("call _" + f.name);

    // pop off the parameters and destroy them
    for (int i = f.parameters.size() - 1; i >= 0; i--)
    {
      Emit("pop r1");
      DestructVariable("r1", f.parameters[i]);
    }

    return f.return_type;
  }

  // variable
  if (IsLValue(Token()))
  {
    ssType t;
    sString name = cMangledVariable(t);
    ConstructVariable("r0", t);
    MoveVariable(t, "r0", name);
    return t;
  }

  // boolean constant
  if (IsBooleanConstant(Token()))
    return cBooleanConstant();

  // numeric constant
  if (IsNumericConstant(Token()))
    return cNumericConstant();

  Error("Unknown symbol");
  return TYPE_VOID;
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::cStringLiteral()
{
  sStringLiteral sl;
  sl.label = "s_" + itos(StringLiterals.size());
  sl.value = TokenInc();
  StringLiterals.push_back(sl);

  sString command = "sload r0 " + sl.label;
  Emit(command);

  return TYPE_STRING;
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::cBooleanConstant()
{
  sString c = TokenInc();
  if (c == "true")
    Emit("mov r0 1");
  else if (c == "false")
    Emit("mov r0 0");
  else
    Error("Not a boolean constant");
  return TYPE_BOOL;
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::cNumericConstant()
{
  sString s = TokenInc();
  
  char command[1024];
  sprintf(command, "mov r0 %s", s.c_str());
  Emit(command);

  int num_decimals = 0;
  for (int i = 0; i < s.length(); i++)
    if (s[i] == '.')
      num_decimals++;

  return (num_decimals == 0 ? TYPE_INT : TYPE_FLOAT);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::Error(sString message)
{
  if (Pass == FIRST_PASS)
    message += " -- Pass 1";
  else if (Pass == SECOND_PASS)
    message += " -- Pass 2";

  if (CurrentToken < (int)Tokens.size())
    throw sScriptException(message, Tokens[CurrentToken]);
  else
    throw sScriptException(message);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::Emit(sString s)
{
  outbuffer->Write(s.length(), (byte*)s.c_str());
  outbuffer->WriteByte('\r');
  outbuffer->WriteByte('\n');
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::EmitGlobalVariableDefinitions()
{
  for (int i = 0; i < GlobalVariables.size(); i++)
    Emit("global _" + GlobalVariables[i].name);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::AddDependancy(sString name)
{
  Dependancies.push_back(name);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::EmitDependancies()
{
  for (int i = 0; i < Dependancies.size(); i++)
    Emit("depend " + Dependancies[i]);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::EmitInitializeFunction()
{
  Emit("function initialize");
  for (int i = 0; i < GlobalVariables.size(); i++)
    ConstructVariable("_" + GlobalVariables[i].name, GlobalVariables[i].data_type);
  Emit("ret");
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::EmitUninitializeFunction()
{
  Emit("function uninitialize");
  for (int i = 0; i < GlobalVariables.size(); i++)
    DestructVariable("_" + GlobalVariables[i].name, GlobalVariables[i].data_type);
  Emit("ret");
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::EmitStringLiterals()
{
  Emit("string empty \"\"");
  for (int i = 0; i < StringLiterals.size(); i++)
    Emit("string " + StringLiterals[i].label + " " + StringLiterals[i].value);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::Next()
{
  CurrentToken++;
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::Prev()
{
  CurrentToken--;
}

////////////////////////////////////////////////////////////////////////////////

sString
sCompiler::Token()
{
  if (CurrentToken < 0)
    throw sScriptException("Went before beginning of file");
  if (CurrentToken > Tokens.size() - 1)
    throw sScriptException("Went past end of file");
  return Tokens[CurrentToken].t;
}

////////////////////////////////////////////////////////////////////////////////

sString
sCompiler::TokenInc()
{
  sString s = Token();
  Next();
  return s;
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::Match(sString s)
{
  if (Token() != s)
    Error("Expected: '" + s + "'");
  Next();
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::MatchTypes(ssType t1, ssType t2)
{
  if (t1 != t2)
    Error("Incompatible types in expression");
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::StringToType(sString s)
{
  // native types
  if (s == "void")   return TYPE_VOID;
  if (s == "int")    return TYPE_INT;
  if (s == "bool")   return TYPE_BOOL;
  if (s == "string") return TYPE_STRING;
  if (s == "float")  return TYPE_FLOAT;

  // user types
  for (int i = 0; i < UserTypes.size(); i++)
    if (s == UserTypes[i].name)
      return TYPE_USER + i;

  Error("Unknown Type");
  return 0;             // must return something or VC++ issues a warning
}

////////////////////////////////////////////////////////////////////////////////

int
sCompiler::SizeOf(ssType type)
{
  if (IsArrayType(type))
    return ArrayTypes[type - TYPE_ARRAY].initial_size;

  if (IsUserType(type))
    return UserTypes[type - TYPE_USER].fields.size();

  switch (type)
  {
    case TYPE_VOID:   return 0;
    case TYPE_INT:    return 1;
    case TYPE_BOOL:   return 1;
    case TYPE_STRING: return 1;
    case TYPE_FLOAT:  return 1;
    default:          Error("Unknown data size"); return -1;
  }
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::TypeOf(sString s)
{
  // globals
  for (int i = 0; i < GlobalVariables.size(); i++)
    if (GlobalVariables[i].name == s)
      return GlobalVariables[i].data_type;

  // parameters
  for (int i = 0; i < Parameters.size(); i++)
    if (Parameters[i].name == s)
      return Parameters[i].data_type;

  // locals
  for (int i = 0; i < LocalVariableLevels.size(); i++)
    for (int j = 0; j < LocalVariableLevels[i].size(); j++)
      if (LocalVariableLevels[i][j].name == s)
        return LocalVariableLevels[i][j].data_type;

  Error("Could not take type of identifier");
  return TYPE_NONE;     // must return something or VC++ issues a warning
}

////////////////////////////////////////////////////////////////////////////////

sFunction&
sCompiler::GetFunction(sString name)
{
  for (int i = 0; i < SystemFunctions.size(); i++)
    if (SystemFunctions[i].name == name)
      return SystemFunctions[i];
  for (int i = 0; i < ExternalFunctions.size(); i++)
    if (ExternalFunctions[i].name == name)
      return ExternalFunctions[i];
  for (int i = 0; i < Functions.size(); i++)
    if (Functions[i].name == name)
      return Functions[i];
  Error("Unknown function");
  return Functions[0];  // must return something or VC++ issues a warning
}

////////////////////////////////////////////////////////////////////////////////

sString
sCompiler::NewLabel()
{
  return sString(".label_") + itos(LabelIndex++);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::PostLabel(sString label)
{
  Emit("label " + label);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::BeginScopeLevel()
{
  // add new scope level
  sVector<sVariable> new_scope_level;
  LocalVariableLevels.push_back(new_scope_level);
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::EndScopeLevel()
{
  // destroy variables scope level
  sVector<sVariable>& scope_level = LocalVariableLevels[LocalVariableLevels.size() - 1];
  for (int i = scope_level.size() - 1; i >= 0; i--)
  {
    DestructVariable("_" + scope_level[i].name, scope_level[i].data_type);
    Emit("ld _" + scope_level[i].name);
  }

  // remove scope level
  LocalVariableLevels.pop_back();
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::ConstructVariable(sString name, ssType type)
{
  if (IsArrayType(type))
  {
    sArrayType& at = ArrayTypes[type - TYPE_ARRAY];

    // allocate the array
    Emit("alloc " + name + " " + itos(at.initial_size));

    // construct the elements
    for (int i = 0; i < at.initial_size; i++)
      ConstructVariable(name + ":" + itos(i), at.sub_type);

    return;
  }

  if (IsUserType(type))
  {
    sUserType& ut = UserTypes[type - TYPE_USER];
    
    // construct the type
    Emit("alloc " + name + " " + itos(ut.fields.size()));

    // construct the fields
    for (int i = 0; i < ut.fields.size(); i++)
    {
      sString offset_string = ":" + itos(i);
      ConstructVariable(name + offset_string, ut.fields[i].data_type);
    }
    return;
  }

  switch (type)
  {
    case TYPE_VOID:   return;
    case TYPE_INT:    return;
    case TYPE_BOOL:   return;
    case TYPE_STRING: Emit("sload " + name + " empty"); return;
    case TYPE_FLOAT:  return;
    default: Error("Unknown data type - Cannot construct");
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::DestructVariable(sString name, ssType type)
{
  if (IsArrayType(type))
  {
    sArrayType& at = ArrayTypes[type - TYPE_ARRAY];

    for (int i = 0; i < at.initial_size; i++)
      DestructVariable(name + ":" + itos(i), at.sub_type);

    Emit("free " + name);
    return;
  }

  if (IsUserType(type))
  {
    // free the fields
    sUserType& ut = UserTypes[type - TYPE_USER];
    for (int i = 0; i < ut.fields.size(); i++)
    {
      sString offset_string = ":" + itos(i);
      DestructVariable(name + offset_string, ut.fields[i].data_type);
    }

    // free the variable
    Emit("free " + name);
    return;
  }

  switch (type)
  {
    case TYPE_VOID:   return;
    case TYPE_INT:    return;
    case TYPE_BOOL:   return;
    case TYPE_STRING: Emit("sfree " + name); return;
    case TYPE_FLOAT:  return;
    default: Error("Unknown data type -- Cannot destruct");
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sCompiler::MoveVariable(ssType type, sString dest, sString src)
{
  if (IsArrayType(type))
  {
    sArrayType& at = ArrayTypes[type - TYPE_ARRAY];

    for (int i = 0; i < at.initial_size; i++)
      MoveVariable(at.sub_type, dest + ":" + itos(i), src + ":" + itos(i));

    return;
  }

  if (IsUserType(type))
  {
    sUserType& ut = UserTypes[type - TYPE_USER];
    for (int i = 0; i < ut.fields.size(); i++)
    {
      sString offset_string = ":" + itos(i);
      MoveVariable(ut.fields[i].data_type, dest + offset_string, src + offset_string);
    }
    return;
  }

  switch (type)
  {
    case TYPE_VOID:  Error("Can't move void data type"); break;
    case TYPE_BOOL:  Emit("mov " + dest + " " + src);    break;
    case TYPE_INT:   Emit("mov " + dest + " " + src);    break;
    case TYPE_FLOAT: Emit("mov " + dest + " " + src);    break;
    case TYPE_STRING: Emit("smov " + dest + " " + src);  break;
    default: Error("Unknown data type - Cannot move");
  }
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsUserType(ssType t)
{
  return (t >= TYPE_USER && t < TYPE_ARRAY);
}

////////////////////////////////////////////////////////////////////////////////

int
sCompiler::OffsetOfField(ssType t, sString field)
{
  if (!IsUserType(t))
    Error("Can't get offset of non user type");

  sUserType& ut = UserTypes[t - TYPE_USER];
  for (int i = 0; i < ut.fields.size(); i++)
  {
    if (ut.fields[i].name == field)
      return i;
  }

  Error("Field is not in type");
  return 0;
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::TypeOfField(ssType t, sString field)
{
  if (!IsUserType(t))
    Error("Can't get type of non user type");

  sUserType& ut = UserTypes[t - TYPE_USER];
  for (int i = 0; i < ut.fields.size(); i++)
  {
    if (ut.fields[i].name == field)
      return ut.fields[i].data_type;
  }

  Error("Field is not in type");
  return TYPE_NONE;     // must return something or VC++ issues a warning
}

////////////////////////////////////////////////////////////////////////////////

bool
sCompiler::IsArrayType(ssType t)
{
  return t >= TYPE_ARRAY;
}

////////////////////////////////////////////////////////////////////////////////

ssType
sCompiler::AddArrayType(sArrayType a)
{
  ArrayTypes.push_back(a);
  return (TYPE_ARRAY + ArrayTypes.size() - 1);
}

////////////////////////////////////////////////////////////////////////////////

sString
sCompiler::itos(int i)
{
  char str[80];
  sprintf(str, "%d", i);
  return str;
}

////////////////////////////////////////////////////////////////////////////////
