#include <string.h>
#include "WindowStyle.hpp"
#include "packed.h"


////////////////////////////////////////////////////////////////////////////////

sWindowStyle::sWindowStyle()
{
}

////////////////////////////////////////////////////////////////////////////////

sWindowStyle::~sWindowStyle()
{
}

////////////////////////////////////////////////////////////////////////////////

void
sWindowStyle::Create(int width, int height)
{
  for (int i = 0; i < 9; i++)
  {
    m_Bitmaps[i].Resize(width, height);
    memset(m_Bitmaps[i].GetPixels(), 0, width * height * sizeof(RGBA));
  }
}

////////////////////////////////////////////////////////////////////////////////

PACKED_STRUCT(WINDOWSTYLE_HEADER)
  byte signature[4];
  word version;
  byte edge_width;  // only valid if version == 1
  byte reserved[57];
END_STRUCT(WINDOWSTYLE_HEADER)

ASSERT_STRUCT_SIZE(WINDOWSTYLE_HEADER, 64)

////////////////////////////////////////////////////////////////////////////////

bool
sWindowStyle::Load(const char* filename)
{
  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;
  
  // read the header
  WINDOWSTYLE_HEADER header;
  fread(&header, 1, sizeof(header), file);

  // check the header
  if (memcmp(header.signature, ".rws", 4) != 0 ||
      (header.version != 1 && header.version != 2))
  {
    fclose(file);
    return false;
  }

  if (header.version == 1)
  {
    // free the old bitmaps and read the new ones
    for (int i = 0; i < 9; i++)
      ReadBitmap(file, m_Bitmaps + i, header.edge_width);
  }
  else
  {
    // free the old bitmaps and read the new ones
    for (int i = 0; i < 9; i++)
      ReadBitmap(file, m_Bitmaps + i);
  }

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sWindowStyle::Save(const char* filename) const
{
  FILE* file = fopen(filename, "wb");
  if (file == NULL)
    return false;

  // write header
  WINDOWSTYLE_HEADER header;
  memset(&header, 0, sizeof(header));
  memcpy(header.signature, ".rws", 4);
  header.version = 2;
  fwrite(&header, 1, sizeof(header), file);

  // write the bitmaps
  for (int i = 0; i < 9; i++)
    WriteBitmap(file, m_Bitmaps + i);

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sWindowStyle::Import(const char* filename, RGBA transColor)
{
  CImage32 image;
  int winWidth, winHeight;
  int i,j;
  RGBA transparent;

  if (!image.Load(filename))
    return false;

  if (image.GetWidth()%3 || image.GetHeight()%3)
    return false;

  winWidth = image.GetWidth() / 3;
  winHeight = image.GetHeight() / 3;
  for (i=0; i<9; i++)
    m_Bitmaps[i].Resize(winWidth, winHeight);

  // now to check for transparent color
  transparent = transColor;
  transparent.alpha = 0;
  for (j=0; j<image.GetHeight(); j++)
    for (i=0; i<image.GetWidth(); i++)
      if (image.GetPixels()[(j*image.GetWidth()) + i].green == transColor.green)
        if (image.GetPixels()[(j*image.GetWidth()) + i].blue == transColor.blue)
          if (image.GetPixels()[(j*image.GetWidth()) + i].red == transColor.red)
            if (image.GetPixels()[(j*image.GetWidth()) + i].alpha == transColor.alpha)
              image.GetPixels()[j*image.GetWidth() + i] = transparent;

#define CopyEdge(offX, startY, endY, window)                 \
{                                                            \
  for (i=startY; i<endY; i++)                                \
    memcpy(window.GetPixels() + (i-startY)*winWidth,         \
           image.GetPixels() + (i*image.GetWidth()) + offX,  \
           winWidth*sizeof(RGBA));                           \
}
               
  CopyEdge(0,          0,           winHeight,   m_Bitmaps[0]);
  CopyEdge(winWidth,   0,           winHeight,   m_Bitmaps[1]);
  CopyEdge(winWidth*2, 0,           winHeight,   m_Bitmaps[2]);
  CopyEdge(0,          winHeight,   winHeight*2, m_Bitmaps[7]);
  CopyEdge(winWidth,   winHeight,   winHeight*2, m_Bitmaps[8]);
  CopyEdge(winWidth*2, winHeight,   winHeight*2, m_Bitmaps[3]);
  CopyEdge(0,          winHeight*2, winHeight*3, m_Bitmaps[6]);
  CopyEdge(winWidth,   winHeight*2, winHeight*3, m_Bitmaps[5]);
  CopyEdge(winWidth*2, winHeight*2, winHeight*3, m_Bitmaps[4]);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

void
sWindowStyle::WriteBitmap(FILE* file, const CImage32* bitmap)
{
  word w = bitmap->GetWidth();
  word h = bitmap->GetHeight();
  fwrite(&w, 1, 2, file);
  fwrite(&h, 1, 2, file);
  fwrite(bitmap->GetPixels(), sizeof(RGBA), w * h, file);
}

////////////////////////////////////////////////////////////////////////////////

void
sWindowStyle::ReadBitmap(FILE* file, CImage32* bitmap)
{
  word w, h;
  fread(&w, 1, 2, file);
  fread(&h, 1, 2, file);
  bitmap->Resize(w, h);
  fread(bitmap->GetPixels(), sizeof(RGBA), w * h, file);
}

////////////////////////////////////////////////////////////////////////////////

void
sWindowStyle::ReadBitmap(FILE* file, CImage32* bitmap, int edge_width)
{
  bitmap->Resize(edge_width, edge_width);
  fread(bitmap->GetPixels(), sizeof(RGBA), edge_width * edge_width, file);
}

////////////////////////////////////////////////////////////////////////////////
