#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "Tileset.hpp"
#include "Image32.hpp"
#include "x++.hpp"
#include "packed.h"
#include "common_palettes.h"



////////////////////////////////////////////////////////////////////////////////

sTileset::sTileset()
: m_TileWidth(16)
, m_TileHeight(16)
, m_NumTiles(0)
, m_Tiles(NULL)
{
}

////////////////////////////////////////////////////////////////////////////////

sTileset::sTileset(int num_tiles)
: m_TileWidth(16)
, m_TileHeight(16)
, m_NumTiles(0)
, m_Tiles(NULL)
{
  AppendTiles(num_tiles);
}

////////////////////////////////////////////////////////////////////////////////

sTileset::sTileset(const char* filename)
: m_TileWidth(16)
, m_TileHeight(16)
, m_NumTiles(0)
, m_Tiles(NULL)
{
  Load(filename);
}

////////////////////////////////////////////////////////////////////////////////

sTileset::sTileset(const sTileset& tileset)
{
  m_TileWidth  = tileset.m_TileWidth;
  m_TileHeight = tileset.m_TileHeight;
  m_NumTiles   = tileset.m_NumTiles;
  m_Tiles      = duplicate(tileset.m_Tiles, m_NumTiles);
}

////////////////////////////////////////////////////////////////////////////////

sTileset::~sTileset()
{
  delete[] m_Tiles;
}

////////////////////////////////////////////////////////////////////////////////

sTileset&
sTileset::operator=(sTileset& tileset)
{
  m_TileWidth  = tileset.m_TileWidth;
  m_TileHeight = tileset.m_TileHeight;
  m_NumTiles   = tileset.m_NumTiles;

  delete[] m_Tiles;
  m_Tiles = duplicate(tileset.m_Tiles, m_NumTiles);

  return *this;
}

////////////////////////////////////////////////////////////////////////////////

bool
sTileset::Create(int num_tiles)
{
  m_TileWidth  = 16;
  m_TileHeight = 16;

  delete[] m_Tiles;
  m_NumTiles = 1;
  m_Tiles = new sTile[1];
  return true;
}

////////////////////////////////////////////////////////////////////////////////

PACKED_STRUCT(TILESET_HEADER)
  byte signature[4];
  word version;
  word num_tiles;
  word tile_width;
  word tile_height;
  word tile_bpp;
  byte compression;
  byte reserved[241];
END_STRUCT(TILESET_HEADER)

PACKED_STRUCT(TILE_INFORMATION_BLOCK)
  byte obsolete__;
  byte animated;
  word nexttile;
  word delay;
  byte reflective;
  byte blocked;
  byte reserved[24];
END_STRUCT(TILE_INFORMATION_BLOCK)

////////////////////////////////////////////////////////////////////////////////

// make sure all structs are the correct size
ASSERT_STRUCT_SIZE(TILESET_HEADER,         256)
ASSERT_STRUCT_SIZE(TILE_INFORMATION_BLOCK, 32)

////////////////////////////////////////////////////////////////////////////////

bool
sTileset::Load(const char* filename)
{
  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  bool result = LoadFromFile(file);

  fclose(file);
  return result;
}

////////////////////////////////////////////////////////////////////////////////

bool
sTileset::Import_Image(const char* filename)
{
  // load the new image
  CImage32 image;
  if (image.Load(filename) == false)
    return false;

  // check valid size
  if (image.GetWidth() % 16 != 0 ||
      image.GetHeight() % 16 != 0)
    return false;

  int   i_width  = image.GetWidth();
  int   i_height = image.GetHeight();
  RGBA* i_pixels = image.GetPixels();

  // free the old tileset
  delete[] m_Tiles;

  // fill the members
  m_NumTiles = (i_width / 16) * (i_height / 16);
  m_Tiles = new sTile[m_NumTiles];

  // set the pixels
  for (int x = 0; x < (int)image.GetWidth(); x++)
    for (int y = 0; y < (int)image.GetHeight(); y++)
    {
      int tilenum = (y / 16) * (i_width / 16) + (x / 16);
      int tilexoffset = x % 16;
      int tileyoffset = y % 16;

      m_Tiles[tilenum].GetPixels()[tileyoffset * 16 + tilexoffset] =
        i_pixels[y * i_width + x];
    }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sTileset::Import_VSP(const char* filename)
{
  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  word version;
  RGB palette[256];
  word numtiles;

  fread(&version,  1, 2,   file);
  fread(palette,   3, 256, file);
  fread(&numtiles, 1, 2,   file);

  // free the old tileset
  delete[] m_Tiles;

  // allocate the new one
  m_NumTiles = numtiles;
  m_Tiles = new sTile[numtiles];

  // decode the file
  if (version == 2)
  {
    for (int i = 0; i < numtiles; i++)
    {
      byte tile[256];

      fread(tile, 1, 256, file);
      for (int x = 0; x < 16; x++)
        for (int y = 0; y < 16; y++)
        {
          RGBA color;
          byte c = tile[y * 16 + x];
          if (c == 0)
          {
            color.red = 0;
            color.green = 0;
            color.blue = 0;
            color.alpha = 0;
          }
          else
          {
            color.red   = (byte)(palette[c].red   * 4);
            color.green = (byte)(palette[c].green * 4);
            color.blue  = (byte)(palette[c].blue  * 4);
            color.alpha = 255;
          }

          m_Tiles[i].GetPixels()[y * 16 + x] = color;
        }
    }
  }
  else
  {
    // for that wierd thing aen never told me in the vsp code
    fgetc(file);
    fgetc(file);
    fgetc(file);
    fgetc(file);

    // create a temporary buffer while setting up the decoding stuff...
    byte* buffer = (byte*)malloc(numtiles * 256);
    int len = numtiles * 256;
    int counter = 0;

    //start decoding!
    while (counter < len)
    {
      int c = fgetc(file);

      // if the c was 255 then it's a compressed value
      if (c == 255)
      {
        int run   = fgetc(file);
        int color = fgetc(file);

        for (int i = 0; i < run; i++)
        {
          buffer[counter] = color;
          counter++;
        }
      }
      else  // it's a normal value
      {
        buffer[counter] = c;
        counter++;
      }
    }

    // now, tranfer the decoded stuff into the tiles' data structure
    for (int i = 0; i < numtiles; i++)
      for (int j = 0; j < 256; j++)
      {
        if (buffer[i * 256 + j] == 0)
        {
          m_Tiles[i].GetPixels()[j].red   = 0;
          m_Tiles[i].GetPixels()[j].green = 0;
          m_Tiles[i].GetPixels()[j].blue  = 0;
          m_Tiles[i].GetPixels()[j].alpha = 0;
        }
        else
        {
          m_Tiles[i].GetPixels()[j].red   = (palette[buffer[i * 256 + j]].red)   * 4;
          m_Tiles[i].GetPixels()[j].green = (palette[buffer[i * 256 + j]].green) * 4;
          m_Tiles[i].GetPixels()[j].blue  = (palette[buffer[i * 256 + j]].blue)  * 4;
          m_Tiles[i].GetPixels()[j].alpha = 255;
        }
      }
  }

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sTileset::Import_TST(const char* filename)
{
  // TST file format created by Christoper B. Matthews for the RPG Toolkit Development System

  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  // read header
  word version;
  word numtiles;
  word detail;
  fread(&version, 1, 2, file);
  fread(&numtiles, 1, 2, file);
  fread(&detail, 1, 2, file);

  // check header for errors
  // only support details 2, 4, 6
  if (version != 20 && (detail == 2 || detail == 4 || detail == 6))
  {
    fclose(file);
    return false;
  }

  // allocate new tiles
  delete[] m_Tiles;
  m_NumTiles = numtiles;
  m_Tiles = new sTile[m_NumTiles];

  // read them from file
  for (int i = 0; i < m_NumTiles; i++)
  {
    sTile& tile = m_Tiles[i];
    switch (detail)
    {
      case 2: // 16x16, 24-bit color
      {
        for (int ix = 0; ix < 16; ix++)
          for (int iy = 0; iy < 16; iy++)
          {
            RGB rgb;
            fread(&rgb, 1, 3, file);
            byte alpha = 255;
            if (rgb.red == 0 && rgb.green == 1 && rgb.blue == 2)
              alpha = 0;
            tile.GetPixels()[iy * 16 + ix].red   = rgb.red;
            tile.GetPixels()[iy * 16 + ix].green = rgb.green;
            tile.GetPixels()[iy * 16 + ix].blue  = rgb.blue;
            tile.GetPixels()[iy * 16 + ix].alpha = alpha;
          }
        break;
      }

      case 4: // 16x16, 8-bit color
      case 6: // 16x16, 4-bit color
              // these two are effectively the same format
      {
        for (int ix = 0; ix < 16; ix++)
          for (int iy = 0; iy < 16; iy++)
          {
            byte b = fgetc(file);
            RGB rgb = dos_palette[b];
            byte alpha = 255;
            if (b == 255)
              alpha = 0;

            tile.GetPixels()[iy * 16 + ix].red   = rgb.red;
            tile.GetPixels()[iy * 16 + ix].green = rgb.green;
            tile.GetPixels()[iy * 16 + ix].blue  = rgb.blue;
            tile.GetPixels()[iy * 16 + ix].alpha = alpha;
          }

        break;
      }
    }
  }

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sTileset::Save(const char* filename) const
{
  // open the file
  FILE* file = fopen(filename, "wb");
  if (file == NULL)
    return false;

  bool result = SaveToFile(file);

  fclose(file);
  return result;
}

////////////////////////////////////////////////////////////////////////////////

bool
sTileset::LoadFromFile(FILE* file)
{
  // load the header
  TILESET_HEADER header;
  fread(&header, 1, sizeof(header), file);

  // check the header
  if (memcmp(header.signature, ".rts", 4) != 0 ||
      header.version != 1 ||
      header.tile_bpp != 32)
  {
    return false;
  }

  m_TileWidth  = header.tile_width;
  m_TileHeight = header.tile_height;

  // clear out the old tiles
  delete[] m_Tiles;
  m_NumTiles = header.num_tiles;
  m_Tiles = new sTile[m_NumTiles];

  // load the tiles
  for (int i = 0; i < m_NumTiles; i++)
  {
    RGBA* pixels = new RGBA[m_TileWidth * m_TileHeight];
    fread(pixels, sizeof(RGBA), m_TileWidth * m_TileHeight, file);
    m_Tiles[i].Resize(m_TileWidth, m_TileHeight);
    memcpy(m_Tiles[i].GetPixels(), pixels, m_TileWidth * m_TileHeight * sizeof(RGBA));
    delete[] pixels;
  }

  // load the tile info blocks
  for (int i = 0; i < m_NumTiles; i++)
  {
    TILE_INFORMATION_BLOCK tib;
    fread(&tib, 1, sizeof(tib), file);

    m_Tiles[i].SetAnimated(tib.animated ? true : false);
    m_Tiles[i].SetNextTile(tib.nexttile);
    m_Tiles[i].SetDelay(tib.delay);
    m_Tiles[i].SetReflective(tib.reflective ? true : false);
    
    // load obstruction data if it exists
    if (tib.blocked)
    {
      for (int iy = 0; iy < m_Tiles[i].GetHeight(); iy++)
        for (int ix = 0; ix < m_Tiles[i].GetWidth(); ix++)
          m_Tiles[i].SetObstructed(ix, iy, fgetc(file) != 0);
    }

  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sTileset::SaveToFile(FILE* file) const
{
  // write header
  TILESET_HEADER header;
  memset(&header, 0, sizeof(header));
  memcpy(header.signature, ".rts", 4);
  header.version = 1;
  header.num_tiles = m_NumTiles;
  header.tile_width = m_TileWidth;
  header.tile_height = m_TileHeight;
  header.tile_bpp = 32;
  header.compression = 0;
  fwrite(&header, 1, sizeof(header), file);

  // write the tiles
  for (int i = 0; i < m_NumTiles; i++)
    fwrite(m_Tiles[i].GetPixels(), sizeof(RGBA), m_TileWidth * m_TileHeight, file);

  // write the tile information blocks
  TILE_INFORMATION_BLOCK tib;
  memset(&tib, 0, sizeof(tib));
  for (int i = 0; i < m_NumTiles; i++)
  {
    tib.animated   = m_Tiles[i].IsAnimated();
    tib.nexttile   = m_Tiles[i].GetNextTile();
    tib.delay      = m_Tiles[i].GetDelay();
    tib.reflective = m_Tiles[i].IsReflective();
    tib.blocked    = 1;
    fwrite(&tib, 1, sizeof(tib), file);

    // write obstruction data
    for (int iy = 0; iy < m_Tiles[i].GetHeight(); iy++)
      for (int ix = 0; ix < m_Tiles[i].GetWidth(); ix++)
        fputc(m_Tiles[i].IsObstructed(ix, iy), file);
  }

  // write obstruction data

  return true;
}

////////////////////////////////////////////////////////////////////////////////

void
sTileset::Clear()
{
  m_NumTiles = 0;
  delete[] m_Tiles;
  m_Tiles = NULL;
}

////////////////////////////////////////////////////////////////////////////////

bool
sTileset::BuildFromImage(CImage32& image, int tile_width, int tile_height)
{
  if (image.GetWidth() % tile_width != 0 ||
      image.GetHeight() % tile_height != 0)
    return false;

  // clear out the old tiles
  delete[] m_Tiles;
  m_NumTiles = 0;
  m_Tiles = NULL;

  m_TileWidth = tile_width;
  m_TileHeight = tile_height;

  int num_x_tiles = image.GetWidth() / tile_width;
  int num_y_tiles = image.GetHeight() / tile_height;

  // grab each tile from the image
  for (int ty = 0; ty < num_y_tiles; ty++)
    for (int tx = 0; tx < num_x_tiles; tx++)
    {
      int src_x = tx * tile_width;
      int src_y = ty * tile_height;

      // grab a tile
      RGBA* tile = new RGBA[tile_width * tile_height];
      for (int iy = 0; iy < tile_height; iy++)
      {
        memcpy(tile + iy * tile_width,
               image.GetPixels() + (src_y + iy) * image.GetWidth() + src_x,
               tile_width * sizeof(RGBA));
      }

      // if the tile is not in the tileset already, add it
      bool in_tileset = false;
      for (int i = 0; i < m_NumTiles; i++)
        if (memcmp(tile, m_Tiles[i].GetPixels(), tile_width * tile_height * sizeof(RGBA)) == 0)
        {
          in_tileset = true;
          break;
        }

      if (!in_tileset)
      {
        AppendTiles(1);
        memcpy(m_Tiles[m_NumTiles - 1].GetPixels(), tile, tile_width * tile_height * sizeof(RGBA));
      }

      delete[] tile;
    }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

int
sTileset::GetTileWidth() const
{
  return m_TileWidth;
}

////////////////////////////////////////////////////////////////////////////////

int
sTileset::GetTileHeight() const
{
  return m_TileHeight;
}

////////////////////////////////////////////////////////////////////////////////

void
sTileset::SetTileSize(int w, int h)
{
  m_TileWidth  = w;
  m_TileHeight = h;
  for (int i = 0; i < m_NumTiles; i++)
    m_Tiles[i].Resize(w, h);
}

////////////////////////////////////////////////////////////////////////////////

void
sTileset::InsertTiles(int insert_at, int num_tiles)
{
  // resize the tile array
  resize(m_Tiles, m_NumTiles, m_NumTiles + num_tiles);
  m_NumTiles += num_tiles;

  // move the tiles back
  for (int i = m_NumTiles - 1; i >= insert_at + num_tiles; i--)
  {
    m_Tiles[i] = m_Tiles[i - num_tiles];
    m_Tiles[i - num_tiles].Clear();
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sTileset::AppendTiles(int num_tiles)
{
  // allocate the new tiles
  sTile* new_tiles = new sTile[m_NumTiles + num_tiles];
  for (int i = 0; i < m_NumTiles + num_tiles; i++)
    new_tiles[i].Resize(m_TileWidth, m_TileHeight);

  // copy in the old tiles
  for (int i = 0; i < m_NumTiles; i++)
    new_tiles[i] = m_Tiles[i];  
  
  m_NumTiles += num_tiles;
  delete[] m_Tiles;
  m_Tiles = new_tiles;
}

////////////////////////////////////////////////////////////////////////////////

void
sTileset::DeleteTiles(int delete_at, int num_tiles)
{
  sTile* new_tiles = new sTile[m_NumTiles - num_tiles];
  for (int i = 0; i < m_NumTiles - num_tiles; i++)
    new_tiles[i].Resize(m_TileWidth, m_TileHeight);

  for (int i = 0; i < delete_at; i++)
    new_tiles[i] = m_Tiles[i];
  for (int i = delete_at; i < m_NumTiles - num_tiles; i++)
    new_tiles[i] = m_Tiles[i + num_tiles];

  m_NumTiles -= num_tiles;
  delete[] m_Tiles;
  m_Tiles = new_tiles;
}

////////////////////////////////////////////////////////////////////////////////

bool
sTileset::InsertImage(int insert_at, const char* filename)
{
  sTileset tileset;
  if (!tileset.Import_Image(filename))
    return false;

  InsertTiles(insert_at, tileset.GetNumTiles());

  for (int i = 0; i < tileset.GetNumTiles(); i++)
    m_Tiles[insert_at + i] = tileset.GetTile(i);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sTileset::AppendImage(const char* filename)
{
  sTileset tileset;
  if (!tileset.Import_Image(filename))
    return false;

  int iInsertAt = GetNumTiles();
  AppendTiles(tileset.GetNumTiles());

  for (int i = 0; i < tileset.GetNumTiles(); i++)
    m_Tiles[iInsertAt + i] = tileset.GetTile(i);

  return true;
}

////////////////////////////////////////////////////////////////////////////////
