#include <stdio.h>
#include <string.h>
#include "Font.hpp"
#include "x++.hpp"
#include "packed.h"


////////////////////////////////////////////////////////////////////////////////

sFont::sFont(int num_characters, int width, int height)
{
  m_NumCharacters = num_characters;
  m_Characters = new sFontCharacter[num_characters];
  for (int i = 0; i < num_characters; i++)
    m_Characters[i].Resize(width, height);
}

////////////////////////////////////////////////////////////////////////////////

sFont::~sFont()
{
  delete[] m_Characters;
}

////////////////////////////////////////////////////////////////////////////////

PACKED_STRUCT(FONT_HEADER)
  byte signature[4];
  word version;
  word num_characters;
  byte reserved[248];
END_STRUCT(FONT_HEADER)

PACKED_STRUCT(CHARACTER_HEADER)
  word width;
  word height;
  byte reserved[28];
END_STRUCT(CHARACTER_HEADER)

ASSERT_STRUCT_SIZE(FONT_HEADER, 256)
ASSERT_STRUCT_SIZE(CHARACTER_HEADER, 32);

////////////////////////////////////////////////////////////////////////////////

bool
sFont::Load(const char* filename)
{
  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  // read and check header
  FONT_HEADER header;
  fread(&header, 1, sizeof(header), file);
  if (memcmp(header.signature, ".rfn", 4) != 0 ||
      (header.version != 1 && header.version != 2))
  {
    fclose(file);
    return false;
  }

  // allocate characters
  m_NumCharacters = header.num_characters;
  m_Characters    = new sFontCharacter[m_NumCharacters];

  // read them
  for (int i = 0; i < m_NumCharacters; i++)
  {
    CHARACTER_HEADER character_header;
    fread(&character_header, 1, sizeof(character_header), file);
    m_Characters[i].Resize(character_header.width, character_header.height);

    // version 1 = greyscale
    if (header.version == 1)
    {
      byte* buffer = new byte[character_header.width * character_header.height];
      fread(buffer, 1, character_header.width * character_header.height, file);

      for (int j = 0; j < character_header.width * character_header.height; j++)
      {
        m_Characters[i].GetPixels()[j].red   = 255;
        m_Characters[i].GetPixels()[j].green = 255;
        m_Characters[i].GetPixels()[j].blue  = 255;
        m_Characters[i].GetPixels()[j].alpha = buffer[j];
      }
      
      delete[] buffer;
    }
    else  // version 2 (RGBA)
      fread(m_Characters[i].GetPixels(), sizeof(RGBA), character_header.width * character_header.height, file);
  }

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sFont::Save(const char* filename) const
{
  FILE* file = fopen(filename, "wb");
  if (file == NULL)
    return false;

  // write header
  FONT_HEADER header;
  memset(&header, 0, sizeof(header));
  memcpy(header.signature, ".rfn", 4);
  header.version = 2;
  header.num_characters = m_NumCharacters;
  fwrite(&header, 1, sizeof(header), file);

  // write characters
  for (int i = 0; i < m_NumCharacters; i++)
  {
    CHARACTER_HEADER character_header;
    memset(&character_header, 0, sizeof(character_header));
    character_header.width  = m_Characters[i].GetWidth();
    character_header.height = m_Characters[i].GetHeight();
    fwrite(&character_header, 1, sizeof(character_header), file);
    fwrite(m_Characters[i].GetPixels(), character_header.width * character_header.height, sizeof(RGBA), file);
  }

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

void
sFont::SetNumCharacters(int num_characters)
{
  sFontCharacter* newCharacters = new sFontCharacter[num_characters];
  int copy_count = m_NumCharacters < num_characters ? m_NumCharacters : num_characters;
  for (int i = 0; i < copy_count; i++)
    newCharacters[i] = m_Characters[i];

  // free old characters
  delete[] m_Characters;
  m_Characters = newCharacters;
  m_NumCharacters = num_characters;
}

////////////////////////////////////////////////////////////////////////////////
