#define DIRECTDRAW_VERSION 0x0300
#include <windows.h>
#include <ddraw.h>
#include <stdio.h>
#include "colorspace.h"
#include "win32x.h"
#include "../common/video.h"
#include "x++.hpp"
#include "resource.h"



// these masks are used to calculate averages of 15-bit and 16-bit color pixels
#define MASK565 0xF7DE  // 11110 111110 11110
#define MASK555 0x7BDE  // 0 11110 11110 11110



// DATA TYPES //

typedef struct _IMAGE
{
  int width;
  int height;

  word* rgb;
  byte* alpha;

  bool (*blit_routine)(_IMAGE* image, int x, int y);

  RGBA* locked_pixels;
}* IMAGE;


struct CONFIGURATION
{
  bool fullscreen;
  bool vsync;
  bool fast_translucency;
};



// FUNCTION PROTOTYPES //

static void LoadConfiguration();
static void SaveConfiguration();
static BOOL CALLBACK ConfigureDialogProc(HWND window, UINT message, WPARAM wparam, LPARAM lparam);

static bool InitFullscreen();
static bool SetDisplayMode();
static bool CreateSurfaces();
static bool InitWindowed();

static bool CloseFullscreen();
static bool CloseWindowed();

static void FillImagePixels(IMAGE image, RGBA* pixels);
static void OptimizeBlitRoutine(IMAGE image);

static bool NullBlit(IMAGE image, int x, int y);
static bool TileBlit(IMAGE image, int x, int y);
static bool SpriteBlit(IMAGE image, int x, int y);
static bool NormalBlit(IMAGE image, int x, int y);



// INLINE FUNCTIONS //

inline word PackPixel565(RGBA pixel)
{
  return (word)(((pixel.red   >> 3) << 11) +
                ((pixel.green >> 2) <<  5) +
                ((pixel.blue  >> 3) <<  0));
}

inline RGBA UnpackPixel565(word pixel)
{
  RGBA rgba = {
    ((pixel & 0xF800) >> 11) << 3, // 11111 000000 00000
    ((pixel & 0x07E0) >>  5) << 2, // 00000 111111 00000
    ((pixel & 0x001F) >>  0) << 3, // 00000 000000 11111
  };
  return rgba;
}

inline word PackPixel555(RGBA pixel)
{
  return (word)(
    ((pixel.red   >> 3) << 10) + 
    ((pixel.green >> 3) << 5) +
    ((pixel.blue  >> 3) << 0));
}

inline RGBA UnpackPixel555(word pixel)
{
  RGBA rgba = {
    ((pixel & 0x7C00) >> 10) << 3, // 0 11111 00000 00000
    ((pixel & 0x03E0) >>  5) << 3, // 0 00000 11111 00000
    ((pixel & 0x001F) >>  0) << 3, // 0 00000 00000 11111
  };
  return rgba;
}

static int AlphaTable[256] =
{
/* 00-10 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 10-30 */ 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
/* 30-50 */ 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40,
/* 50-70 */ 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60,
/* 70-90 */ 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
/* 90-B0 */ 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0,
/* B0-D0 */ 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0,
/* D0-F0 */ 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0,
/* F0-FF */ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
};

// returns (hex): 00, 20, 40, 60, 80, A0, C0, E0, FF
inline int FastAlphaLevel(int alpha)
{
  return AlphaTable[alpha];
}



// GLOBAL VARIABLES

static CONFIGURATION Configuration;
static enum { RGB565, RGB555 } PixelFormat;

static HWND  SphereWindow;
static word* ScreenBuffer;

static LONG OldWindowStyle;
static LONG OldWindowStyleEx;

// fullscreen output
static LPDIRECTDRAW        dd;
static LPDIRECTDRAWSURFACE ddPrimary;
static LPDIRECTDRAWSURFACE ddSecondary;

// windowed output
static HDC     RenderDC;
static HBITMAP RenderBitmap;



////////////////////////////////////////////////////////////////////////////////

void EXPORT GetDriverInfo(DRIVERINFO* driverinfo)
{
  driverinfo->name   = "Standard 16-bit Color";
  driverinfo->author = "Chad Austin";
  driverinfo->date   = __DATE__;
  driverinfo->version = "1.00";
  driverinfo->description = "15/16-bit color output in both windowed and fullscreen modes";
}

////////////////////////////////////////////////////////////////////////////////

void EXPORT ConfigureDriver(HWND parent)
{
  LoadConfiguration();
  DialogBox(DriverInstance, MAKEINTRESOURCE(IDD_CONFIGURE), parent, ConfigureDialogProc);
  SaveConfiguration();
}

////////////////////////////////////////////////////////////////////////////////

void LoadConfiguration()
{
  char config_file_name[MAX_PATH];
  GetDriverConfigFile(config_file_name);

  Configuration.fullscreen        = (GetPrivateProfileInt("standard16", "Fullscreen",       1, config_file_name) != 0);
  Configuration.vsync             = (GetPrivateProfileInt("standard16", "VSync",            1, config_file_name) != 0);
  Configuration.fast_translucency = (GetPrivateProfileInt("standard16", "FastTranslucency", 1, config_file_name) != 0);
}

////////////////////////////////////////////////////////////////////////////////

void SaveConfiguration()
{
  char config_file_name[MAX_PATH];
  GetDriverConfigFile(config_file_name);

  WritePrivateProfileInt("standard16", "Fullscreen",       Configuration.fullscreen,        config_file_name);
  WritePrivateProfileInt("standard16", "VSync",            Configuration.vsync,             config_file_name);
  WritePrivateProfileInt("standard16", "FastTranslucency", Configuration.fast_translucency, config_file_name);
}

////////////////////////////////////////////////////////////////////////////////

BOOL CALLBACK ConfigureDialogProc(HWND window, UINT message, WPARAM wparam, LPARAM lparam)
{
  switch (message)
  {
    case WM_INITDIALOG:
      // set check boxes
      SendDlgItemMessage(window, IDC_FULLSCREEN,       BM_SETCHECK, (Configuration.fullscreen        ? BST_CHECKED : BST_UNCHECKED), 0);
      SendDlgItemMessage(window, IDC_VSYNC,            BM_SETCHECK, (Configuration.vsync             ? BST_CHECKED : BST_UNCHECKED), 0);
      SendDlgItemMessage(window, IDC_FASTTRANSLUCENCY, BM_SETCHECK, (Configuration.fast_translucency ? BST_CHECKED : BST_UNCHECKED), 0);

      // update the check states
      SendMessage(window, WM_COMMAND, MAKEWPARAM(IDC_FULLSCREEN, BN_PUSHED), 0);

      return TRUE;

    ////////////////////////////////////////////////////////////////////////////

    case WM_COMMAND:
      switch (LOWORD(wparam))
      {
        case IDOK:
          Configuration.fullscreen        = (IsDlgButtonChecked(window, IDC_FULLSCREEN) != FALSE);
          Configuration.vsync             = (IsDlgButtonChecked(window, IDC_VSYNC) != FALSE);
          Configuration.fast_translucency = (IsDlgButtonChecked(window, IDC_FASTTRANSLUCENCY) != FALSE);

          EndDialog(window, 1);
          return TRUE;

        case IDCANCEL:
          EndDialog(window, 0);
          return TRUE;

        case IDC_FULLSCREEN:
          EnableWindow(GetDlgItem(window, IDC_VSYNC), IsDlgButtonChecked(window, IDC_FULLSCREEN));
          return TRUE;
      }
      return FALSE;

    ////////////////////////////////////////////////////////////////////////////

    default:
      return FALSE;
  }
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT InitVideoDriver(HWND window, int screen_width, int screen_height)
{
  SphereWindow = window;
  ScreenWidth  = screen_width;
  ScreenHeight = screen_height;

  // set default clipping rectangle
  SetClippingRectangle(0, 0, screen_width, screen_height);
  
  LoadConfiguration();
  if (Configuration.fullscreen)
    return InitFullscreen();
  else
    return InitWindowed();

  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool InitFullscreen()
{
  HRESULT ddrval;
  bool    retval;

  // store old window styles
  OldWindowStyle = GetWindowLong(SphereWindow, GWL_STYLE);
  OldWindowStyleEx = GetWindowLong(SphereWindow, GWL_EXSTYLE);

  SetWindowLong(SphereWindow, GWL_STYLE, WS_POPUP);
  SetWindowLong(SphereWindow, GWL_EXSTYLE, 0);

  // create DirectDraw object
  ddrval = DirectDrawCreate(NULL, &dd, NULL);
  if (ddrval != DD_OK)
  {
    MessageBox(SphereWindow, "DirectDrawCreate() failed", "standard16", MB_OK);
    return false;
  }

  // set application behavior
  ddrval = dd->SetCooperativeLevel(SphereWindow, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN);
  if (ddrval != DD_OK)
  {
    dd->Release();
    MessageBox(SphereWindow, "SetCooperativeLevel() failed", "standard16", MB_OK);
    return false;
  }

  // set display mode
  retval = SetDisplayMode();
  if (retval == false)
  {
    dd->Release();
    MessageBox(SphereWindow, "SetDisplayMode() failed", "standard16", MB_OK);
    return false;
  }

  // create surfaces
  retval = CreateSurfaces();
  if (retval == false)
  {
    dd->Release();
    MessageBox(SphereWindow, "CreateSurfaces() failed", "standard16", MB_OK);
    return false;
  }

  ScreenBuffer = new word[ScreenWidth * ScreenHeight];

  ShowCursor(FALSE);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool SetDisplayMode()
{
  HRESULT ddrval = dd->SetDisplayMode(ScreenWidth, ScreenHeight, 16);
  if (ddrval != DD_OK)
    return false;

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool CreateSurfaces()
{
  // define the surface
  DDSURFACEDESC ddsd;
  ddsd.dwSize = sizeof(ddsd);
  
  if (Configuration.vsync)
  {
    ddsd.dwFlags           = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
    ddsd.ddsCaps.dwCaps    = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP | DDSCAPS_COMPLEX;
    ddsd.dwBackBufferCount = 1;
  }
  else
  {
    ddsd.dwFlags        = DDSD_CAPS;
    ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
  }

  // create the primary surface
  HRESULT ddrval = dd->CreateSurface(&ddsd, &ddPrimary, NULL);
  if (ddrval != DD_OK)
    return false;

  if (Configuration.vsync)
  {
    ddsd.ddsCaps.dwCaps = DDSCAPS_BACKBUFFER;
    ddrval = ddPrimary->GetAttachedSurface(&ddsd.ddsCaps, &ddSecondary);
    if (ddrval != DD_OK)
    {
      ddPrimary->Release();
      return false;
    }
  }

  // determine bits per pixel
  DDPIXELFORMAT ddpf;
  ddpf.dwSize = sizeof(ddpf);
  ddpf.dwFlags = DDPF_RGB;
  ddrval = ddPrimary->GetPixelFormat(&ddpf);
  if (ddrval != DD_OK)
  {
    dd->Release();
    return false;
  }

  // 5:6:5 -- F800 07E0 001F
  // 5:5:5 -- 7C00 03E0 001F

  if (ddpf.dwRBitMask == 0xF800)
    PixelFormat = RGB565;
  else if (ddpf.dwRBitMask == 0x7C00)
    PixelFormat = RGB555;
  else
  {
    dd->Release();
    return false;
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool InitWindowed()
{
  // create the render DC
  RenderDC = CreateCompatibleDC(NULL);
  if (RenderDC == NULL)
    return false;

  // define/create the DIB section
  BITMAPINFO bmi;
  memset(&bmi, 0, sizeof(bmi));
  BITMAPINFOHEADER& bmih = bmi.bmiHeader;
  bmih.biSize        = sizeof(bmih);
  bmih.biWidth       = ScreenWidth;
  bmih.biHeight      = -ScreenHeight;
  bmih.biPlanes      = 1;
  bmih.biBitCount    = 16;
  bmih.biCompression = BI_RGB;
  RenderBitmap = CreateDIBSection(RenderDC, &bmi, DIB_RGB_COLORS, (void**)&ScreenBuffer, NULL, 0);
  if (RenderBitmap == NULL)
  {
    DeleteDC(RenderDC);
    return false;
  }

  SelectObject(RenderDC, RenderBitmap);

  // center the window
  RECT WindowRect = { 0, 0, ScreenWidth, ScreenHeight };
  AdjustWindowRectEx(
    &WindowRect,
    GetWindowLong(SphereWindow, GWL_STYLE),
    (GetMenu(SphereWindow) ? TRUE : FALSE),
    GetWindowLong(SphereWindow, GWL_EXSTYLE));

  int window_width  = WindowRect.right - WindowRect.left;
  int window_height = WindowRect.bottom - WindowRect.top;

  MoveWindow(
    SphereWindow,
    (GetSystemMetrics(SM_CXSCREEN) - window_width) / 2,
    (GetSystemMetrics(SM_CYSCREEN) - window_height) / 2,
    window_width,
    window_height,
    TRUE);

  // we know that 16-bit color DIBs are always 5:5:5
  PixelFormat = RGB555;

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT CloseVideoDriver()
{
  if (Configuration.fullscreen)
    return CloseFullscreen();
  else
    return CloseWindowed();
}

////////////////////////////////////////////////////////////////////////////////

bool CloseFullscreen()
{
  SetWindowLong(SphereWindow, GWL_STYLE, OldWindowStyle);
  SetWindowLong(SphereWindow, GWL_EXSTYLE, OldWindowStyleEx);

  ShowCursor(TRUE);
  delete[] ScreenBuffer;
  dd->Release();
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool CloseWindowed()
{
  DeleteDC(RenderDC);
  DeleteObject(RenderBitmap);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT FlipScreen()
{
  if (Configuration.fullscreen)
  {
    LPDIRECTDRAWSURFACE surface;
    if (Configuration.vsync)
      surface = ddSecondary;
    else
      surface = ddPrimary;

    // lock the surface
    DDSURFACEDESC ddsd;
    ddsd.dwSize = sizeof(ddsd);
    HRESULT ddrval = surface->Lock(NULL, &ddsd, DDLOCK_WAIT, NULL);

    // if the surface was lost, restore it
    if (ddrval == DDERR_SURFACELOST)
    {
      surface->Restore();
      if (surface == ddSecondary)
        ddPrimary->Restore();

      // attempt to lock again
      ddrval = surface->Lock(NULL, &ddsd, DDLOCK_WAIT, NULL);
      if (ddrval != DD_OK)
      {
        Sleep(100);
        return false;
      }
    }

    for (int i = 0; i < ScreenHeight; i++)
      memcpy((byte*)ddsd.lpSurface + i * ddsd.lPitch, ScreenBuffer + i * ScreenWidth, ScreenWidth * 2);

    // unlock the surface and do the flip!
    surface->Unlock(NULL);
    if (Configuration.vsync)
      ddPrimary->Flip(NULL, DDFLIP_WAIT);
  }
  else
  {
    // if odd width...
    if (ScreenWidth % 2 == 1)
    {
      // make sure the lines begin on dword boundaries
      for (int i = ScreenHeight - 1; i >= 0; i--)
      {
        memmove(
          ScreenBuffer + i * (ScreenWidth + 1),
          ScreenBuffer + i * ScreenWidth,
          ScreenWidth * 2);
      }
    }

    // blit the render buffer to the window
    HDC dc = GetDC(SphereWindow);
    BitBlt(dc, 0, 0, ScreenWidth, ScreenHeight, RenderDC, 0, 0, SRCCOPY);
    ReleaseDC(SphereWindow, dc);
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT ApplyColorMask(RGBA mask)
{
  if (mask.alpha == 0)   // no mask
    return true;

  if (mask.alpha == 255) // complete mask
  {
    word pixel = (PixelFormat == RGB565 ? PackPixel565(mask) : PackPixel555(mask));

    word* dest = ScreenBuffer;
    while (dest < ScreenBuffer + ScreenWidth * ScreenHeight)
      *dest++ = pixel;

    return true;
  }

  // fast translucency
  if (Configuration.fast_translucency)
  {
    word MASK  = (PixelFormat == RGB565 ? MASK565 : MASK555);
    word color = (PixelFormat == RGB565 ? PackPixel565(mask) : PackPixel555(mask));

    switch (FastAlphaLevel(mask.alpha))
    {
      case 0x00: // 0% ---------------------------------------------------------
        return true;

      case 0x20: // 12.5% ------------------------------------------------------
      {
        word src;
        word* dest = ScreenBuffer;
        while (dest < ScreenBuffer + ScreenWidth * ScreenHeight)
        {
          src   = (*dest & MASK) / 2 + (color & MASK) / 2;
          src   = (*dest & MASK) / 2 + (src   & MASK) / 2;
          *dest = (*dest & MASK) / 2 + (src   & MASK) / 2;
          dest++;
        }
        return true;
      }

      case 0x40: // 25% --------------------------------------------------------
      {
        word src;
        word* dest = ScreenBuffer;
        while (dest < ScreenBuffer + ScreenWidth * ScreenHeight)
        {
          src   = (*dest & MASK) / 2 + (color & MASK) / 2;
          *dest = (*dest & MASK) / 2 + (src   & MASK) / 2;
          dest++;
        }
        return true;
      }

      case 0x60: // 37.5% ------------------------------------------------------
      {
        word src;
        word* dest = ScreenBuffer;
        while (dest < ScreenBuffer + ScreenWidth * ScreenHeight)
        {
          src   = (*dest & MASK) / 2 + (color & MASK) / 2;
          *dest = (*dest & MASK) / 2 + (src   & MASK) / 2;
          *dest = (*dest & MASK) / 2 + (src   & MASK) / 2;
          dest++;
        }
        return true;
      }

      case 0x80: // 50% --------------------------------------------------------
      {
        word* dest = ScreenBuffer;
        while (dest < ScreenBuffer + ScreenWidth * ScreenHeight)
        {
          *dest = (*dest & MASK) / 2 + (color & MASK) / 2;
          dest++;
        }
        return true;
      }

      case 0xA0: // 62.5% ------------------------------------------------------
      {
        word src;
        word* dest = ScreenBuffer;
        while (dest < ScreenBuffer + ScreenWidth * ScreenHeight)
        {
          *dest = (*dest & MASK) / 2 + (color & MASK) / 2;
          src   = (*dest & MASK) / 2 + (color & MASK) / 2;
          *dest = (*dest & MASK) / 2 + (src   & MASK) / 2;
          dest++;
        }
        return true;
      }

      case 0xC0: // 75% --------------------------------------------------------
      {
        word* dest = ScreenBuffer;
        while (dest < ScreenBuffer + ScreenWidth * ScreenHeight)
        {
          *dest = (*dest & MASK) / 2 + (color & MASK) / 2;
          *dest = (*dest & MASK) / 2 + (color & MASK) / 2;
          dest++;
        }
        return true;
      }

      case 0xE0: // 87.5% ------------------------------------------------------
      {
        word* dest = ScreenBuffer;
        while (dest < ScreenBuffer + ScreenWidth * ScreenHeight)
        {
          *dest = (*dest & MASK) / 2 + (color & MASK) / 2;
          *dest = (*dest & MASK) / 2 + (color & MASK) / 2;
          *dest = (*dest & MASK) / 2 + (color & MASK) / 2;
          dest++;
        }
        return true;
      }

      case 0xFF: // 100% -------------------------------------------------------
      {
        word* dest = ScreenBuffer;
        while (dest < ScreenBuffer + ScreenWidth * ScreenHeight)
          *dest++ = color;
        return true;
      }

    } // end switch
  }

  // slow translucency
  else
  {
    if (PixelFormat == RGB565)
    {
      // 5:6:5
      word* dest = ScreenBuffer;
      while (dest < ScreenBuffer + ScreenWidth * ScreenHeight)
      {
        RGBA pixel = UnpackPixel565(*dest);
        pixel.red   = (mask.red   * mask.alpha + pixel.red   * (255 - mask.alpha)) / 256;
        pixel.green = (mask.green * mask.alpha + pixel.green * (255 - mask.alpha)) / 256;
        pixel.blue  = (mask.blue  * mask.alpha + pixel.blue  * (255 - mask.alpha)) / 256;
        *dest = PackPixel565(pixel);
        dest++;
      }
    }
    else
    {
      // 5:5:5
      word* dest = ScreenBuffer;
      while (dest < ScreenBuffer + ScreenWidth * ScreenHeight)
      {
        RGBA pixel = UnpackPixel555(*dest);
        pixel.red   = (mask.red   * mask.alpha + pixel.red   * (255 - mask.alpha)) / 256;
        pixel.green = (mask.green * mask.alpha + pixel.green * (255 - mask.alpha)) / 256;
        pixel.blue  = (mask.blue  * mask.alpha + pixel.blue  * (255 - mask.alpha)) / 256;
        *dest = PackPixel555(pixel);
        dest++;
      }
    }
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

IMAGE EXPORT CreateImage(int width, int height, RGBA* pixels)
{
  IMAGE image = new _IMAGE;
  image->width  = width;
  image->height = height;

  FillImagePixels(image, pixels);
  OptimizeBlitRoutine(image);
  return image;
}

////////////////////////////////////////////////////////////////////////////////

void FillImagePixels(IMAGE image, RGBA* pixels)
{
  // rgb
  image->rgb = new word[image->width * image->height];
  if (PixelFormat == RGB565)
  {
    for (int i = 0; i < image->width * image->height; i++)
      image->rgb[i] = PackPixel565(pixels[i]);
  }
  else
  {
    for (int i = 0; i < image->width * image->height; i++)
      image->rgb[i] = PackPixel555(pixels[i]);
  }

  // alpha
  image->alpha = new byte[image->width * image->height];
  for (int i = 0; i < image->width * image->height; i++)
    image->alpha[i] = FastAlphaLevel(pixels[i].alpha);
}

////////////////////////////////////////////////////////////////////////////////

void OptimizeBlitRoutine(IMAGE image)
{
  // null blit
  bool is_empty = true;
  for (int i = 0; i < image->width * image->height; i++)
    if (image->alpha[i] != 0)
    {
      is_empty = false;
      break;
    }
  if (is_empty)
  {
    image->blit_routine = NullBlit;
    return;
  }

  // tile blit
  bool is_tile = true;
  for (int i = 0; i < image->width * image->height; i++)
    if (image->alpha[i] != 255)
    {
      is_tile = false;
      break;
    }
  if (is_tile)
  {
    image->blit_routine = TileBlit;
    return;
  }

  // sprite blit
  bool is_sprite = true;
  for (int i = 0; i < image->width * image->height; i++)
    if (image->alpha[i] != 0 &&
        image->alpha[i] != 255)
    {
      is_sprite = false;
      break;
    }
  if (is_sprite)
  {
    image->blit_routine = SpriteBlit;
    return;
  }

  // normal blit
  image->blit_routine = NormalBlit;
}

////////////////////////////////////////////////////////////////////////////////

IMAGE EXPORT GrabImage(int x, int y, int width, int height)
{
  if (x < 0 ||
      y > 0 ||
      x + width > ScreenWidth ||
      y + height > ScreenHeight)
    return NULL;

  IMAGE image = new _IMAGE;
  image->width        = width;
  image->height       = height;
  image->blit_routine = TileBlit;
  
  image->rgb = new word[width * height];
  image->alpha = new byte[width * height];

  for (int iy = 0; iy < height; iy++)
    memcpy(image->rgb + iy * width,
           ScreenBuffer + (y + iy) * ScreenWidth + x,
           width * 2);
  
  memset(image->alpha, 255, width * height);

  return image;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT DestroyImage(IMAGE image)
{
  delete[] image->rgb;
  delete[] image->alpha;
  delete image;
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT BlitImage(IMAGE image, int x, int y)
{
  // don't draw it if it's off the screen
  if (x + (int)image->width < ClippingRectangle.left ||
      y + (int)image->height < ClippingRectangle.top ||
      x >= ClippingRectangle.right ||
      y >= ClippingRectangle.bottom)
    return true;

  return image->blit_routine(image, x, y);
}

////////////////////////////////////////////////////////////////////////////////

bool NullBlit(IMAGE image, int x, int y)
{
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool TileBlit(IMAGE image, int x, int y)
{
  calculate_clipping_metrics(image->width, image->height);

  for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
    memcpy(ScreenBuffer + (y + iy) * ScreenWidth + x + image_offset_x,
           image->rgb + iy * image->width + image_offset_x,
           image_blit_width * 2);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool SpriteBlit(IMAGE image, int x, int y)
{
  calculate_clipping_metrics(image->width, image->height);

  for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
    for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
    {
      if (image->alpha[iy * image->width + ix])
        ScreenBuffer[(y + iy) * ScreenWidth + (x + ix)] =
          image->rgb[iy * image->width + ix];
    }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool NormalBlit(IMAGE image, int x, int y)
{
  calculate_clipping_metrics(image->width, image->height);

  // FAST TRANSLUCENCY
  if (Configuration.fast_translucency)
  {
    word MASK = (PixelFormat == RGB565 ? MASK565 : MASK555);

    for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
      for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
      {
        word* dest = ScreenBuffer + (y + iy) * ScreenWidth + (x + ix);
        word  src  = image->rgb[iy * image->width + ix];

        switch (image->alpha[iy * image->width + ix])
        {
          case 255:
            *dest = src;
            break;

          case 224:
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            break;

          case 192:
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            break;

          case 160:
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            src   = (*dest & MASK) / 2 + (src & MASK) / 2;
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            break;

          case 128:
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            break;

          case 96:
            src   = (*dest & MASK) / 2 + (src & MASK) / 2;
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            break;

          case 64:
            src   = (*dest & MASK) / 2 + (src & MASK) / 2;
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            break;

          case 32:
            src   = (*dest & MASK) / 2 + (src & MASK) / 2;
            src   = (*dest & MASK) / 2 + (src & MASK) / 2;
            *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
            break;
        }
      }

  }

  // ACCURATE TRANSLUCENCY
  else
  {
    if (PixelFormat == RGB565)
    {
      // 5:6:5
      for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
        for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
        {
          word* dest   = ScreenBuffer + (y + iy) * ScreenWidth + (x + ix);
          word  src    = image->rgb[iy * image->width + ix];
          int   alpha  = image->alpha[iy * image->width + ix];
      
          RGBA out = UnpackPixel565(*dest);
          RGBA in  = UnpackPixel565(src);
          out.red   = (in.red   * alpha + out.red   * (256 - alpha)) / 256;
          out.green = (in.green * alpha + out.green * (256 - alpha)) / 256;
          out.blue  = (in.blue  * alpha + out.blue  * (256 - alpha)) / 256;

          *dest = PackPixel565(out);
        }
    }
    else
    {
      // 5:5:5
      for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
        for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
        {
          word* dest   = ScreenBuffer + (y + iy) * ScreenWidth + (x + ix);
          word  src    = image->rgb[iy * image->width + ix];
          int   alpha  = image->alpha[iy * image->width + ix];
      
          RGBA out = UnpackPixel555(*dest);
          RGBA in  = UnpackPixel555(src);
          out.red   = (in.red   * alpha + out.red   * (256 - alpha)) / 256;
          out.green = (in.green * alpha + out.green * (256 - alpha)) / 256;
          out.blue  = (in.blue  * alpha + out.blue  * (256 - alpha)) / 256;

          *dest = PackPixel555(out);
        }
    }
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

int EXPORT GetImageWidth(IMAGE image)
{
  return image->width;
}

////////////////////////////////////////////////////////////////////////////////

int EXPORT GetImageHeight(IMAGE image)
{
  return image->height;
}

////////////////////////////////////////////////////////////////////////////////

RGBA* EXPORT LockImage(IMAGE image)
{
  image->locked_pixels = new RGBA[image->width * image->height];

  // rgb
  if (PixelFormat == RGB565)
  {
    // 5:6:5
    for (int i = 0; i < image->width * image->height; i++)
      image->locked_pixels[i] = UnpackPixel565(image->rgb[i]);
  }
  else
  {
    // 5:5:5
    for (int i = 0; i < image->width * image->height; i++)
      image->locked_pixels[i] = UnpackPixel555(image->rgb[i]);
  }

  // alpha
  for (int i = 0; i < image->width * image->height; i++)
    image->locked_pixels[i].alpha = image->alpha[i];

  return image->locked_pixels;
}

////////////////////////////////////////////////////////////////////////////////

void EXPORT UnlockImage(IMAGE image)
{
  delete[] image->rgb;
  delete[] image->alpha;

  FillImagePixels(image, image->locked_pixels);
  OptimizeBlitRoutine(image);
  delete[] image->locked_pixels;
}

////////////////////////////////////////////////////////////////////////////////

void EXPORT DirectBlit(int x, int y, int w, int h, RGBA* pixels, int method)
{
  if (method == 0)
    return;

  calculate_clipping_metrics(w, h);

  // direct copy (no alpha)
  if (method == 1)
  {

    if (PixelFormat == RGB565)
    {
      for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
        for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
        {
          ScreenBuffer[(y + iy) * ScreenWidth + x + ix] = PackPixel565(pixels[iy * w + ix]);
        }
    }
    else
    {
      for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
        for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
        {
          ScreenBuffer[(y + iy) * ScreenWidth + x + ix] = PackPixel555(pixels[iy * w + ix]);
        }
    }

  }
  
  // alpha blit
  else if (method == 2)
  {

    // FAST TRANSLUCENCY
    if (Configuration.fast_translucency)
    {
      if (PixelFormat == RGB565)
      {

        // 5:6:5
        word MASK = MASK565;

        for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
          for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
          {
            word* dest = ScreenBuffer + (y + iy) * ScreenWidth + (x + ix);
            word  src  = PackPixel555(pixels[iy * w + ix]);

            switch (pixels[iy * w + ix].alpha)
            {
              case 255:
                *dest = src;
                break;

              case 224:
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 192:
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 160:
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                src   = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 128:
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 96:
                src   = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 64:
                src   = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 32:
                src   = (*dest & MASK) / 2 + (src & MASK) / 2;
                src   = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;
            }
          }
      }
      else
      {

        // 5:5:5
        word MASK = MASK555;

        for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
          for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
          {
            word* dest = ScreenBuffer + (y + iy) * ScreenWidth + (x + ix);
            word  src  = PackPixel555(pixels[iy * w + ix]);

            switch (pixels[iy * w + ix].alpha)
            {
              case 255:
                *dest = src;
                break;

              case 224:
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 192:
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 160:
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                src   = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 128:
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 96:
                src   = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 64:
                src   = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;

              case 32:
                src   = (*dest & MASK) / 2 + (src & MASK) / 2;
                src   = (*dest & MASK) / 2 + (src & MASK) / 2;
                *dest = (*dest & MASK) / 2 + (src & MASK) / 2;
                break;
            }
          }

      }
    }

    // ACCURATE TRANSLUCENCY
    else
    {
      if (PixelFormat == RGB565)
      {
        // 5:6:5
        for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
          for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
          {
            word* dest  = ScreenBuffer + (y + iy) * ScreenWidth + (x + ix);
            int   alpha = pixels[iy * w + ix].alpha;
      
            RGBA out = UnpackPixel565(*dest);
            RGBA in  = pixels[iy * w + ix];
            out.red   = (in.red   * alpha + out.red   * (256 - alpha)) / 256;
            out.green = (in.green * alpha + out.green * (256 - alpha)) / 256;
            out.blue  = (in.blue  * alpha + out.blue  * (256 - alpha)) / 256;

            *dest = PackPixel565(out);
          }
      }
      else
      {
        // 5:5:5
        for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
          for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
          {
            word* dest  = ScreenBuffer + (y + iy) * ScreenWidth + (x + ix);
            int   alpha = pixels[iy * w + ix].alpha;
      
            RGBA out = UnpackPixel555(*dest);
            RGBA in  = pixels[iy * w + ix];
            out.red   = (in.red   * alpha + out.red   * (256 - alpha)) / 256;
            out.green = (in.green * alpha + out.green * (256 - alpha)) / 256;
            out.blue  = (in.blue  * alpha + out.blue  * (256 - alpha)) / 256;

            *dest = PackPixel555(out);
          }
      }
    }

  } // end if
}

////////////////////////////////////////////////////////////////////////////////

void EXPORT DirectGrab(int x, int y, int w, int h, RGBA* pixels)
{
  if (x < 0 ||
      y < 0 ||
      x + w > ScreenWidth ||
      y + h > ScreenHeight)
    return;

  // 5:6:5
  if (PixelFormat == RGB565)
  {
    for (int iy = 0; iy < h; iy++)
      for (int ix = 0; ix < w; ix++)
      {
        pixels[iy * w + ix]       = UnpackPixel565(ScreenBuffer[(y + iy) * ScreenWidth + x + ix]);
        pixels[iy * w + ix].alpha = 255;
      }
  }
  // 5:5:5
  else
  {
    for (int iy = 0; iy < h; iy++)
      for (int ix = 0; ix < w; ix++)
      {
        pixels[iy * w + ix]       = UnpackPixel555(ScreenBuffer[(y + iy) * ScreenHeight + x + x]);
        pixels[iy * w + ix].alpha = 255;
      }
  }
}

////////////////////////////////////////////////////////////////////////////////
