#include <windows.h>
#include <gl/gl.h>
#include <stdio.h>
#include <math.h>
#include "resource.h"
#include "colorspace.h"
#pragma warning (disable:4244)

#define EXPORT __stdcall


typedef struct tagIMAGE
{
  int width;
  int height;
	GLint name;
	GLfloat top;
	GLfloat bottom;
	GLfloat left;
	GLfloat right;
  RGBA* pixels;  // for locking
}* IMAGE;


typedef struct
{
  const char* name;
  const char* author;
  const char* date;
  const char* version;
  const char* description;
} DRIVERINFO;


typedef struct
{
	unsigned int bpp;
    bool scale;
	bool fullscreen;
	bool bilinear;
    bool stencil;
} DRIVERCONFIG;


#define SCALE() (DriverConfig.scale ? 2 : 1)


DRIVERCONFIG DriverConfig;


#ifdef _DEBUG
unsigned int frames = 0;
unsigned int start_time;
#endif

GLfloat texture_filtering;

////////////////////////////////////////////////////////////////////////////////

HINSTANCE DriverInstance;

BOOL WINAPI DllMain(HINSTANCE inst, DWORD reason, LPVOID reserved)
{
  DriverInstance = inst;
  return TRUE;
}

////////////////////////////////////////////////////////////////////////////////

void GetConfigFile(char configfile[520])
{
  GetModuleFileName(DriverInstance, configfile, 512);
  if (strrchr(configfile, '\\'))
  {
    *strrchr(configfile, '\\') = 0;
    strcat(configfile, "\\");
  }
  else
    configfile[0] = 0;
  strcat(configfile, "sphere_gl.cfg");
}

////////////////////////////////////////////////////////////////////////////////

void LoadDriverConfig(void)
{
	char configfile[520];

	GetConfigFile(configfile);

	DriverConfig.bpp = GetPrivateProfileInt("sphere_gl", "bpp", 16, configfile);
    DriverConfig.scale = GetPrivateProfileInt("sphere_gl", "scale", 0, configfile);
	DriverConfig.bilinear = GetPrivateProfileInt("sphere_gl", "bilinear", 1, configfile);
	DriverConfig.fullscreen = GetPrivateProfileInt("sphere_gl", "fullscreen", 0, configfile);
    DriverConfig.stencil = GetPrivateProfileInt("sphere_gl", "stencil", 0, configfile);
}

////////////////////////////////////////////////////////////////////////////////

void SaveDriverConfig(void)
{
	char configfile[520];
	char tempstr[255];

	GetConfigFile(configfile);

	sprintf(tempstr, "%u", DriverConfig.bpp);
	WritePrivateProfileString("sphere_gl", "bpp", tempstr, configfile);

	sprintf(tempstr, "%u", DriverConfig.scale);
	WritePrivateProfileString("sphere_gl", "scale", tempstr, configfile);

	sprintf(tempstr, "%u", DriverConfig.fullscreen);
	WritePrivateProfileString("sphere_gl", "fullscreen", tempstr, configfile);

	sprintf(tempstr, "%u", DriverConfig.bilinear);
	WritePrivateProfileString("sphere_gl", "bilinear", tempstr, configfile);
    
	sprintf(tempstr, "%u", DriverConfig.stencil);
	WritePrivateProfileString("sphere_gl", "stencil", tempstr, configfile);

}

////////////////////////////////////////////////////////////////////////////////

void EXPORT GetDriverInfo(DRIVERINFO* driverinfo)
{
	driverinfo->name        = "OpenGL";
	driverinfo->author      = "Jamie Gennis, Modified by Kisai";
	driverinfo->date        = __DATE__;
	driverinfo->version     = "0.22";
	driverinfo->description = "OpenGL Sphere Video Driver";
}

////////////////////////////////////////////////////////////////////////////////

BOOL CALLBACK ConfigureDriverDialogProc(HWND window, UINT msg, WPARAM wparam, LPARAM lparam);

void EXPORT ConfigureDriver(HWND parent)
{
  LoadDriverConfig();

  DialogBox(
    DriverInstance,
    MAKEINTRESOURCE(IDD_CONFIGURE),
    parent,
    ConfigureDriverDialogProc);
}

////////////////////////////////////////////////////////////////////////////////


BOOL CALLBACK ConfigureDriverDialogProc(HWND window, UINT msg, WPARAM wparam, LPARAM lparam)
{
	switch(msg)
	{
	case WM_INITDIALOG:
		switch(DriverConfig.bpp)
		{
		case 24:
			CheckDlgButton(window, IDC_BPP_24, BST_CHECKED);
			break;
		case 32:
			CheckDlgButton(window, IDC_BPP_32, BST_CHECKED);
			break;
		default:
			CheckDlgButton(window, IDC_BPP_16, BST_CHECKED);
			break;
		}
        if(DriverConfig.scale)
            CheckDlgButton(window, IDC_SCALE, BST_CHECKED);
		if(DriverConfig.bilinear)
			CheckDlgButton(window, IDC_BILINEAR, BST_CHECKED);
		if(DriverConfig.fullscreen)
			CheckDlgButton(window, IDC_FULLSCREEN, BST_CHECKED);
		if(DriverConfig.stencil)
			CheckDlgButton(window, IDC_STENCIL, BST_CHECKED);
			
		return 0;

		case WM_COMMAND:
			switch(LOWORD(wparam))
			{
			case IDOK:
				if(IsDlgButtonChecked(window, IDC_BPP_24) == BST_CHECKED)
					DriverConfig.bpp = 24;
				else if(IsDlgButtonChecked(window, IDC_BPP_32) == BST_CHECKED)
					DriverConfig.bpp = 32;
				else
					DriverConfig.bpp = 16;

                DriverConfig.scale = (IsDlgButtonChecked(window, IDC_SCALE) == BST_CHECKED);
				DriverConfig.bilinear = (IsDlgButtonChecked(window, IDC_BILINEAR) == BST_CHECKED);
				DriverConfig.fullscreen = (IsDlgButtonChecked(window, IDC_FULLSCREEN) == BST_CHECKED);
				DriverConfig.stencil = (IsDlgButtonChecked(window, IDC_STENCIL) == BST_CHECKED);

				SaveDriverConfig();
				EndDialog(window, 0);
				return 0;

			case IDCANCEL:
				EndDialog(window, 0);
				return 0;
			}
		default:
			return 0;
	}
}

////////////////////////////////////////////////////////////////////////////////

HWND hwnd;
HDC  hdc;
HGLRC hrc;
DWORD WindowStyle;
DWORD ExWindowStyle;
DEVMODE PrevMode;
LRESULT(CALLBACK *WindwProc)(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam);

LRESULT CALLBACK MyWinProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uMsg)
	{
	case WM_MOUSEMOVE:
		SetCursor(NULL);
	}
	return WindwProc(hWnd, uMsg, wParam, lParam);
}

////////////////////////////////////////////////////////////////////////////////

int ScreenWidth, ScreenHeight;

bool EXPORT InitVideoDriver(HWND window, int screen_width, int screen_height)
{
	int screenwidth = GetSystemMetrics(SM_CXSCREEN);
	int screenheight = GetSystemMetrics(SM_CYSCREEN);
	int Hz;
	int format;
	PIXELFORMATDESCRIPTOR pfd =
	{ 
		sizeof(PIXELFORMATDESCRIPTOR),	// size of this pfd 
		1,								// version number 
		PFD_DRAW_TO_WINDOW |			// support window 
		PFD_SUPPORT_OPENGL |			// support OpenGL 
		//PFD_GENERIC_ACCELERATED |
		PFD_DOUBLEBUFFER,				// double buffered
		PFD_TYPE_RGBA,					// RGBA type 
		0,								// color depth 
		0, 0, 0, 0, 0, 0,				// color bits
		0,								// alpha buffer 
		0,								// shift bit
		0,								// accumulation buffer 
		0, 0, 0, 0,						// accum bits
		0,								// z-buffer 
		8,								// stencil buffer 
		0,								// auxiliary buffer 
		PFD_MAIN_PLANE,					// main layer 
		0,								// reserved 
		0, 0, 0							// layer masks ignored 
	};

  ScreenWidth = screen_width;
  ScreenHeight = screen_height;

	LoadDriverConfig();

	hwnd = window;
	WindowStyle = GetWindowLong(hwnd, GWL_STYLE);
	ExWindowStyle = GetWindowLong(hwnd, GWL_EXSTYLE);
	
		PrevMode.dmSize = sizeof(DEVMODE);
		PrevMode.dmDriverExtra = 0;
		EnumDisplaySettings(NULL, ENUM_CURRENT_SETTINGS, &PrevMode);
		Hz = PrevMode.dmDisplayFrequency;

	if(!DriverConfig.fullscreen)
	{
		RECT rect = { 0, 0, ScreenWidth * SCALE(), ScreenHeight * SCALE() };
		int winwidth, winheight;
		SetWindowLong(hwnd, GWL_STYLE, WS_POPUP | WS_CAPTION | WS_CLIPSIBLINGS);
		SetWindowLong(hwnd, GWL_EXSTYLE, 0);
		AdjustWindowRect(&rect, WS_BORDER | WS_DLGFRAME | WS_CLIPSIBLINGS, (GetMenu(hwnd) ? TRUE : FALSE));
		winwidth = rect.right - rect.left;
		winheight = rect.bottom - rect.top;
		SetWindowPos(hwnd, HWND_TOP,
			(screenwidth - winwidth) / 2,
			(screenheight - winheight) / 2,
			winwidth, winheight, SWP_SHOWWINDOW);
	}
	else
	{
		int mode = 0;
		DEVMODE DevMode;
	
		// Find display mode
		DevMode.dmSize = sizeof(DEVMODE);
		DevMode.dmDriverExtra = 0;
		while(1)
		{
			if(!EnumDisplaySettings(NULL, mode, &DevMode))
			{
				MessageBox(hwnd, "Unable to find correct display settings.", "Video Error", MB_ICONERROR);
				return false;
			}
			if(DevMode.dmBitsPerPel == DriverConfig.bpp &&
				DevMode.dmPelsWidth == (unsigned)(ScreenWidth * SCALE()) &&
				DevMode.dmPelsHeight == (unsigned)(ScreenHeight * SCALE()))
				break;
			mode++;
		}
		DevMode.dmFields = DM_BITSPERPEL
			| DM_PELSWIDTH
			| DM_PELSHEIGHT;
		if(ChangeDisplaySettings(&DevMode, CDS_FULLSCREEN) != DISP_CHANGE_SUCCESSFUL)
		{
			MessageBox(hwnd, "Unable to set display mode.", "Video Error", MB_ICONERROR);
			return false;
		}

		// Set up window
		SetWindowLong(hwnd, GWL_STYLE, WS_POPUP | WS_CLIPSIBLINGS);
		SetWindowLong(hwnd, GWL_EXSTYLE, 0);
		SetWindowPos(hwnd, HWND_TOPMOST, 0, 0,
			ScreenWidth * SCALE(), ScreenHeight * SCALE(), SWP_SHOWWINDOW);

		// Set up my window proc to hide the mouse
		WindwProc = (LRESULT(CALLBACK*)(HWND, UINT, WPARAM, LPARAM)) GetWindowLong(hwnd, GWL_WNDPROC);
		SetWindowLong(hwnd, GWL_WNDPROC, (LONG)MyWinProc);
	}
	
	// Get the DC of the window
	hdc = GetDC(hwnd);
	if(!hdc)
	{
		MessageBox(hwnd, "Error getting window DC.", "Video Error", MB_ICONERROR);
		return false;
	}

	// Set the pfd
	pfd.cColorBits = DriverConfig.bpp;
	format = ChoosePixelFormat(hdc, &pfd);
	if(!SetPixelFormat(hdc, format, &pfd))
	{
		MessageBox(hwnd, "Error setting pfd.", "Video Error", MB_ICONERROR);
		return false;
	}

	// Create Render Context
	hrc = wglCreateContext(hdc);
	if(!hrc)
	{
		MessageBox(hwnd, "Error creating render context.", "Video Error", MB_ICONERROR);
		return false;
	}

	// Make context current
	if(!wglMakeCurrent(hdc, hrc))
	{
		MessageBox(hwnd, "Unable to make render context current.", "Video Error", MB_ICONERROR);
		return false;
	}
glPixelStorei(GL_PACK_ALIGNMENT,1);
glPixelStorei(GL_UNPACK_ALIGNMENT,1);



	// Enable stencil buffer
	if(DriverConfig.stencil)
	{
	glClearStencil(0x0);
	glEnable(GL_STENCIL_TEST);
	}

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, ScreenWidth, 0, ScreenHeight, -1, 1);
	glMatrixMode(GL_MODELVIEW);

	if(DriverConfig.bilinear)
		texture_filtering = (float)GL_LINEAR;
	else
		texture_filtering = (float)GL_NEAREST;

	

#ifdef _DEBUG
	start_time = timeGetTime();
#endif

	return true;
}

////////////////////////////////////////////////////////////////////////////////
bool EXPORT CloseVideoDriver(void)
{
	// Make context not current
	wglMakeCurrent(NULL, NULL);


	// Release DC
	ReleaseDC(hwnd, hdc);
    
 	// Delete Context
	wglDeleteContext(hrc);
   

	// Change display mode back
	if(DriverConfig.fullscreen)
	{
		PrevMode.dmFields = DM_BITSPERPEL
			| DM_PELSWIDTH
			| DM_PELSHEIGHT
			| DM_DISPLAYFREQUENCY;
		ChangeDisplaySettings(&PrevMode, 0);

		// restore window proc
		SetWindowLong(hwnd, GWL_WNDPROC, (LONG)WindwProc);
	}

	// Restore window styles
	SetWindowLong(hwnd, GWL_STYLE, WindowStyle);
	SetWindowLong(hwnd, GWL_EXSTYLE, ExWindowStyle);


#ifdef _DEBUG
	{
		char fps_str[512];
		unsigned int curr_time;
		double elapsed_time, FPS;
		curr_time = timeGetTime();
		if(curr_time < start_time)
		{
			curr_time = 0xFFFFFFFF - start_time + curr_time;
			start_time = 0;
		}
		elapsed_time = curr_time - start_time;
		FPS = (double)frames / (elapsed_time/1000);
		sprintf(fps_str, "FPS: %f", FPS);
		MessageBox(hwnd, fps_str, "Frames Per Second", MB_OK);
	}
#endif

	return true;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT FlipScreen(void)
{
#ifdef _DEBUG
	frames++;
#endif
	SwapBuffers(hdc);
	glClear(GL_COLOR_BUFFER_BIT);
	return true;
}

////////////////////////////////////////////////////////////////////////////////

RECT ClippingRectangle;

bool EXPORT SetClippingRectangle(int x, int y, int w, int h)
{
	GLint left = x,
		right = x+w,
		top = ScreenHeight-y,
		bottom = top - h;
	glClear(GL_STENCIL_BUFFER_BIT);
	glStencilFunc(GL_ALWAYS, 0x1, 0x1);
	glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);
	glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
	glBegin(GL_QUADS);
		glVertex2i(0,0);
		glVertex2i(left,0);
		glVertex2i(left,ScreenHeight);
		glVertex2i(0,ScreenHeight);

		glVertex2i(0,0);
		glVertex2i(0,bottom);
		glVertex2i(ScreenWidth,bottom);
		glVertex2i(ScreenWidth,0);

		glVertex2i(ScreenWidth,ScreenHeight);
		glVertex2i(right,ScreenHeight);
		glVertex2i(right,0);
		glVertex2i(ScreenWidth,0);

		glVertex2i(ScreenWidth,ScreenHeight);
		glVertex2i(ScreenWidth,top);
		glVertex2i(0,top);
		glVertex2i(0,ScreenHeight);
	glEnd();
	glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	glStencilFunc(GL_NOTEQUAL, 0x1, 0x1);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

  SetRect(&ClippingRectangle, x, y, x + w, y + h);

	return true;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT GetClippingRectangle(int* x, int* y, int* w, int* h)
{
  *x = ClippingRectangle.left;
  *y = ClippingRectangle.top;
  *w = ClippingRectangle.right - ClippingRectangle.left;
  *h = ClippingRectangle.bottom - ClippingRectangle.top;
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT ApplyColorMask(RGBA mask)
{
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_BLEND);
	glBegin(GL_QUADS);
	glColor4ubv((GLubyte*)&mask);
	glVertex2i(0,0);
	glVertex2i(0,ScreenHeight);
	glVertex2i(ScreenWidth,ScreenHeight);
	glVertex2i(ScreenWidth,0);
	glEnd();
	return true;
}

////////////////////////////////////////////////////////////////////////////////

void scale_bitmap(RGBA* dest, RGBA *src, int dest_w, int dest_h, int src_w, int src_h)
{
	int j;
	memset(dest, 0, dest_h * dest_w * sizeof(RGBA));
	for(j=0;j<src_h;j++)
		memcpy(dest + (j*dest_w), src + (j*src_w), src_w * sizeof(RGBA));
}

////////////////////////////////////////////////////////////////////////////////

IMAGE EXPORT CreateImage(int width, int height, RGBA* data)
{
	RGBA* tmp;
	double log2_width, log2_height;

	IMAGE image = malloc(sizeof(struct tagIMAGE));

	image->width = width;
	image->height = height;
  image->pixels = malloc(width * height * sizeof(RGBA));
  memcpy(image->pixels, data, width * height * sizeof(RGBA));

	log2_width = log10(width)/log10(2);
	log2_height = log10(height)/log10(2);

	if(log2_width != floor(log2_width))
	{
		width = 1<<(int)(ceil(log2_width));
	}
	if(log2_height != floor(log2_height))
	{
		height = 1<<(int)(ceil(log2_height));
	}

	if(image->width != width || image->height != height)
	{
		tmp = malloc(width*height*sizeof(RGBA));
		scale_bitmap(tmp, data, width, height, image->width, image->height);
	}
	else
		tmp = data;

	glGenTextures(1, &image->name);
	glBindTexture(GL_TEXTURE_2D, image->name);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, texture_filtering);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, texture_filtering);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, width, height,
		0, GL_RGBA, GL_UNSIGNED_BYTE, tmp);
	// it's flipped
	image->top = 0;
	image->bottom = (float)image->height/(float)height;
	image->left = 0;
	image->right = (float)image->width/(float)width;

	if(tmp != data)
		free(tmp);
	
	return image;
}

////////////////////////////////////////////////////////////////////////////////

IMAGE EXPORT GrabImage(int x, int y, int width, int height)
{
	RGBA* tmp;
  IMAGE ret;
	int orig_width = width,
		orig_height = height;
	y = ScreenHeight-y;

	width = (int)((float)width * SCALE());
	height = (int)((float)height * SCALE());
	x = (int)((float)x * SCALE());
	y = (int)((float)y * SCALE());

	tmp = (RGBA*)malloc(width * height * sizeof(RGBA));
	glReadPixels(x,y-height,
		width, height,
		GL_RGBA, GL_UNSIGNED_BYTE,
		tmp);

	ret = CreateImage(width, height, tmp);
	ret->width = orig_width;
	ret->height = orig_height;
	// It's not flipped
	
	ret->top = ret->bottom;
	ret->bottom = 0;
  
  free(tmp);
	return ret;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT DestroyImage(IMAGE image)
{
	if(!glIsTexture(image->name))
		return false;

	glDeleteTextures(1, &image->name);
  free(image->pixels);
	free(image);
	return true;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT BlitImage(IMAGE image, int x, int y)
{
	int tex_top = 0, tex_bot = 1;
	y = ScreenHeight-y-image->height;

	if(!glIsTexture(image->name))
		return false;

	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_BLEND);
	glBindTexture(GL_TEXTURE_2D, image->name);
	glEnable(GL_TEXTURE_2D);
	
	glBegin(GL_QUADS);
	glColor4f(1,1,1,1 );
	glTexCoord2f(image->left,image->bottom);
	glVertex2i(x, y);
	glTexCoord2f(image->right,image->bottom);
	glVertex2i(x + image->width, y);
	glTexCoord2f(image->right,image->top);
	glVertex2i(x + image->width, y + image->height);
	glTexCoord2f(image->left,image->top);
	glVertex2i(x, y + image->height);
	glEnd();
	glDisable(GL_TEXTURE_2D);
	
	return true;
}

////////////////////////////////////////////////////////////////////////////////

int EXPORT GetImageWidth(IMAGE image)
{
  return image->width;
}

////////////////////////////////////////////////////////////////////////////////

int EXPORT GetImageHeight(IMAGE image)
{
  return image->height;
}

////////////////////////////////////////////////////////////////////////////////

RGBA* EXPORT LockImage(IMAGE image)
{
  return image->pixels;
}

////////////////////////////////////////////////////////////////////////////////

void EXPORT UnlockImage(IMAGE image)
{
  double log2_width;
  double log2_height;
  int width;
  int height;
  RGBA* tmp;

	// destroy the old texture
  glDeleteTextures(1, &image->name);

  // create the new stuff

	width = image->width;
  height = image->height;
  log2_width = log10(width)/log10(2);
	log2_height = log10(height)/log10(2);

	if(log2_width != floor(log2_width))
	{
		width = 1<<(int)(ceil(log2_width));
	}
	if(log2_height != floor(log2_height))
	{
		height = 1<<(int)(ceil(log2_height));
	}

	if(image->width != width || image->height != height)
	{
		tmp = malloc(width*height*sizeof(RGBA));
		scale_bitmap(tmp, image->pixels, width, height, image->width, image->height);
	}
	else
		tmp = image->pixels;

	glGenTextures(1, &image->name);
	glBindTexture(GL_TEXTURE_2D, image->name);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, texture_filtering);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, texture_filtering);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, width, height,
		0, GL_RGBA, GL_UNSIGNED_BYTE, tmp);

  if (tmp != image->pixels)
    free(image->pixels);
}

////////////////////////////////////////////////////////////////////////////////
void EXPORT DirectBlit(int x, int y, int w, int h, RGBA* pixels, int method)
{
 int i;
	RGBA* new_pixels = (RGBA*)malloc(sizeof(RGBA) * w * h);

	for (i = 0; i < h; i++) 
	{
    memcpy(new_pixels + i * w, pixels + (h - i - 1) * w, w * sizeof(RGBA));
	}	
	if (method == 0)
    return;

  if (method == 1)
  {
    glPixelZoom(SCALE(),SCALE());
	glRasterPos2i( x, y);

	  glDrawPixels(	w,
				h,
				GL_RGBA,
				GL_UNSIGNED_BYTE, 
				new_pixels);
	  
  
  }
  else if (method == 2)
  {
    glPixelZoom(SCALE(),SCALE());

	  glRasterPos2i( x, y );
	  glDrawPixels(	w,
				h,
				GL_RGBA,
				GL_UNSIGNED_BYTE, 
				pixels);
 
    
  } // end if
free(new_pixels);
}

////////////////////////////////////////////////////////////////////////////////

void EXPORT DirectGrab(int x, int y, int w, int h, RGBA* pixels)
{
  int i;
	RGBA* new_pixels = (RGBA*)malloc(sizeof(RGBA) * w * h);

	
	if (x < 0 ||
      y < 0 ||
      x + w > ScreenWidth ||
      y + h > ScreenHeight)
    return;

  glReadPixels(x,
				y,
				w,
				h,
				GL_RGBA,
				GL_UNSIGNED_BYTE,
				new_pixels);

for (i = 0; i < h; i++) 
	{
    memcpy(pixels + i * w, new_pixels + (h - i - 1) * w, w * sizeof(RGBA));
	}	
	

}

////////////////////////////////////////////////////////////////////////////////


