#include <windows.h>
#include <mmsystem.h>
#include <stdio.h>
#include <stdlib.h>
#include "audio.h"
#include "internal.h"
#include "fmod.h"


#define ENABLE_SOUNDEFFECTS
#define ENABLE_MUSICMODULES
//#define ENABLE_DIRECTMUSIC

#ifdef ENABLE_DIRECTMUSIC
  #include <dmusici.h>
  #include <dmusicc.h>
#endif

enum MusicType { MUSIC_MP3, MUSIC_WAV, MUSIC_MOD, MUSIC_MIDI };


typedef struct
{
  int            volume;
  long           channel;
  FSOUND_SAMPLE* sample;
} _SOUNDEFFECT;


typedef struct
{
  MusicType      musictype;
  FSOUND_STREAM* stream;
  FMUSIC_MODULE* module;
  int            volume;
  long           streamchannel;
  bool           is_playing;
  #ifdef ENABLE_DIRECTMUSIC
    IDirectMusic            *directmusic;
    IDirectMusicLoader      *directmusicloader;
    IDirectMusicSegment     *directmusicsegment;
    IDirectMusicSegmentState *directmusicsegmentstate;
    IDirectMusicPerformance *directmusicperformance;
  #endif
} _MUSICMODULE;


static HWND  SphereWindow;
static bool  Stereo;
static dword BaseMode;
static bool  AudioEnabled;
static int   GlobalMusicVolume;


////////////////////////////////////////////////////////////////////////////////

bool InitAudio(HWND window, SPHERECONFIG* config)
{
  int output_driver;
  int mixer;

  SphereWindow = window;

  // set output type
  output_driver = FSOUND_OUTPUT_NOSOUND;
  switch (config->output_driver)
  {
    case 0: output_driver = FSOUND_OUTPUT_NOSOUND; break;
    case 1: output_driver = FSOUND_OUTPUT_WINMM;   break;
    case 2: output_driver = FSOUND_OUTPUT_DSOUND;  break;
  }

retry:

  FSOUND_SetOutput(output_driver);

  // set mixer
  mixer = FSOUND_MIXER_AUTODETECT;
  switch (config->mixer)
  {
    case 0: mixer = FSOUND_MIXER_AUTODETECT; break;
    case 1: mixer = FSOUND_MIXER_QUALITY_AUTODETECT; break;
    case 2: mixer = FSOUND_MIXER_BLENDMODE; break;
    case 3: mixer = FSOUND_MIXER_MMXP5; break;
    case 4: mixer = FSOUND_MIXER_MMXP6; break;
    case 5: mixer = FSOUND_MIXER_QUALITY_FPU; break;
    case 6: mixer = FSOUND_MIXER_QUALITY_MMXP5; break;
    case 7: mixer = FSOUND_MIXER_QUALITY_MMXP6; break;
  }
  FSOUND_SetMixer(mixer);

  // initialize audio
  if (!FSOUND_Init(config->bit_rate, 64, 0))
  {
    // try reinitialization with no sound
    if (output_driver != FSOUND_OUTPUT_NOSOUND)
    {
      output_driver = FSOUND_OUTPUT_NOSOUND;
      goto retry;  // I know...  :)
    }

    return false;
  }

  // set up master music and sound volumes
  FSOUND_SetSFXMasterVolume(config->sound_volume);
  GlobalMusicVolume = config->music_volume;

  // calculate modes for MP3 streaming
  Stereo = config->stereo ? true : false;
  BaseMode = FSOUND_2D | FSOUND_UNSIGNED;
  BaseMode |= (config->stereo ? FSOUND_STEREO : FSOUND_MONO);
  BaseMode |= (config->bit_rate == 16 ? FSOUND_16BITS : FSOUND_8BITS);

  AudioEnabled = (output_driver != FSOUND_OUTPUT_NOSOUND);

  #ifdef ENABLE_DIRECTMUSIC
    if (CoInitialize(NULL) < 0)
      return false;
  #else
    mciSendString("set time format milliseconds", NULL, NULL, NULL);
  #endif
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool CloseAudio(void)
{
  FSOUND_Close();
  #ifdef ENABLE_DIRECTMUSIC
    CoUninitialize();
  #endif
  return true;
}

////////////////////////////////////////////////////////////////////////////////

SOUNDEFFECT LoadSoundEffect(const char* _filename)
{
#ifdef ENABLE_SOUNDEFFECTS

  char filename[MAX_PATH];
  _SOUNDEFFECT* _se = (_SOUNDEFFECT*)malloc(sizeof(_SOUNDEFFECT));

  strcpy(filename, _filename);

  if (stricmp(filename + strlen(filename) - 4, ".wav") == 0)
    _se->sample = FSOUND_Sample_LoadWav(FSOUND_FREE, filename, FSOUND_NORMAL);
  else if (stricmp(filename + strlen(filename) - 4, ".raw") == 0)
    _se->sample = FSOUND_Sample_LoadRaw(FSOUND_FREE, filename, FSOUND_NORMAL);
  else
  {
    free(_se);
    return NULL;
  }

  if (_se->sample == NULL)
  {
    free(_se);
    return NULL;
  }

  _se->volume = 255;
  return (SOUNDEFFECT)_se;

#else

  return (SOUNDEFFECT)1;

#endif
}

////////////////////////////////////////////////////////////////////////////////

bool DestroySoundEffect(SOUNDEFFECT se)
{
#ifdef ENABLE_SOUNDEFFECTS

  _SOUNDEFFECT* _se = (_SOUNDEFFECT*)se;

  StopSoundEffect(se);
  FSOUND_Sample_Free(_se->sample);
  free(_se);

#endif

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool PlaySoundEffect(SOUNDEFFECT se)
{
#ifdef ENABLE_SOUNDEFFECTS

  _SOUNDEFFECT* _se = (_SOUNDEFFECT*)se;
  _se->channel = FSOUND_PlaySound(FSOUND_FREE, _se->sample);

#endif

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool StopSoundEffect(SOUNDEFFECT se)
{
#ifdef ENABLE_SOUNDEFFECTS

  _SOUNDEFFECT* _se = (_SOUNDEFFECT*)se;

  if (FSOUND_IsPlaying(_se->channel))
    FSOUND_Sample_Free(_se->sample);

#endif

  return true;
}

////////////////////////////////////////////////////////////////////////////////

MUSICMODULE LoadMusicModule(const char* filename)
{
  _MUSICMODULE* _mm = (_MUSICMODULE*)malloc(sizeof(_MUSICMODULE));

  // If file is a MIDI
  if (stricmp(filename + strlen(filename) - 4, ".mid") == 0 ||
      stricmp(filename + strlen(filename) - 5, ".midi") == 0)
  {
    #ifdef ENABLE_DIRECTMUSIC
      
      if (FAILED(CoCreateInstance(CLSID_DirectMusicPerformance, NULL, CLSCTX_INPROC, IID_IDirectMusicPerformance2, (void**)&_mm->directmusicperformance)))
        return false;

      if (FAILED(CoCreateInstance(CLSID_DirectMusicLoader, NULL, CLSCTX_INPROC, IID_IDirectMusicLoader, (void**)&_mm->directmusicloader)))
      {
        _mm->directmusicperformance->Release();
        return false;
      }

      DMUS_OBJECTDESC ObjDesc;
      WCHAR wcFilename[MAX_PATH];

      ObjDesc.dwSize = sizeof(DMUS_OBJECTDESC);
      ObjDesc.dwValidData = DMUS_OBJ_CLASS | DMUS_OBJ_FILENAME | DMUS_OBJ_FULLPATH;
      ObjDesc.guidClass = CLSID_DirectMusicSegment;
      MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED, filename, -1, wcFilename, MAX_PATH);
      wcscpy(ObjDesc.wszFileName, wcFilename);

      //MessageBox(NULL, "Got to here", "Hah", MB_OK);

      if (FAILED(_mm->directmusicloader->GetObject(&ObjDesc, IID_IDirectMusicSegment2, (void**)&_mm->directmusicsegment)))
      {
        _mm->directmusicloader->Release();
        _mm->directmusicperformance->Release();
        return false;
      }

      //MessageBox(NULL, "Got to here", "Hah", MB_OK);
      _mm->directmusic = NULL;
      _mm->directmusicperformance->Init(&_mm->directmusic, NULL, NULL);
      _mm->directmusicperformance->AddPort(NULL);
      _mm->directmusicsegment->SetParam(GUID_StandardMIDIFile, -1, 0, 0, (void*)_mm->directmusicperformance);
      _mm->directmusicsegment->SetParam(GUID_Download, -1, 0, 0, (void*)_mm->directmusicperformance);
      _mm->directmusicsegment->SetRepeats(-1);
      _mm->directmusicperformance->PlaySegment(_mm->directmusicsegment, 0, 0, &_mm->directmusicsegmentstate);
    #else
      char temp[520];
      char ret[520];

      sprintf(temp, "open \"%s\" type sequencer alias midifile", filename);
      if (mciSendString(temp, ret, 512, NULL) != 0) // if it failed
      {
        free(_mm);
        return false;
      }

      UpdateSystem();
    #endif

    _mm->musictype = MUSIC_MIDI;
  }
  else if (stricmp(filename + strlen(filename) - 4, ".s3m") == 0 ||
           stricmp(filename + strlen(filename) - 4, ".mod") == 0 ||
           stricmp(filename + strlen(filename) - 3, ".it") == 0  ||
           stricmp(filename + strlen(filename) - 3, ".xm") == 0)
  {
    _mm->module = FMUSIC_LoadSong((char*)filename);
    if (_mm->module == NULL)
    {
      free(_mm);
      return false;
    }
    _mm->musictype = MUSIC_MOD;
  }
  else if (stricmp(filename + strlen(filename) - 4, ".wav") == 0)
  {
    _mm->stream = FSOUND_Stream_OpenWav((char*)filename, FSOUND_NORMAL | FSOUND_STREAMABLE | FSOUND_LOOP_NORMAL);
    if (_mm->stream == NULL)
    {
      free(_mm);
      return false;
    }
    _mm->musictype = MUSIC_WAV;
  }
  else // mp3
  {
    _mm->stream = FSOUND_Stream_OpenMpeg((char*)filename, FSOUND_NORMAL | FSOUND_STREAMABLE | FSOUND_LOOP_NORMAL);
    if (_mm->stream == NULL)
    {
      free(_mm);
      return false;
    }
    _mm->musictype = MUSIC_MP3;
  }

  _mm->volume = 255;
  _mm->is_playing = false;

  return (MUSICMODULE)_mm;
}

////////////////////////////////////////////////////////////////////////////////

bool DestroyMusicModule(MUSICMODULE mm)
{
  _MUSICMODULE* _mm = (_MUSICMODULE*)mm;

  StopMusicModule(mm);

  if (_mm->musictype == MUSIC_MIDI)
  {
    #ifdef ENABLE_DIRECTMUSIC
      _mm->directmusicperformance->Stop(NULL, NULL, 0, 0);
      _mm->directmusicsegmentstate->Release();
      _mm->directmusicsegment->Release();
      _mm->directmusicperformance->CloseDown();
      _mm->directmusicperformance->Release();
      _mm->directmusicloader->Release();
      _mm->directmusic->Release();
    #else
      char ret[520];
      mciSendString("close midifile", ret, 512, NULL);
      UpdateSystem();
    #endif
  }
  else if (_mm->musictype == MUSIC_MOD)
  {
    FMUSIC_FreeSong(_mm->module);
  }
  else // mp3
  {
    FSOUND_Stream_Close(_mm->stream);
  }

  free(_mm);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool StartMusicModule(MUSICMODULE mm)
{
  _MUSICMODULE* _mm = (_MUSICMODULE*)mm;

  if (AudioEnabled == false)
    return true;

  if (_mm->musictype == MUSIC_MIDI)
  {
    #ifdef ENABLE_DIRECTMUSIC
      //_mm->directmusicperformance->PlaySegment(_mm->directmusicsegment, 0, 0, NULL);
    #else
      char ret[512];
      mciSendString("seek midifile to start", ret, 500, SphereWindow);
      mciSendString("play midifile notify", ret, 500, SphereWindow);
      UpdateSystem();
    #endif
  }
  else if (_mm->musictype == MUSIC_MOD)
  {
    FMUSIC_PlaySong(_mm->module);
  }
  else
  {
    _mm->streamchannel = FSOUND_Stream_Play(FSOUND_FREE, _mm->stream);
  }

  _mm->is_playing = true;
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool StopMusicModule(MUSICMODULE mm)
{
  _MUSICMODULE* _mm = (_MUSICMODULE*)mm;

  if (AudioEnabled == false)
    return true;

  if (_mm->is_playing)
  {
    if (_mm->musictype == MUSIC_MIDI)
    {
      #ifdef ENABLE_DIRECTMUSIC
        _mm->directmusicperformance->Stop(NULL, NULL, 0, 0);
      #else
        char ret[512];
        mciSendString("stop midifile", ret, 500, NULL);
        mciSendString("seek midifile to start", ret, 500, NULL);
        UpdateSystem();
      #endif
    }
    else if (_mm->musictype == MUSIC_MOD)
    {
      FMUSIC_StopSong(_mm->module);
    }
    else
    {
      FSOUND_Stream_Stop(_mm->stream);
    }
    _mm->is_playing = false;
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

int GetMusicTime(MUSICMODULE mm)
{
  _MUSICMODULE* _mm = (_MUSICMODULE*)mm;
  if (_mm->musictype == MUSIC_MOD)
    return FMUSIC_GetTime(_mm->module);
  else if (_mm->musictype == MUSIC_MP3)
    return FSOUND_Stream_GetTime(_mm->stream);
  else if (_mm->musictype == MUSIC_MIDI)
  {
    #ifdef ENABLE_DIRECTMUSIC
      MUSIC_TIME *theTime;
      theTime = new MUSIC_TIME;
      _mm->directmusicsegmentstate->GetSeek(theTime);
      return *theTime;
    #else
      char buf[80];
      mciSendString("status midifile position", buf, 80, NULL);
      return atoi(buf);
    #endif
  }

  return 1;
}

////////////////////////////////////////////////////////////////////////////////

bool SetGlobalSoundVolume(int volume)
{
  if (AudioEnabled == false)
    return true;

  FSOUND_SetSFXMasterVolume(volume);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

int GetGlobalSoundVolume()
{
  if (AudioEnabled == false)
    return 0;

  return FSOUND_GetSFXMasterVolume();
}

////////////////////////////////////////////////////////////////////////////////

bool SetGlobalMusicVolume(int volume)
{
  if (AudioEnabled == false)
    return true;

  GlobalMusicVolume = volume;
  return true;
}

////////////////////////////////////////////////////////////////////////////////

int GetGlobalMusicVolume()
{
  if (AudioEnabled == false)
    return 0;

  return GlobalMusicVolume;
}

////////////////////////////////////////////////////////////////////////////////

bool SetSoundVolumeAbsolute(SOUNDEFFECT se, int volume)
{
  if (AudioEnabled == false)
    return true;
  
  _SOUNDEFFECT* _se = (_SOUNDEFFECT*)se;
  _se->volume = volume;
  FSOUND_SetVolumeAbsolute(_se->channel, _se->volume);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool SetSoundVolume(SOUNDEFFECT se, int volume)
{
  if (AudioEnabled == false)
    return true;
  
  _SOUNDEFFECT* _se = (_SOUNDEFFECT*)se;
  _se->volume = volume;
  FSOUND_SetVolume(_se->channel, _se->volume);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

int GetSoundVolume(SOUNDEFFECT se)
{
  if (AudioEnabled == false)
    return 0;

  _SOUNDEFFECT* _se = (_SOUNDEFFECT*)se;

  return FSOUND_GetVolume(_se->channel);
}

////////////////////////////////////////////////////////////////////////////////

bool SetMusicVolume(MUSICMODULE mm, int volume)
{
  if (AudioEnabled == false)
    return true;

  _MUSICMODULE* _mm = (_MUSICMODULE*)mm;
  _mm->volume = volume;
  FMUSIC_SetMasterVolume(_mm->module, _mm->volume);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

int GetMusicVolume(MUSICMODULE mm)
{
  if (AudioEnabled == false)
    return false;

  _MUSICMODULE* _mm = (_MUSICMODULE*)mm;

  return FMUSIC_GetMasterVolume(_mm->module);
}

////////////////////////////////////////////////////////////////////////////////
