#include <stddef.h>
#include "smap.hpp"


////////////////////////////////////////////////////////////////////////////////

SMAP::SMAP()
: m_DelayMap(NULL)
, m_Tiles(NULL)
{
}

////////////////////////////////////////////////////////////////////////////////

SMAP::~SMAP()
{
  DeleteMap();

  for (int i = 0; i < m_Map.GetTileset().GetNumTiles(); i++)
    DestroyImage(m_Tiles[i]);
  delete[] m_Tiles;
}

////////////////////////////////////////////////////////////////////////////////

bool
SMAP::Load(const char* filename)
{
  DeleteMap();

  if (!m_Map.Load(filename))
    return false;

  if (m_Map.GetTileset().GetNumTiles() == 0)
    return false;

  // initialize max delay map to 0
  CreateMap();
  SetMap(m_Map.GetTileset());

  m_Tiles = new IMAGE[m_Map.GetTileset().GetNumTiles()];
  for (int i = 0; i < m_Map.GetTileset().GetNumTiles(); i++)
  {
    sTile& tile = m_Map.GetTileset().GetTile(i);
    m_Tiles[i] = CreateImage(tile.GetWidth(), tile.GetHeight(), tile.GetPixels());
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

void
SMAP::UpdateAnimations()
{
  sTileset& tileset = m_Map.GetTileset();

  // go through each tile and update the timer on it
  // if the tile should be switched, it is
  for (int il = 0; il < m_Map.GetNumLayers(); il++)
  {
    sLayer& layer = m_Map.GetLayer(il);
    for (int iy = 0; iy < layer.GetHeight(); iy++)
      for (int ix = 0; ix < layer.GetWidth(); ix++)
      {
        int oldtile = layer.GetTile(ix, iy);
        if (tileset.GetTile(oldtile).IsAnimated())
        {
          int offset = iy * layer.GetWidth() + ix;
          if (--m_DelayMap[il][offset] <= 0)
          {
            int newtile = tileset.GetTile(oldtile).GetNextTile();

            // change the map's tile and update the delay map
            layer.SetTile(ix, iy, newtile);
            m_DelayMap[il][offset] = tileset.GetTile(newtile).GetDelay();
          }
        }
      }
  }
}

////////////////////////////////////////////////////////////////////////////////

IMAGE
SMAP::GetTileImage(int tile)
{
  return m_Tiles[tile];
}

////////////////////////////////////////////////////////////////////////////////

sMap&
SMAP::GetMap()
{
  return m_Map;
}

////////////////////////////////////////////////////////////////////////////////

sTileset&
SMAP::GetTileset()
{
  return m_Map.GetTileset();
}

////////////////////////////////////////////////////////////////////////////////

void
SMAP::CreateMap()
{
  // initialize delay map to 0
  m_DelayMap = new int*[m_Map.GetNumLayers()];
  for (int i = 0; i < m_Map.GetNumLayers(); i++)
  {
    const sLayer& layer = m_Map.GetLayer(i);
    m_DelayMap[i] = new int[layer.GetWidth() * layer.GetHeight()];

    for (int j = 0; j < layer.GetWidth() * layer.GetHeight(); j++)
      m_DelayMap[i][j] = 0;
  }
}

////////////////////////////////////////////////////////////////////////////////

void
SMAP::DeleteMap()
{
  if (m_DelayMap)
  {
    for (int i = 0; i < m_Map.GetNumLayers(); i++)
      delete[] m_DelayMap[i];
    delete[] m_DelayMap;
  }
  m_DelayMap = NULL;
}

////////////////////////////////////////////////////////////////////////////////

void
SMAP::SetMap(const sTileset& tileset)
{
  for (int i = 0; i < m_Map.GetNumLayers(); i++)
  {
    const sLayer& layer = m_Map.GetLayer(i);
    int width  = layer.GetWidth();
    int height = layer.GetHeight();

    for (int j = 0; j < width * height; j++)
    {
      int tile = m_Map.GetLayer(i).GetTile(j % width, j / width);
      m_DelayMap[i][j] = tileset.GetTile(tile).GetDelay();
    }
  }
}

////////////////////////////////////////////////////////////////////////////////
