#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "sfont.hpp"
#include "x++.hpp"


////////////////////////////////////////////////////////////////////////////////

SFONT::SFONT()
: m_FontColors(NULL)
, m_NumFontColors(0)

, CurrentFontColor(NULL)

, m_MaxWidth(0)
, m_MaxHeight(0)
{
}

////////////////////////////////////////////////////////////////////////////////

SFONT::~SFONT()
{
  Destroy();
}

////////////////////////////////////////////////////////////////////////////////

bool
SFONT::Load(const char* filename)
{
  Destroy();

  // load file
  if (m_Font.Load(filename) == false)
    return false;

  // Sphere fonts must have 256 characters
  if (m_Font.GetNumCharacters() != 256)
    return false;

  // calculate maximum width and height
  m_MaxWidth = 0;
  m_MaxHeight = 0;
  for (int i = 0; i < m_Font.GetNumCharacters(); i++)
  {
    sFontCharacter& c = m_Font.GetCharacter(i);
    if (c.GetWidth() > m_MaxWidth)
      m_MaxWidth = c.GetWidth();
    if (c.GetHeight() > m_MaxHeight)
      m_MaxHeight = c.GetHeight();
  }

  // default color is white
  SetColor(rgbaWhite);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
SFONT::SetColor(RGBA color)
{
  // search through the cached colors to see if this color has already been created
  for (int i = 0; i < m_NumFontColors; i++)
    if (memcmp(&m_FontColors[i].color, &color, 4) == 0)
    {
      CurrentFontColor = m_FontColors + i;
      return true;
    }

  // create a new font color
  m_FontColors = (FontColor*)realloc(m_FontColors, (m_NumFontColors + 1) * sizeof(FontColor));
  CurrentFontColor = m_FontColors + m_NumFontColors;
  m_NumFontColors++;
  CurrentFontColor->color = color;

  RGBA* imagebuffer = new RGBA[m_MaxWidth * m_MaxHeight];

  // create the color font images
  for (int i = 0; i < 256; i++)
  {
    sFontCharacter& c = m_Font.GetCharacter(i);
    int w = c.GetWidth();
    int h = c.GetHeight();

    for (int j = 0; j < w * h; j++)
      imagebuffer[j] = color;

    RGBA* p = imagebuffer;
    byte* q = c.GetPixels();
    while (p < imagebuffer + w * h)
    {
      (*p).alpha = (*p).alpha * (*q) / 256;
      p++;
      q++;
    }

    CurrentFontColor->images[i] = CreateImage(w, h, imagebuffer);
    if (CurrentFontColor->images[i] == NULL)
    {
      for (int j = 0; j < i; j++)
        DestroyImage(CurrentFontColor->images[j]);
      delete[] imagebuffer;
      m_NumFontColors--;
      return false;
    }

  }

  delete[] imagebuffer;
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
SFONT::DrawText(int x, int y, const char* text) const
{
  while (*text)
  {
    IMAGE image = CurrentFontColor->images[*text];
    BlitImage(image, x, y);
    x += GetImageWidth(image);
    text++;
  }
  return true; 
}

////////////////////////////////////////////////////////////////////////////////

bool
SFONT::DrawTextBox(int x, int y, int w, int h, int offset, const char* text) const
{
  const char* p = text;

  if (*p == 0)
    return true;

  int numwords = 0;
  char** words = NULL;

  // parse the text into words
  bool done = false;
  while (!done)
  {
    words = (char**)realloc(words, sizeof(char*) * ++numwords);
    words[numwords - 1] = (char*)malloc(520 * sizeof(char));

    // add non-whitespace characters
    int w = 0;
    while (*p && !isspace(*p))
      words[numwords - 1][w++] = *p++;
    words[numwords - 1][w] = 0;

    if (*p != 0)
    {
      // skip whitespace characters
      while (*p != 0 && isspace(*p))
        p++;
    }

    if (*p == 0)
      done = true;
  }

  // render the words
  int cx = 0;
  int cy = 0;
  for (int i = 0; i < numwords; i++)
  {
    char* wd = words[i];

    if (cx + GetStringWidth(wd) > w)
    {
      cy += 12;
      cx = 0;
    }

    DrawText(cx + x, cy + y + offset, wd);

    cx += GetStringWidth(wd) + GetStringWidth(" ");
  }

  // free words
  for (int i = 0; i < numwords; i++)
    free(words[i]);
  free(words);
 
  return true;
}

////////////////////////////////////////////////////////////////////////////////

int
SFONT::GetMaxHeight() const
{
  return m_MaxHeight;
}

////////////////////////////////////////////////////////////////////////////////

int
SFONT::GetStringWidth(const char* string) const
{
  int width = 0;
  while (*string)
  {
    width += m_Font.GetCharacter(*string).GetWidth();
    string++;
  }
  return width;
}

////////////////////////////////////////////////////////////////////////////////

void
SFONT::Destroy()
{
  // free all the cached colors
  for (int i = 0; i < m_NumFontColors; i++)
  {
    // destroy the old images
    for (int j = 0; j < 256; j++)
      DestroyImage(m_FontColors[i].images[j]);
  }

  if (m_FontColors != NULL)
    free(m_FontColors);

  m_FontColors = NULL;
  m_NumFontColors = 0;
}

////////////////////////////////////////////////////////////////////////////////
