#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "script.hpp"
#include "../common/SS_Tables.hpp"

#include "vm_interface.hpp"
#include "render.h"
#include "inputx.h"

#include "smap.hpp"
#include "sfont.hpp"
#include "swindowstyle.hpp"
#include "simage.hpp"

#include "system.h"
#include "input.h"
#include "timer.h"

#include "packed.h"
#include "x++.hpp"



#define MAX_FRAME_SKIP 10



PACKED_STRUCT(SCRIPTHEADER)
  byte  signature[4];
  word  version;
  word  numlabels;
  dword codesize;
  word  num_dependancies;
  byte  reserved[50];
END_STRUCT(SCRIPTHEADER)


// SphereScript data types

struct SS_COLOR
{
  dword red;
  dword green;
  dword blue;
  dword alpha;
};

struct SS_IMAGE
{
  IMAGE handle;
};

struct SS_SURFACE
{
  CImage32* handle;
};

struct SS_FILE
{
  DATAFILE* handle;
};


SS_OPCODE GetOpcode(int value);


////////////////////////////////////////////////////////////////////////////////

CScript::CScript(CVM_Interface* vm_interface)
: VM_Interface(vm_interface)

, num_dependancies(0)
, dependancy_names(NULL)
, dependancies(NULL)

, numlabels(0)
, labels(NULL)
, labeloffsets(NULL)
, code(NULL)
{
}

////////////////////////////////////////////////////////////////////////////////

CScript::~CScript()
{
  ExecuteMangledFunction("uninitialize");

  for (int i = 0; i < num_dependancies; i++)
    delete[] dependancy_names[i];

  delete[] dependancy_names;
  delete[] dependancies;

  for (int i = 0; i < numlabels; i++)
    delete[] labels[i];
  delete[] labels;
  delete[] labeloffsets;
  delete[] code;
}

////////////////////////////////////////////////////////////////////////////////

bool
CScript::Load(const char* filename)
{
  // open the file
  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  // read and check header
  SCRIPTHEADER header;
  fread(&header, 1, sizeof(header), file);
  if (memcmp(header.signature, ".ssx", 4) != 0 ||
      header.version != 2)
  {
    fclose(file);
    return false;
  }

  // fill the data structure

  num_dependancies = header.num_dependancies;
  dependancy_names = new char*[header.num_dependancies];
  dependancies = new CScript*[header.num_dependancies];

  numlabels = header.numlabels;

  labels = new char*[numlabels];
  labeloffsets = new dword[numlabels];
  code = new dword[header.codesize];

  // read the label names and offsets
  for (int i = 0; i < numlabels; i++)
  {
    word length;
    fread(&length, 1, 2, file);
    labels[i] = new char[length];
    fread(labels[i], 1, length, file);
    fread(labeloffsets + i, 1, 4, file);
  }

  // read dependancies and ask engine for them
  for (int i = 0; i < num_dependancies; i++)
  {
    word length;
    fread(&length, 1, 2, file);
    char* name = new char[length];
    fread(name, 1, length, file);

    dependancy_names[i] = name;
    dependancies[i] = NULL;
  }

  fread(code, 4, header.codesize, file);
  fclose(file);

  ExecuteMangledFunction("initialize");
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CScript::ExecuteFunction(const char* function)
{
  char* mangled = new char[strlen(function) + 2];
  sprintf(mangled, "_%s", function);
  bool result = ExecuteMangledFunction(mangled);
  delete[] mangled;
  return result;
}

////////////////////////////////////////////////////////////////////////////////

int
CScript::GetNumDependancies()
{
  return num_dependancies;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CScript::GetDependancyName(int i)
{
  return dependancy_names[i];
}

////////////////////////////////////////////////////////////////////////////////

void
CScript::ResolveDependancy(int i, CScript* script)
{
  dependancies[i] = script;
}

////////////////////////////////////////////////////////////////////////////////

CScript*
CScript::GetDependancy(int i)
{
  return dependancies[i];
}

////////////////////////////////////////////////////////////////////////////////

bool
CScript::HasExport(const char* name) const
{
  for (int i = 0; i < numlabels; i++)
    if (strcmp(name, labels[i]) == 0)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool
CScript::ExecuteMangledFunction(const char* function, sdword* retval, int stacksize, sdword* stack, int parameter_level)
{
  struct CALLSTACKLEVEL
  {
    dword* location;
    int    parameter_level;

    int     num_locals;
    sdword* locals;
  };

  // make a new execution context
  contexts.push(ExecutionContext());
  ExecutionContext& c = contexts.top();

  // set up timing variables for renderer
  c.frame_rate = 0;
  c.should_render = true;
  c.frames_skipped = 0;
  c.ideal_time = 0;

  // internal stack is used for call/ret
  CALLSTACKLEVEL* callstack = NULL;
  int callstacksize = 0;

  if (stacksize == 0 || stack == NULL) 
  {
    c.stacksize = 0;
    c.stack     = NULL;
  }
  else
  {
    c.stacksize = stacksize;
    c.stack = (sdword*)malloc(sizeof(sdword) * stacksize);
    memcpy(c.stack, stack, sizeof(sdword) * stacksize);
  }
  c.parameter_level = parameter_level;

  c.num_locals = 0;
  c.locals     = NULL;

  // find the label we want to start at
  dword* location = NULL;
  for (int i = 0; i < numlabels; i++)
  {
    if (strcmp(function, labels[i]) == 0)
    {
      location = code + labeloffsets[i];
      break;
    }
  }
  if (location == NULL)
    return false;

  // clear registers
  memset(c.registers, 0, sizeof(c.registers));

  // interpret the code
  bool done = false;
  while (!done)
  {
    dword opcode = *location++;
    SS_OPCODE o = GetOpcode(opcode);

    sdword _p1; // spare slots
    sdword _p2; 
    sdword* p1; // parameters
    sdword* p2;
    if (o.num_parameters >= 1)
      p1 = GrabParameter(location, &_p1);
    if (o.num_parameters >= 2)
      p2 = GrabParameter(location, &_p2);

    switch (opcode)
    {
      // INTEGER
      case 0x00: // mov
        (*p1) = (*p2);
        break;
      case 0x01: // add
        (*p1) += (*p2);
        break;
      case 0x02: // sub
        (*p1) -= (*p2);
        break;
      case 0x03: // mul
        (*p1) *= (*p2);
        break;
      case 0x04: // div
        (*p1) /= (*p2);
        break;
      case 0x05: // mod
        (*p1) %= (*p2);
        break;
      case 0x06: // neg
        (*p1) = -(*p1);
        break;
      case 0x07: // clr
        (*p1) = 0;
        break;

      // INTEGER COMPARISON
      case 0x10: // cmpe
        (*p1) = (*p1) == (*p2);
        break;
      case 0x11: // cmpg
        (*p1) = (*p1) > (*p2);
        break;
      case 0x12: // cmpge
        (*p1) = (*p1) >= (*p2);
        break;
      case 0x13: // cmpl
        (*p1) = (*p1) < (*p2);
        break;
      case 0x14: // cmple
        (*p1) = (*p1) <= (*p2);
        break;

      // STRING
      case 0x20: // sload
      {
        const char* str = (const char*)p2;
        (*p1) = (sdword)malloc(strlen(str) + 1);
        strcpy((char*)(*p1), str);
        break;
      }
      case 0x21: // sfree
        free((char*)(*p1));
        break;
      case 0x22: // smov
        (*p1) = (sdword)realloc((char*)(*p1), strlen((char*)(*p2)) + 1);
        strcpy((char*)(*p1), (char*)(*p2));
        break;
      case 0x23: // sadd
        (*p1) = (sdword)realloc((char*)(*p1), strlen((char*)(*p1)) + strlen((char*)(*p2)) + 1);
        strcat((char*)(*p1), (char*)(*p2));
        break;
      case 0x24: // sclr
        (*p1) = (sdword)realloc((char*)(*p1), 1);
        strcpy((char*)(*p1), "");
        break;

      // STRING COMPARISON
      case 0x30: // scmpe
      {
        char* str = (char*)(*p1);
        (*p1) = strcmp((char*)(*p1), (char*)(*p2)) == 0;
        free(str);
        break;
      }
      case 0x31: // scmpg
      {
        char* str = (char*)(*p1);
        (*p1) = strcmp((char*)(*p1), (char*)(*p2)) > 0;
        free(str);
        break;
      }
      case 0x32: // scmpge
      {
        char* str = (char*)(*p1);
        (*p1) = strcmp((char*)(*p1), (char*)(*p2)) >= 0;
        free(str);
        break;
      }
      case 0x33: // scmpl
      {
        char* str = (char*)(*p1);
        (*p1) = strcmp((char*)(*p1), (char*)(*p2)) < 0;
        free(str);
        break;
      }
      case 0x34: // scmple
      {
        char* str = (char*)(*p1);
        (*p1) = strcmp((char*)(*p1), (char*)(*p2)) <= 0;
        free(str);
        break;
      }

      // FLOATING POINT
      case 0x40: // fadd
        *(float*)(p1) = *(float*)(p1) + *(float*)(p2);
        break;
      case 0x41: // fsub
        *(float*)(p1) = *(float*)(p1) - *(float*)(p2);
        break;
      case 0x42: // fmul
        *(float*)(p1) = *(float*)(p1) * *(float*)(p2);
        break;
      case 0x43: // fdiv
        *(float*)(p1) = *(float*)(p1) / *(float*)(p2);
        break;
      case 0x44: // fneg
        *(float*)(p1) = -*(float*)(p1);
        break;

      // FLOATING POINT COMPARISON
      case 0x50: // fcmpe
        (*p1) = *(float*)(p1) == *(float*)(p2);
        break;
      case 0x51: // fcmpg
        (*p1) = *(float*)(p1) > *(float*)(p2);
        break;
      case 0x52: // fcmpge
        (*p1) = *(float*)(p1) >= *(float*)(p2);
        break;
      case 0x53: // fcmpl
        (*p1) = *(float*)(p1) < *(float*)(p2);
        break;
      case 0x54: // fcmple
        (*p1) = *(float*)(p1) <= *(float*)(p2);
        break;

      // BOOLEAN
      case 0x60: // and
        (*p1) = (*p1) && (*p2);
        break;
      case 0x61: // or
        (*p1) = (*p1) || (*p2);
        break;
      case 0x62: // xor
        (*p1) = (*p1) != (*p2);
        break;
      case 0x63: // not
        (*p1) = !(*p1);
        break;

      // STACK
      case 0x70: // push
        c.stack = (sdword*)realloc(c.stack, (c.stacksize + 1) * 4);
        c.stack[c.stacksize] = (*p1);
        c.stacksize++;
        break;
      case 0x71: // pop
        (*p1) = c.stack[c.stacksize - 1];
        c.stack = (sdword*)realloc(c.stack, (c.stacksize - 1) * 4);
        c.stacksize--;
        break;
      case 0x72: // call
        callstack = (CALLSTACKLEVEL*)realloc(callstack, (callstacksize + 1) * sizeof(CALLSTACKLEVEL));
        callstack[callstacksize].location = location;
        callstack[callstacksize].parameter_level = c.parameter_level;
        callstack[callstacksize].num_locals = c.num_locals;
        callstack[callstacksize].locals = c.locals;
        callstacksize++;

        location = (dword*)p1;
        c.parameter_level = c.stacksize;
        c.num_locals = 0;
        c.locals = NULL;
        break;
      case 0x73: // ret
      {
        realloc(c.locals, 0);

        if (callstacksize == 0)
        {
          done = true;
          break;
        }

        location          = callstack[callstacksize - 1].location;
        c.parameter_level = callstack[callstacksize - 1].parameter_level;
        c.num_locals      = callstack[callstacksize - 1].num_locals;
        c.locals          = callstack[callstacksize - 1].locals;

        callstack = (CALLSTACKLEVEL*)realloc(callstack, (callstacksize - 1) * sizeof(CALLSTACKLEVEL));
        callstacksize--;
        break;
      }
      case 0x74: // syscall
      {
        int old_parameter_level = c.parameter_level;
        c.parameter_level = c.stacksize - GetNumParameters(*p1);
        ExecuteSystemFunction(*p1);
        c.parameter_level = old_parameter_level;
        break;
      }
      case 0x75: // extcall
      {
        int old_parameter_level = c.parameter_level;
        c.parameter_level = c.stacksize;

        const char* function = (const char*)p1;
        
        // search through dependancies for function
        for (int i = 0; i < num_dependancies; i++)
        {
          CScript* script = dependancies[i];
          if (script->HasExport(function))
          {
            // call the function
            if (!script->ExecuteMangledFunction(function, c.registers, c.stacksize, c.stack, c.parameter_level))
            {
              char message[1024];
              sprintf(message, "Error: Could not find exported function '%s'", function);
              VM_Interface->ExitWithMessage(message);
            }
          }
        }

        c.parameter_level = old_parameter_level;
        break;
      }

      // JUMP
      case 0x80: // jmp
        location = (dword*)p1;
        break;
      case 0x81: // jmpt
        if (*p1)
          location = (dword*)p2;
        break;
      case 0x82: // jmpf
        if (!*p1)
          location = (dword*)p2;
        break;

      // MEMORY
      case 0x90: // alloc
        (*p1) = (sdword)calloc((*p2) * 4, 1);
        break;
      case 0x91: // free
        free((void*)(*p1));
        break;

      // VM COMMANDS
      case 0xA0: // lc
        c.locals = (sdword*)realloc(c.locals, (c.num_locals + 1) * 4);
        c.locals[c.num_locals] = 0;
        c.num_locals++;
        break;
      case 0xA1: // ld
        c.locals = (sdword*)realloc(c.locals, (c.num_locals - 1) * 4);
        c.num_locals--;
        break;
      case 0xA2: // parameter
        c.parameter_level--;
        break;

      default:
      {
        char message[520];
        sprintf(message, "Invalid opcode: %x", opcode);
        VM_Interface->ExitWithMessage(message);
        break;
      }
    }

    UpdateSystem();

    if (VM_Interface->ShouldExit())
      done = true;
  }

  // free the stacks
  realloc(c.stack, 0);
  realloc(callstack, 0);

  // put return value in
  if (retval)
    *retval = c.registers[0];

  contexts.pop();

  return true;
}

////////////////////////////////////////////////////////////////////////////////

void
CScript::ExecuteSystemFunction(int ordinal)
{
  ExecutionContext& c = contexts.top();

  switch (ordinal)
  {
    ////////////////////////////////////////////////////////////////////////////

    // ENGINE

    case 0x0100: // void Exit(void)
      VM_Interface->Exit();
      break;

    case 0x0101: // void ExitMessage(string message)
      VM_Interface->ExitWithMessage((char*)c.stack[c.stacksize - 1]);
      break;

    case 0x0102: // void ChangeMusic(string song)
    {
      char* song = (char*)c.stack[c.parameter_level + 0];
      VM_Interface->ChangeSong(song);
      break;
    }

    case 0x0103: // void PlaySound(string sound)
    {
      char* sound = (char*)c.stack[c.parameter_level + 0];
      VM_Interface->PlayEffect(sound);
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // TIMING
    case 0x0200: // int GetTime()
      c.registers[0] = GetTime();
      break;

    case 0x0201: // void Delay(int milliseconds)
    {
      dword milliseconds = (dword)c.stack[c.parameter_level + 0];
      dword start = GetTime();
      while (GetTime() - start < milliseconds)
        UpdateSystem();
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // INPUT

    case 0x0300: // void LockInput()
      break;

    case 0x0301: // void UnlockInput()
      break;

    case 0x0302: // bool AnyKeyPressed()
      RefreshInput();
      c.registers[0] = AnyKeyPressed();
      break;

    case 0x0303: // bool KeyPressed(int keycode)
      RefreshInput();
      c.registers[0] = KeyPressed(c.stack[c.parameter_level + 0]);
      break;

    case 0x0304: // int GetMouseX()
      RefreshInput();
      c.registers[0] = GetMouseX();
      break;

    case 0x0305: // int GetMouseY()
      RefreshInput();
      c.registers[0] = GetMouseY();
      break;

    case 0x0306: // bool MouseButtonPressed(int buttoncode)
      RefreshInput();
      c.registers[0] = MouseButtonPressed(c.stack[c.parameter_level + 0]);
      break;

    case 0x0307: // void ClearKeyQueue()
      ClearKeyQueue();
      break;

    case 0x0308: // bool KeysLeft()
      c.registers[0] = KeysLeft();
      break;

    case 0x0309: // int GetKey()
      c.registers[0] = GetKey();
      break;

    ////////////////////////////////////////////////////////////////////////////

    // GRAPHICS

    case 0x0400: // void FlipScreen()
    {
      if (c.frame_rate == 0)
        FlipScreen();
      else
      {
        // never skip more than MAX_FRAME_SKIP frames
        if (c.should_render || c.frames_skipped >= MAX_FRAME_SKIP)
        {
          FlipScreen();
          c.frames_skipped = 0;
        }
        else
          c.frames_skipped++;

        if (GetTime() * c.frame_rate < c.ideal_time)
        {
          c.should_render = true;
          while (GetTime() * c.frame_rate < c.ideal_time)
            ;
        }
        else
          c.should_render = false;

        // update timing variables
        c.ideal_time += 1000;
      }
      break;
    }

    case 0x0401: // void SetClippingRectangle(int x, int y, int w, int h)
      SetClippingRectangle(
        c.stack[c.parameter_level + 0],
        c.stack[c.parameter_level + 1],
        c.stack[c.parameter_level + 2],
        c.stack[c.parameter_level + 3]);
      break;

    case 0x0402: // void ApplyColorMask(color mask)
      if (c.should_render)
      {
        SS_COLOR* clr = (SS_COLOR*)c.stack[c.parameter_level + 0];
        RGBA c = { (byte)clr->red, (byte)clr->green, (byte)clr->blue, (byte)clr->alpha };
        ApplyColorMask(c);
      }
      break;

    case 0x0403: // void SetFrameRate(int frame_rate)
      c.frame_rate = c.stack[c.parameter_level + 0];
      c.ideal_time = GetTime() * c.frame_rate + 1000;
      c.should_render = true;
      break;

    case 0x0404: // int GetFrameRate()
      c.registers[0] = c.frame_rate;
      break;

    case 0x0405: // int GetScreenWidth()
      c.registers[0] = GetScreenWidth();
      break;

    case 0x0406: // int GetScreenHeight()
      c.registers[0] = GetScreenHeight();
      break;

    case 0x0407: // void ClearScreen()
      ClearScreen();
      break;

    ////////////////////////////////////////////////////////////////////////////

    // GRAPHICS - IMAGES
    
    case 0x0500: // image LoadImage(string filename)
    { 
      SS_IMAGE* image = new SS_IMAGE;
      char* filename = (char*)c.stack[c.parameter_level + 0];
      image->handle = VM_Interface->LoadImage(filename);
      c.registers[0] = (sdword)image;
      break;
    }

    case 0x0501: // image GrabImage(int x, int y, int w, int h)
    {
      if (c.frame_rate != 0)
        VM_Interface->ExitWithMessage("GrabImage() should never be called if SetFrameRate() is set to something other than 0");

      SS_IMAGE* image = new SS_IMAGE;
      image->handle = GrabImage(
        c.stack[c.parameter_level + 0],
        c.stack[c.parameter_level + 1],
        c.stack[c.parameter_level + 2],
        c.stack[c.parameter_level + 3]);
      c.registers[0] = (sdword)image;
      break;
    }

    case 0x0502: // void DestroyImage(image i)
    {
      SS_IMAGE* image = (SS_IMAGE*)c.stack[c.parameter_level + 0];
      if (image->handle == NULL)
        VM_Interface->ExitWithMessage("Attempted to destroy an empty image");
      else
      {
        DestroyImage(image->handle);
        image->handle = NULL;
      }
      break;
    }

    case 0x0503: // void BlitImage(image i, int x, int y)
      if (c.should_render)
      {
        SS_IMAGE* image = (SS_IMAGE*)c.stack[c.parameter_level + 0];
        BlitImage(image->handle,
          c.stack[c.parameter_level + 1],
          c.stack[c.parameter_level + 2]);
      }
      break;

    case 0x0504: // int GetImageWidth(image i)
    {
      SS_IMAGE* image = (SS_IMAGE*)c.stack[c.parameter_level + 0];
      c.registers[0] = GetImageWidth(image->handle);
      break;
    }

    case 0x0505: // int GetImageHeight(image i)
    {
      SS_IMAGE* image = (SS_IMAGE*)c.stack[c.parameter_level + 0];
      c.registers[0] = GetImageHeight(image->handle);
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // GRAPHICS - TEXT

    case 0x0600: // void SetFont(string filename)
    {
      char* filename = (char*)c.stack[c.parameter_level + 0];
      VM_Interface->SetFont(filename);
      break;
    }

    case 0x0601: // void SetTextColor(color c)
    {
      SS_COLOR* color = (SS_COLOR*)c.stack[c.parameter_level + 0];
      RGBA c = { (byte)color->red, (byte)color->green, (byte)color->blue, (byte)color->alpha };
      VM_Interface->GetFont()->SetColor(c);
      break;
    }

    case 0x0602: // void DrawText(int x, int y, string text)
      if (c.should_render)
      {
        int x = c.stack[c.parameter_level + 0];
        int y = c.stack[c.parameter_level + 1];
        char* text = (char*)c.stack[c.parameter_level + 2];
        VM_Interface->GetFont()->DrawText(x, y, text);
      }
      break;

    case 0x0603: // void DrawTextBox(int x, int y, int w, int h, int offset, string text)
      if (c.should_render)
      {
        int x = c.stack[c.parameter_level + 0];
        int y = c.stack[c.parameter_level + 1];
        int w = c.stack[c.parameter_level + 2];
        int h = c.stack[c.parameter_level + 3];
        int offset = c.stack[c.parameter_level + 4];
        char* text = (char*)c.stack[c.parameter_level + 5];
        VM_Interface->GetFont()->DrawTextBox(x, y, w, h, offset, text);
      }
      break;

    ////////////////////////////////////////////////////////////////////////////

    // GRAPHICS - WINDOWS

    case 0x0700: // void SetWindowStyle(string filename)
    {
      char* filename = (char*)c.stack[c.parameter_level + 0];
      VM_Interface->SetWindowStyle(filename);
      break;
    }

    case 0x0701: // void DrawWindow(int x, int y, int w, int h)
      if (c.should_render)
      {
        int x = c.stack[c.parameter_level + 0];
        int y = c.stack[c.parameter_level + 1];
        int w = c.stack[c.parameter_level + 2];
        int h = c.stack[c.parameter_level + 3];
        VM_Interface->GetWindowStyle()->DrawWindow(x, y, w, h);
      }
      break;

    ////////////////////////////////////////////////////////////////////////////

    // GRAPHICS - EFFECTS

    case 0x0800: // void FadeIn(int milliseconds)
      FadeIn(c.stack[c.parameter_level + 0]);
      break;

    case 0x0801: // void FadeOut(int milliseconds)
      FadeOut(c.stack[c.parameter_level + 0]);
      break;

    ////////////////////////////////////////////////////////////////////////////

    // MAP ENGINE

    case 0x0900: // void MapEngine()
      VM_Interface->MapEngine();
      break;

    case 0x0901: // void ChangeMap(string filename)
    {
      char* filename = (char*)c.stack[c.parameter_level + 0];
      VM_Interface->SetMap(filename);
      break;
    }

    case 0x0902: // int GetTile(int x, int y, int layer)
    {
      int x = c.stack[c.parameter_level + 0];
      int y = c.stack[c.parameter_level + 1];
      int l = c.stack[c.parameter_level + 2];
      c.registers[0] = VM_Interface->GetMap()->GetMap().GetLayer(l).GetTile(x, y);
      break;
    }

    case 0x0903: // void SetTile(int x, int y, int layer, int tile)
    {
      int x = c.stack[c.parameter_level + 0];
      int y = c.stack[c.parameter_level + 1];
      int l = c.stack[c.parameter_level + 2];
      int t = c.stack[c.parameter_level + 3];
      VM_Interface->GetMap()->GetMap().GetLayer(l).SetTile(x, y, t);
      break;
    }

    case 0x0904: // void SetColorMask(color mask, int num_frames)
    {
      SS_COLOR* mask = (SS_COLOR*)c.stack[c.parameter_level + 0];
      RGBA clr = { (byte)mask->red, (byte)mask->green, (byte)mask->blue, (byte)mask->alpha };
      int num_frames = c.stack[c.parameter_level + 1];
      VM_Interface->SetColorMask(clr, num_frames);
      break;
    }

    case 0x0905: // void AddFrameHook(int update_rate, string function)
    {
      int   update_rate = c.stack[c.parameter_level + 0];
      char* function    = (char*)c.stack[c.parameter_level + 1];
      VM_Interface->AddFrameHook(update_rate, function);
      break;
    }

    case 0x0906: // void RemoveFrameHook(string function)
    {
      char* function = (char*)c.stack[c.parameter_level];
      VM_Interface->RemoveFrameHook(function);
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // MAP ENGINE - PARTY

    case 0x0A00: // void ClearParty()
      VM_Interface->ClearParty();
      break;

    case 0x0A01: // void AddPartyCharacter(string filename)
    {
      char* filename = (char*)c.stack[c.parameter_level + 0];
      VM_Interface->AddPartyCharacter(filename);
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // MAP ENGINE - PERSONS

    case 0x0B00: // void CreatePerson
      break;
    case 0x0B01: // void DestroyPerson
      break;
    case 0x0B02: // void SetPersonText
      break;
    case 0x0B03: // void WarpPerson
      break;
    case 0x0B04: // QueuePersonCommand
      break;

    ////////////////////////////////////////////////////////////////////////////

    // MAP ENGINE - DOODADS

    case 0x1600: // void SetDoodadFrame(int frame)
    {
      int frame = c.stack[c.parameter_level + 0];
      VM_Interface->SetDoodadFrame(frame);
      break;
    }

    case 0x1601: // void SetDoodadObstructive(bool obstructive)
    {
      bool obstructive = c.stack[c.parameter_level + 0] != 0;
      VM_Interface->SetDoodadObstructive(obstructive);
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // STRING

    case 0x0C00: // int StringLength(string s)
      c.registers[0] = strlen((char*)c.stack[c.parameter_level + 0]);
      break;

    case 0x0C01: // int StringWidth(string s)
    {  
      char* s = (char*)c.stack[c.parameter_level + 0];
      c.registers[0] = VM_Interface->GetFont()->GetStringWidth(s);
      break;
    }

    case 0x0C02: // string LeftString(string s, int i)
    {
      char* s = (char*)c.stack[c.parameter_level + 0];
      int i = c.stack[c.parameter_level + 1];
      char* r = (char*)malloc(i + 1);
      memcpy(r, s, i);
      r[i] = 0;
      c.registers[0] = (sdword)r;
      break;
    }

    case 0x0C03: // string RightString(string s, int i)
    {
      char* s = (char*)c.stack[c.parameter_level + 0];
      int i = c.stack[c.parameter_level + 1];
      char* r = (char*)malloc(i + 1);
      memcpy(r, s + strlen(s) - i, i);
      r[i] = 0;
      c.registers[0] = (sdword)r;
      break;
    }

    case 0x0C04: // string MidString(string s, int i, int l)
    {
      char* s = (char*)c.stack[c.parameter_level + 0];
      int i = c.stack[c.parameter_level + 1];
      int l = c.stack[c.parameter_level + 2];
      char* r = (char*)malloc(l + 1);
      memcpy(r, s + i, l);
      r[l] = 0;
      c.registers[0] = (sdword)r;
      break;
    }

    case 0x0C05: // string Character(int c)
    {
      char* s = (char*)malloc(2);
      s[0] = (char)(c.stack[c.parameter_level + 0]);
      s[1] = 0;
      c.registers[0] = (sdword)s;
      break;
    }

    case 0x0C06: // int Ascii(string s)
    {
      char* s = (char*)c.stack[c.parameter_level + 0];
      c.registers[0] = s[0];
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // MATH

    case 0x0D00: // float abs(float f)
    {
      float f = *(float*)(c.stack + c.parameter_level + 0);
      *(float*)&(c.registers[0]) = (float)fabs(f);
      break;
    }

    case 0x0D01: // float sin(float f)
    {
      float f = *(float*)(c.stack + c.parameter_level + 0);
      *(float*)&(c.registers[0]) = (float)sin(f);
      break;
    }

    case 0x0D02: // float cos(float f)
    {
      float f = *(float*)(c.stack + c.parameter_level + 0);
      *(float*)&(c.registers[0]) = (float)cos(f);
      break;
    }

    case 0x0D03: // float tan(float f)
    {
      float f = *(float*)(c.stack + c.parameter_level + 0);
      *(float*)&(c.registers[0]) = (float)tan(f);
      break;
    }

    case 0x0D04: // float asin(float f)
    {
      float f = *(float*)(c.stack + c.parameter_level + 0);
      *(float*)&(c.registers[0]) = (float)asin(f);
      break;
    }

    case 0x0D05: // float acos(float f)
    {
      float f = *(float*)(c.stack + c.parameter_level + 0);
      *(float*)&(c.registers[0]) = (float)acos(f);
      break;
    }

    case 0x0D06: // float atan(float f)
    {
      float f = *(float*)(c.stack + c.parameter_level + 0);
      *(float*)&(c.registers[0]) = (float)atan(f);
      break;
    }

    case 0x0D07: // int random(int min, int max)
    {
      int min = c.stack[c.parameter_level + 0];
      int max = c.stack[c.parameter_level + 1];
      c.registers[0] = (rand() % (max - min + 1)) + min;
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // SURFACES

    case 0x0E00: // surface CreateSurface(int w, int h)
    {
      SS_SURFACE* surface = (SS_SURFACE*)malloc(sizeof(SS_SURFACE));
      int w = c.stack[c.parameter_level + 0];
      int h = c.stack[c.parameter_level + 1];
      surface->handle = new CImage32(w, h);
      c.registers[0] = (sdword)surface;
      break;
    }

    case 0x0E01: // void DestroySurface(surface s)
    {
      SS_SURFACE* surface = (SS_SURFACE*)c.stack[c.parameter_level + 0];
      delete surface->handle;
      break;
    }

    case 0x0E02: // image CreateImageFromSurface(surface s)
    {
      SS_IMAGE* image = (SS_IMAGE*)malloc(sizeof(SS_IMAGE));
      SS_SURFACE* surface = (SS_SURFACE*)c.stack[c.parameter_level + 0];
      image->handle = CreateImage(
        surface->handle->GetWidth(),
        surface->handle->GetHeight(),
        surface->handle->GetPixels());
      c.registers[0] = (sdword)image;
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // SURFACES - PEN/BRUSH

    case 0x0F00: // void SetPenColor(color c)
    {
      SS_COLOR* color = (SS_COLOR*)c.stack[c.parameter_level + 0];
      RGBA clr = { (byte)color->red, (byte)color->green, (byte)color->blue, (byte)color->alpha };
      c.CurrentPenColor = clr;
      break;
    }

    case 0x0F01: // void SetBrushColor(color c)
    {
      SS_COLOR* color = (SS_COLOR*)c.stack[c.parameter_level + 0];
      RGBA clr = { (byte)color->red, (byte)color->green, (byte)color->blue, (byte)color->alpha };
      c.CurrentBrushColor = clr;
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // SURFACES - GRAPHICS PRIMITIVES

    case 0x1000: // void SetPixel(surface s, int x, int y)
    {
      SS_SURFACE* surface = (SS_SURFACE*)c.stack[c.parameter_level + 0];
      int x = c.stack[c.parameter_level + 1];
      int y = c.stack[c.parameter_level + 2];
      surface->handle->SetPixel(x, y, c.CurrentPenColor);
      break;
    }

    case 0x1001: // void Line(surface s, int x1, int y1, int x2, int y2)
    {
      SS_SURFACE* surface = (SS_SURFACE*)c.stack[c.parameter_level + 0];
      int x1 = c.stack[c.parameter_level + 1];
      int y1 = c.stack[c.parameter_level + 2];
      int x2 = c.stack[c.parameter_level + 3];
      int y2 = c.stack[c.parameter_level + 4];
      surface->handle->Line(x1, y1, x2, y2, c.CurrentPenColor);
      break;
    }

    case 0x1002: // void Circle(surface s, int x, int y, int r)
    {
      SS_SURFACE* surface = (SS_SURFACE*)c.stack[c.parameter_level + 0];
      int x = c.stack[c.parameter_level + 1];
      int y = c.stack[c.parameter_level + 2];
      int r = c.stack[c.parameter_level + 3];
      surface->handle->Circle(x, y, r, c.CurrentPenColor, c.CurrentBrushColor);
      break;
    }

    case 0x1003: // void Rectangle(surface s, int x1, int y1, int x2, int x2)
    {
      SS_SURFACE* surface = (SS_SURFACE*)c.stack[c.parameter_level + 0];
      int x1 = c.stack[c.parameter_level + 1];
      int y1 = c.stack[c.parameter_level + 2];
      int x2 = c.stack[c.parameter_level + 3];
      int y2 = c.stack[c.parameter_level + 4];
      surface->handle->Rectangle(x1, y1, x2, y2, c.CurrentPenColor, c.CurrentBrushColor);
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // COLORS

    case 0x1100: // color BlendColors(color a, color b)
    {
      SS_COLOR* a = (SS_COLOR*)c.stack[c.parameter_level + 0];
      SS_COLOR* b = (SS_COLOR*)c.stack[c.parameter_level + 1];
      SS_COLOR* clr = (SS_COLOR*)malloc(sizeof(SS_COLOR));
      clr->red   = (a->red   + b->red)   / 2;
      clr->green = (a->green + b->green) / 2;
      clr->blue  = (a->blue  + b->blue)  / 2;
      clr->alpha = (a->alpha + b->alpha) / 2;
      c.registers[0] = (sdword)clr;
      break;
    }

    case 0x1101: // color BlendColorsWeighted(color a, color b, int weight)
    {
      SS_COLOR* a = (SS_COLOR*)c.stack[c.parameter_level + 0];
      SS_COLOR* b = (SS_COLOR*)c.stack[c.parameter_level + 1];
      int weight = c.stack[c.parameter_level + 2];

      SS_COLOR* clr = (SS_COLOR*)malloc(sizeof(SS_COLOR));
      clr->red   = (a->red   * weight + b->red   * (256 - weight)) / 256;
      clr->green = (a->green * weight + b->green * (256 - weight)) / 256;
      clr->blue  = (a->blue  * weight + b->blue  * (256 - weight)) / 256;
      clr->alpha = (a->alpha * weight + b->alpha * (256 - weight)) / 256;
      c.registers[0] = (sdword)clr;
      break;
    }

    case 0x1102: // color CreateColor(int r, int g, int b, int a)
    {
      SS_COLOR* clr = (SS_COLOR*)malloc(sizeof(SS_COLOR));
      clr->red   = c.stack[c.parameter_level + 0];
      clr->green = c.stack[c.parameter_level + 1];
      clr->blue  = c.stack[c.parameter_level + 2];
      clr->alpha = c.stack[c.parameter_level + 3];
      c.registers[0] = (sdword)clr;
      break;
    }

    case 0x1103: // color SystemColor(int value)
    {
      SS_COLOR* clr = (SS_COLOR*)malloc(sizeof(SS_COLOR));

      switch (c.stack[c.parameter_level])
      {
        case 0:  // black
          clr->red   = 0;
          clr->green = 0;
          clr->blue  = 0;
          clr->alpha = 255;
          break;
        case 1:  // grey
          clr->red   = 128;
          clr->green = 128;
          clr->blue  = 128;
          clr->alpha = 255;
          break;
        case 2:  // white
          clr->red   = 255;
          clr->green = 255;
          clr->blue  = 255;
          clr->alpha = 255;
          break;
        case 3:  // red
          clr->red   = 255;
          clr->green = 0;
          clr->blue  = 0;
          clr->alpha = 255;
          break;
        case 4:  // green
          clr->red   = 0;
          clr->green = 255;
          clr->blue  = 0;
          clr->alpha = 255;
          break;
        case 5:  // blue
          clr->red   = 0;
          clr->green = 0;
          clr->blue  = 255;
          clr->alpha = 255;
          break;
        case 6:  // cyan
          clr->red   = 0;
          clr->green = 255;
          clr->blue  = 255;
          clr->alpha = 255;
          break;
        case 7:  // magenta
          clr->red   = 255;
          clr->green = 0;
          clr->blue  = 255;
          clr->alpha = 255;
          break;
        case 8:  // yellow
          clr->red   = 255;
          clr->green = 255;
          clr->blue  = 0;
          clr->alpha = 255;
          break;
        case 9:  // transparent black
          clr->red   = 0;
          clr->green = 0;
          clr->blue  = 0;
          clr->alpha = 0;
          break;
        case 10: // transparent grey
          clr->red   = 128;
          clr->green = 128;
          clr->blue  = 128;
          clr->alpha = 0;
          break;
        case 11: // transparent white
          clr->red   = 255;
          clr->green = 255;
          clr->blue  = 255;
          clr->alpha = 0;
          break;
        case 12: // transparent red
          clr->red   = 255;
          clr->green = 0;
          clr->blue  = 0;
          clr->alpha = 0;
          break;
        case 13: // transparent green
          clr->red   = 0;
          clr->green = 255;
          clr->blue  = 0;
          clr->alpha = 0;
          break;
        case 14: // transparent blue
          clr->red   = 0;
          clr->green = 0;
          clr->blue  = 255;
          clr->alpha = 0;
          break;
        case 15: // transparent cyan
          clr->red   = 0;
          clr->green = 255;
          clr->blue  = 255;
          clr->alpha = 0;
          break;
        case 16: // transparent magenta
          clr->red   = 255;
          clr->green = 0;
          clr->blue  = 255;
          clr->alpha = 0;
          break;
        case 17: // transparent yellow
          clr->red   = 255;
          clr->green = 255;
          clr->blue  = 0;
          clr->alpha = 0;
          break;
      }

      c.registers[0] = (sdword)clr;
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // ANIMATION

    case 0x1200: // void PlayAnimation(string filename)
    {
      char* filename = (char*)c.stack[c.parameter_level + 0];
      VM_Interface->PlayAnimation(filename);
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // TYPE CONVERSION

    case 0x1300: // string itos(int i)
    {
      char* s = (char*)malloc(80);
      sprintf(s, "%d", c.stack[c.parameter_level + 0]);
      c.registers[0] = (sdword)s;
      break;
    }

    case 0x1301: // float itof(int i)
    {
      float f = (float)c.stack[c.parameter_level + 0];
      *(float*)c.registers = f;
      break;
    }

    case 0x1302: // int stoi(string s)
    {
      c.registers[0] = atoi((char*)c.stack[c.parameter_level + 0]);
      break;
    }

    case 0x1303: // float stof(string s)
    {
      *(float*)c.registers = (float)atof((char*)c.stack[c.parameter_level + 0]);
      break;
    }

    case 0x1304: // int ftoi(float f)
    {
      c.registers[0] = (sdword)(*((float*)c.stack + c.parameter_level + 0));
      break;
    }

    case 0x1305: // string ftos(float f)
    {
      char* s = (char*)malloc(80);
      sprintf(s, "%g", (double)*((float*)c.stack + c.parameter_level + 0));
      c.registers[0] = (sdword)s;
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    // FILES

    case 0x1400: // file OpenFile(string filename)
    {
      SS_FILE* file = (SS_FILE*)malloc(sizeof(SS_FILE));
      char* filename = (char*)c.stack[c.parameter_level + 0];
      file->handle = VM_Interface->OpenFile(filename);
      c.registers[0] = (sdword)file;
      break;
    }

    case 0x1401: // void CloseFile(file f)
    {
      SS_FILE* file = (SS_FILE*)c.stack[c.parameter_level + 0];
      VM_Interface->CloseFile(file->handle);
      break;
    }

    case 0x1402: // void WriteFileInt(file f, string key, int value)
    {
      SS_FILE* file = (SS_FILE*)c.stack[c.parameter_level + 0];
      char* key = (char*)c.stack[c.parameter_level + 1];
      int value = c.stack[c.parameter_level + 2];
      WriteConfigInt(file->handle->config, "", key, value);
      break;
    }

    case 0x1403: // void WriteFileBool(file f, string key, bool value)
    {
      SS_FILE* file = (SS_FILE*)c.stack[c.parameter_level + 0];
      char* key = (char*)c.stack[c.parameter_level + 1];
      int value = c.stack[c.parameter_level + 2];
      WriteConfigInt(file->handle->config, "", key, value);
      break;
    }

    case 0x1404: // void WriteFileString(file f, string key, string value)
    {
      SS_FILE* file = (SS_FILE*)c.stack[c.parameter_level + 0];
      char* key = (char*)c.stack[c.parameter_level + 1];
      char* value = (char*)c.stack[c.parameter_level + 2];
      WriteConfigString(file->handle->config, "", key, value);
      break;
    }

    case 0x1405: // void WriteFileFloat(file f, string key, float value)
    {
      SS_FILE* file = (SS_FILE*)c.stack[c.parameter_level + 0];
      char* key = (char*)c.stack[c.parameter_level + 1];
      float value = *((float*)c.stack + c.parameter_level + 2);
      WriteConfigFloat(file->handle->config, "", key, value);
      break;
    }

    case 0x1406: // int ReadFileInt(file f, string key, int default)
    {
      SS_FILE* file = (SS_FILE*)c.stack[c.parameter_level + 0];
      char* key = (char*)c.stack[c.parameter_level + 1];
      int def = c.stack[c.parameter_level + 2];

      int result;
      ReadConfigInt(file->handle->config, "", key, &result, def);
      c.registers[0] = result;
      break;
    }

    case 0x1407: // bool ReadFileBool(file f, string key, bool default)
    {
      SS_FILE* file = (SS_FILE*)c.stack[c.parameter_level + 0];
      char* key = (char*)c.stack[c.parameter_level + 1];
      int def = c.stack[c.parameter_level + 2];

      int result;
      ReadConfigInt(file->handle->config, "", key, &result, def);
      c.registers[0] = result;
      break;
    }

    case 0x1408: // string ReadFileString(file f, string key, string default)
    {
      SS_FILE* file = (SS_FILE*)c.stack[c.parameter_level + 0];
      char* key = (char*)c.stack[c.parameter_level + 1];
      char* def = (char*)c.stack[c.parameter_level + 2];

      char* result = (char*)malloc(520);
      ReadConfigString(file->handle->config, "", key, result, 520, def);
      c.registers[0] = (sdword)result;
      break;
    }

    case 0x1409: // float ReadFileFloat(file f, string key, float default)
    {
      SS_FILE* file = (SS_FILE*)c.stack[c.parameter_level + 0];
      char* key = (char*)c.stack[c.parameter_level + 1];
      float def = *((float*)c.stack + c.parameter_level + 2);

      float result;
      ReadConfigFloat(file->handle->config, "", key, &result, def);
      *((float*)c.registers) = result;
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    case 0x1500: // void AddMenuItem(string item)
    {
      char* item = (char*)c.stack[c.parameter_level + 0];
      VM_Interface->AddMenuItem(item);
      break;
    }

    case 0x1501: // int VerticalMenu(int x, int y, int w, int h, int offset)
    {
      int x = c.stack[c.parameter_level + 0];
      int y = c.stack[c.parameter_level + 1];
      int w = c.stack[c.parameter_level + 2];
      int h = c.stack[c.parameter_level + 3];
      int o = c.stack[c.parameter_level + 4];
      c.registers[0] = VM_Interface->ExecuteMenuV(x, y, w, h, o);
      break;
    }

    case 0x1502: // int HorizontalMenu(int x, int y, int w, int h, int offset)
    {
      int x = c.stack[c.parameter_level + 0];
      int y = c.stack[c.parameter_level + 1];
      int w = c.stack[c.parameter_level + 2];
      int h = c.stack[c.parameter_level + 3];
      int o = c.stack[c.parameter_level + 4];
      c.registers[0] = VM_Interface->ExecuteMenuH(x, y, w, h, o);
      break;
    }

    case 0x1503: // void SetMenuPointer(string image_filename)
    {
      char* filename = (char*)c.stack[c.parameter_level + 0];
      VM_Interface->SetMenuPointer(filename);
      break;
    }

    ////////////////////////////////////////////////////////////////////////////

    default:
    {
      char message[520];
      sprintf(message, "Invalid syscall: 0x%X", ordinal);
      VM_Interface->ExitWithMessage(message);
      break;
    }
  }
}

////////////////////////////////////////////////////////////////////////////////

sdword*
CScript::GrabParameter(dword*& location, sdword* spare)
{
  const dword REFERENCE_BIT = 0x8000;
  ExecutionContext& c = contexts.top();

  sdword* p = NULL;
  
  do
  {
    dword operand_type = *location++;
    dword operand_code = *location++;

    sdword* q;
    switch (operand_type & ~REFERENCE_BIT) // without reference code
    {
      case 0x01: // register
        q = c.registers + operand_code;
        break;
      case 0x02: // literal
        q = spare;
        *spare = operand_code;
        break;
      case 0x04: // label
        q = (sdword*)code + operand_code;
        break;
      case 0x08: // local
        q = c.locals + operand_code;
        break;
      case 0x10: // parameter
        q = c.stack + c.parameter_level + operand_code;
        break;
      default:
        q = NULL;
        break;
    }

    if (operand_type & REFERENCE_BIT)  // if it's a reference, follow it
      p = (sdword*)(*p) + *q;
    else
      p = q;

  } while (*location & REFERENCE_BIT);

  return p;
}

////////////////////////////////////////////////////////////////////////////////

int
CScript::GetNumParameters(dword ordinal)
{
  for (int i = 0; i < g_NumSystemFunctions; i++)
    if (ordinal == (dword)g_SystemFunctions[i].ordinal)
    {
      const char* p = g_SystemFunctions[i].parameter_list;

      // if parameter list is empty, no parameters
      if (p[0] == 0)
        return 0;

      // count commas and add one
      int num_parameters = 1;
      while (*p)
      {
        if (*p == ',')
          num_parameters++;
        p++;
      }
      return num_parameters;
    }

  return 0;
}

////////////////////////////////////////////////////////////////////////////////

SS_OPCODE GetOpcode(int value)
{
  SS_OPCODE o = { 0, 0 };
  for (int i = 0; i < g_NumOpcodes; i++)
    if (value == (int)g_Opcodes[i].raw_value)
      return g_Opcodes[i];
  return o;
}

////////////////////////////////////////////////////////////////////////////////
