#ifndef ENGINE_HPP
#define ENGINE_HPP


#include "vm_interface.hpp"
#include "colorspace.h"
#include "sphere.hpp"
#include "smap.hpp"
#include "sspriteset.hpp"
#include "sfont.hpp"
#include "swindowstyle.hpp"
#include "spritesetserver.hpp"
#include "scriptserver.hpp"
#include "script.hpp"
#include "menu.hpp"
#include "audio.h"



class CGameEngine : private CVM_Interface
{
public:
  CGameEngine(const SSystemFiles* system_files);
  ~CGameEngine();

  void Run(const char* game);

private:

  // implementation of VM interface

  void Exit();
  void ExitWithMessage(const char* message);
  bool ShouldExit();

  void ChangeSong(const char* songfile);
  void PlayEffect(const char* effectfile);

  IMAGE LoadImage(const char* filename);

  void SetFont(const char* filename);
  SFONT* GetFont();

  void SetWindowStyle(const char* filename);
  SWINDOWSTYLE* GetWindowStyle();

  void AddFrameHook(int update_rate, const char* function);
  void RemoveFrameHook(const char* function);
  void MapEngine();
  void SetMap(const char* filename);
  SMAP* GetMap();

  void SetColorMask(RGBA mask, int num_frames);

  void SetDoodadFrame(int frame);
  void SetDoodadObstructive(bool obstructive);

  void ClearParty();
  void AddPartyCharacter(const char* filename);

  void PlayAnimation(const char* filename);

  DATAFILE* OpenFile(const char* filename);
  void CloseFile(DATAFILE* file);

  void AddMenuItem(const char* item);
  int ExecuteMenuV(int x, int y, int w, int h, int offset);
  int ExecuteMenuH(int x, int y, int w, int h, int offset);
  void SetMenuPointer(const char* filename);

private:

  // internal methods

  bool InitializeGame();
  void ShutdownGame();
  void RunGame();
  void ShowErrorScreen();

  void ME_ProcessFrameHooks();
  void ME_ProcessInput();
  void ME_CheckWarps();
  void ME_CheckTriggers();
  void ME_CheckDoodads();
  bool ME_Obstructed(const sSpriteset& ss, int mapx, int mapy, int maplayer);
  bool ME_ObstructedDoodad(int doodad, int map_x, int map_y);
  bool ME_ObstructedDoodads(int map_x, int map_y);

  void ME_InitColorMask();
  void ME_UpdateColorMask();

  void ME_RenderScreen();
  void ME_DrawLayer(int layer);
  void ME_PreDrawObjects(int layer);
  void ME_PostDrawObjects(int layer);
  void MapToScreen(int mapx, int mapy, int* screenx, int* screeny);

  void ME_InitParty();
  void ME_ResetParty();

  void ME_AddDoodads();
  void ME_RemoveDoodads();

  void ME_GetCenterCoordinates(int* cx, int* cy);

private:
  const SSystemFiles* m_SystemFiles;

  CSpritesetServer* m_SpritesetServer;
  CScriptServer *m_ScriptServer;

  IMAGE Arrow;
  IMAGE UpArrow;
  IMAGE DownArrow;

  SFONT        Font;
  SWINDOWSTYLE WindowStyle;

  bool ShouldExitEngine;
  char ExitEngineMessage[1024];

  CScript* GameScript;

  // rendering flags
  bool ShowFPS;
  bool FPSKeyPressed;

  sMenu Menu;

  // map data
  bool     MapLoaded;
  SMAP     Map;
  CScript* MapScript;

  bool OnTrigger;

  // background music
  MUSICMODULE Music;

  // sound effects
  int         CurrentEffect;
  SOUNDEFFECT SoundEffects[4];

  // color mask variables
  RGBA PreviousMask;
  RGBA CurrentMask;
  RGBA TargetMask;
  int  FadeTime;        // stored in frames
  int  CurrentFadeTime; // stored in frames, must be >= 0 and < FadeTime

  // frame hooks
  struct FrameHook
  {
    sString function;
    int update_rate;
    int current;
  };

  sVector<FrameHook> FrameHooks;

  struct CAMERA
  {
    // pixel coordinates
    int x;
    int y;
  } Camera;
  bool CameraTracking;

  struct PARTY
  {
    struct PARTYCHARACTER
    {
      int direction;  // 0 = north, 1 = northeast, 2 = east, etc.
      int mapx;       // pixel coordinates
      int mapy;       // ..
      int maplayer;   // z coordinate

      int nextswitch;
      int walkframe;
      SSPRITESET* spriteset;
    };

    struct PARTYCOORDINATES
    {
      int direction;  // 0 = north, 1 = northeast, 2 = east, etc.
      int mapx;       // pixel coordinates
      int mapy;       // ..
      int maplayer;   // z coordinate
    };

    int numcharacters;
    PARTYCHARACTER* characters;

    PARTYCOORDINATES location;
    PARTYCOORDINATES* train;
  } Party;

  struct DOODAD
  {
    sDoodadEntity* doodad;
    SSPRITESET* spriteset;
    int current_frame;
  };

  sVector<DOODAD> Doodads;
  int CurrentDoodad;
};


#endif
