#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "x++.hpp"
#include "configfile.h"

#include "engine.hpp"
#include "render.h"
#include "rendersort.hpp"
#include "inputx.h"
#include "animation.h"
#include "sfont.hpp"
#include "simage.hpp"

#include "input.h"
#include "timer.h"
#include "filesystem.h"


#define FPS 60


////////////////////////////////////////////////////////////////////////////////

CGameEngine::CGameEngine(const SSystemFiles* system_files)
: m_ScriptServer(NULL)
, m_SystemFiles(system_files)

, ShouldExitEngine(false)

, GameScript(NULL)

, MapLoaded(false)
, MapScript(NULL)
, OnTrigger(false)

, Music(INVALID_MUSICMODULE)

, CurrentEffect(0)
, CurrentDoodad(-1)
{
  m_SpritesetServer = new CSpritesetServer();
  m_ScriptServer = new CScriptServer(this);

  for (int i = 0; i < 4; i++)
    SoundEffects[i] = INVALID_SOUNDEFFECT;

  strcpy(ExitEngineMessage, "");
  ME_InitColorMask();
}

////////////////////////////////////////////////////////////////////////////////

CGameEngine::~CGameEngine()
{
  ME_RemoveDoodads();
  delete m_SpritesetServer;
  delete m_ScriptServer;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::Run(const char* game)
{
  EnterDirectory(game);

  // run the game engine
  if (InitializeGame())
  {
    RunGame();
    ShutdownGame();
  }

  LeaveDirectory();
  ShowErrorScreen();
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::Exit()
{
  ExitWithMessage("");
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ExitWithMessage(const char* message)
{
  strcpy(ExitEngineMessage, message);
  ShouldExitEngine = true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CGameEngine::ShouldExit()
{
  return ShouldExitEngine;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ChangeSong(const char* songfile)
{
  // destroy the old song
  if (Music != INVALID_MUSICMODULE)
  {
    DestroyMusicModule(Music);
    Music = INVALID_MUSICMODULE;
  }

  // if no song is specified, just return
  if (strlen(songfile) == 0)
    return;

  // play the song
  char songpath[520];
  sprintf(songpath, "music/%s", songfile);

  Music = LoadMusicModule(songpath);
  if (Music != INVALID_MUSICMODULE)
    StartMusicModule(Music);
  else
  {
    char str[520];
    sprintf(str, "Error: Could not load song '%s'", songfile);
    ExitWithMessage(str);
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::PlayEffect(const char* effectfile)
{
  // PlayEffect("") stops all sounds
  if (strlen(effectfile) == 0)
  {
    for (int i = 0; i < 4; i++)
      if (SoundEffects[i] != INVALID_SOUNDEFFECT)
      {
        DestroySoundEffect(SoundEffects[i]);
        SoundEffects[i] = INVALID_SOUNDEFFECT;
      }

    CurrentEffect = 0;

    return;
  }

  // destroy the currently playing sound
  if (SoundEffects[CurrentEffect] != INVALID_SOUNDEFFECT)
  {
    DestroySoundEffect(SoundEffects[CurrentEffect]);
    SoundEffects[CurrentEffect] = INVALID_SOUNDEFFECT;
  }

  // load the new sound
  char effectfilepath[512];
  sprintf(effectfilepath, "sounds/%s", effectfile);
  SoundEffects[CurrentEffect] = LoadSoundEffect(effectfilepath);
  if (SoundEffects[CurrentEffect] == INVALID_SOUNDEFFECT)
  {
    char msg[256];
    sprintf(msg, "Error: Could not load sound effect '%s'", effectfile);   
    ExitWithMessage(msg);
    return;
  }

  // begin playing it
  PlaySoundEffect(SoundEffects[CurrentEffect]);

  CurrentEffect = (CurrentEffect + 1) % 4;
}

////////////////////////////////////////////////////////////////////////////////

IMAGE
CGameEngine::LoadImage(const char* filename)
{
  char path[1024];
  sprintf(path, "images/%s", filename);
  IMAGE image = SLoadImage(path);
  if (image == NULL)
  {
    char message[1024];
    sprintf(message, "Could not load image '%s'", filename);
    ExitWithMessage(message);
  }
  return image;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::SetFont(const char* filename)
{
  char path[1024];
  sprintf(path, "fonts/%s", filename);
  if (!Font.Load(path))
  {
    char error_message[1024];
    sprintf(error_message, "Error: Could not load font '%s'", filename);
    ExitWithMessage(error_message);
  }
}

////////////////////////////////////////////////////////////////////////////////

SFONT*
CGameEngine::GetFont()
{
  return &Font;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::SetWindowStyle(const char* filename)
{
  char path[1024];
  sprintf(path, "windowstyles/%s", filename);
  if (!WindowStyle.Load(path))
  {
    char error_message[1024];
    sprintf(error_message, "Error: Could not load window style '%s'", filename);
    ExitWithMessage(error_message);
  }
}

////////////////////////////////////////////////////////////////////////////////

SWINDOWSTYLE*
CGameEngine::GetWindowStyle()
{
  return &WindowStyle;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::AddFrameHook(int update_rate, const char* function)
{
  FrameHook fh;
  fh.current = 0;
  fh.update_rate = update_rate;
  fh.function = function;
  FrameHooks.push_back(fh);
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::RemoveFrameHook(const char* function)
{
  for (int i = 0; i < FrameHooks.size(); i++)
    if (FrameHooks[i].function == function)
    {
      FrameHooks.remove(i);
      return;
    }
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::MapEngine()
{
  // rendering throttle/meter
  ShowFPS = false;
  FPSKeyPressed = false;
  dword frames_rendered = 0;
  dword last_fps = 0;
  dword last_update = 0;

  dword frames_skipped = 0;

  dword idealtime = GetTime() * FPS;

  ShouldExitEngine = false;
  while (!ShouldExitEngine)
  {
    dword actualtime = GetTime() * FPS;
    idealtime += 1000; // number of milliseconds in a second

    // RENDER PHASE

    if (ShowFPS)
    {
      dword old_frames_rendered = frames_rendered;

      while (actualtime < idealtime || frames_skipped >= 10)
      {
        ME_RenderScreen();
        frames_skipped = 0;

        // draw FPS
        char fps_string[80];
        sprintf(fps_string, "%d/60", last_fps);
        Font.SetColor(rgbaBlack);
        Font.DrawText(1, 1, fps_string);
        Font.SetColor(rgbaWhite);
        Font.DrawText(0, 0, fps_string);

        FlipScreen();
        frames_rendered++;

        actualtime = GetTime() * FPS;
      }

      if (frames_rendered == old_frames_rendered)
        frames_skipped++;

      if (GetTime() > last_update + 1000)
      {
        last_update = GetTime();
        last_fps = frames_rendered;
        frames_rendered = 0;
      }
    }
    else
    {
      if (actualtime < idealtime || frames_skipped >= 10)
      {
        frames_skipped = 0;
        ME_RenderScreen();
        FlipScreen();
        while (actualtime < idealtime)
          actualtime = GetTime() * FPS;
      }
      else
        frames_skipped++;
    }

    // UPDATE PHASE

    ME_ProcessFrameHooks();
    Map.UpdateAnimations();
    ME_UpdateColorMask();
    ME_ProcessInput();
    ME_CheckWarps();
    ME_CheckTriggers();
    ME_CheckDoodads();
  }

  ClearKeyQueue();
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::SetMap(const char* mapfile)
{
  // free the previous map
  if (MapLoaded)
  {
    ME_RemoveDoodads();

    if (MapScript)
      MapScript->ExecuteFunction("LeaveMap");
    m_ScriptServer->Free(MapScript);
    MapScript = NULL;

    MapLoaded = false;
  }

  // if mapfile is NULL or "", don't load a new map
  if (mapfile == NULL ||
      strlen(mapfile) == 0)
    return;

  char map_path[520];
  sprintf(map_path, "maps/%s", mapfile);

  // load the map
  if (Map.Load(map_path) == false)
  {
    char exitmessage[520];
    sprintf(exitmessage, "Error: Could not load map '%s'", mapfile);
    ExitWithMessage(exitmessage);
    return;
  }

  // make sure we have a valid base layer
  if (Map.GetMap().GetStartLayer() >= Map.GetMap().GetNumLayers())
    ExitWithMessage("Map has invalid start layer");

  int tile_width  = Map.GetTileset().GetTileWidth();
  int tile_height = Map.GetTileset().GetTileHeight();

  // set coordinates
  Party.location.mapx      = Map.GetMap().GetStartX();
  Party.location.mapy      = Map.GetMap().GetStartY();
  Party.location.maplayer  = Map.GetMap().GetStartLayer();
  Party.location.direction = Map.GetMap().GetStartDirection();

  CameraTracking = true;
  Camera.x       = Party.location.mapx;
  Camera.y       = Party.location.mapy;

  // put all characters at the party coordinates
  for (int i = 0; i < Party.numcharacters; i++)
  {
    Party.characters[i].direction  = Party.location.direction;
    Party.characters[i].mapx       = Party.location.mapx;
    Party.characters[i].mapy       = Party.location.mapy;
    Party.characters[i].maplayer   = Party.location.maplayer;
    Party.characters[i].nextswitch = 0;
    Party.characters[i].walkframe  = 0;
  }

  // put the train at the party coordinates
  for (int i = 0; i < Party.numcharacters * 16; i++)
  {
    Party.train[i].mapx      = Party.location.mapx;
    Party.train[i].mapy      = Party.location.mapy;
    Party.train[i].maplayer  = Party.location.maplayer;
    Party.train[i].direction = Party.location.direction;
  }

  // play music
  ChangeSong(Map.GetMap().GetMusicFile());
  if (ShouldExitEngine)
    return;

  // set up the doodads
  ME_AddDoodads();

  // load script
  if (strlen(Map.GetMap().GetScriptFile()))
  {
    char ScriptPath[512];
    sprintf(ScriptPath, "scripts/%s", Map.GetMap().GetScriptFile());
    MapScript = m_ScriptServer->Load(ScriptPath);
    if (MapScript == NULL)
    {
      char str[520];
      sprintf(str, "Error: Could not load script '%s' in map '%s'",
              Map.GetMap().GetScriptFile(), mapfile);
      ExitWithMessage(str);
      return;
    }
  }

  if (MapScript)
    MapScript->ExecuteFunction("EnterMap");

  MapLoaded = true;
}

////////////////////////////////////////////////////////////////////////////////

SMAP*
CGameEngine::GetMap()
{
  return &Map;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::SetColorMask(RGBA mask, int milliseconds)
{
  PreviousMask    = CurrentMask;
  TargetMask      = mask;
  FadeTime        = milliseconds * 60 / 1000;  // convert from milliseconds to frames
  CurrentFadeTime = 0;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::SetDoodadFrame(int frame)
{
  if (CurrentDoodad != -1)
    Doodads[CurrentDoodad].current_frame = frame;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::SetDoodadObstructive(bool obstructive)
{
  if (CurrentDoodad != -1)
    Doodads[CurrentDoodad].doodad->SetObstructive(obstructive);
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ClearParty()
{
  ME_ResetParty();
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::AddPartyCharacter(const char* spriteset)
{
  // add character
  SSPRITESET* ss = m_SpritesetServer->Load(spriteset);
  if (ss == NULL)
  {
    char exitmessage[520];
    sprintf(exitmessage, "Error: Could not load '%s'", spriteset);
    ExitWithMessage(exitmessage);
    return;
  }
  
  Party.characters = (PARTY::PARTYCHARACTER*)realloc(Party.characters, (Party.numcharacters + 1) * sizeof(PARTY::PARTYCHARACTER));
  Party.characters[Party.numcharacters].direction  = Party.location.direction;
  Party.characters[Party.numcharacters].mapx       = Party.location.mapx;
  Party.characters[Party.numcharacters].mapy       = Party.location.mapy;
  Party.characters[Party.numcharacters].maplayer   = Party.location.maplayer;
  Party.characters[Party.numcharacters].nextswitch = 0;
  Party.characters[Party.numcharacters].walkframe  = 0;
  Party.characters[Party.numcharacters].spriteset  = ss;

  // add to train
  Party.train = (PARTY::PARTYCOORDINATES*)realloc(Party.train, ((Party.numcharacters + 1) * 16) * sizeof(PARTY::PARTYCOORDINATES));

  for (int i = Party.numcharacters * 16; i < (Party.numcharacters + 1) * 16; i++)
  {
    if (i == 0)
    {
      Party.train[i].mapx      = Party.location.mapx;
      Party.train[i].mapy      = Party.location.mapy;
      Party.train[i].maplayer  = Party.location.maplayer;
      Party.train[i].direction = Party.location.direction;
    }
    else
      Party.train[i] = Party.train[i - 1];
  }

  Party.numcharacters++;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::PlayAnimation(const char* filename)
{
  char path[1024];
  sprintf(path, "anim/%s", filename);

  if (!::PlayAnimation(path))
  {
    char message[1024];
    sprintf(message, "Could not load animation '%s'", filename);
    ExitWithMessage(message);
  }
}

////////////////////////////////////////////////////////////////////////////////

DATAFILE*
CGameEngine::OpenFile(const char* filename)
{
  DATAFILE* file = new DATAFILE;
  file->config = new CONFIG;
  char path[520];
  sprintf(path, "save/%s", filename);
  file->filename = newstr(path);

  MakeDirectory("save");

  LoadConfig(file->config, path);
  return file;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::CloseFile(DATAFILE* file)
{
  SaveConfig(file->config, file->filename);
  DestroyConfig(file->config);
  delete file->config;
  delete[] file->filename;
  delete file;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::AddMenuItem(const char* item)
{
  Menu.AddItem(item);
}

////////////////////////////////////////////////////////////////////////////////

int
CGameEngine::ExecuteMenuV(int x, int y, int w, int h, int offset)
{
  MENUDISPLAYDATA mdd = {
    &Font,
    &WindowStyle,
    Arrow,
    UpArrow,
    DownArrow,
  };
  int selection = Menu.ExecuteV(mdd, x, y, w, h, offset);
  Menu.Clear();
  return selection;
}

////////////////////////////////////////////////////////////////////////////////

int
CGameEngine::ExecuteMenuH(int x, int y, int w, int h, int offset)
{
  MENUDISPLAYDATA mdd = {
    &Font,
    &WindowStyle,
    Arrow,
    UpArrow,
    DownArrow,
  };
  int selection = Menu.ExecuteH(mdd, x, y, w, h, offset);
  Menu.Clear();
  return selection;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::SetMenuPointer(const char* filename)
{
  DestroyImage(Arrow);

  char path[520];
  sprintf(path, "images/%s", filename);
  Arrow = SLoadImage(path);
  if (Arrow == NULL)
  {
    char message[1024];
    sprintf(message, "Error: Could not load image '%s'", filename);
    ExitWithMessage(message);
  }
}

////////////////////////////////////////////////////////////////////////////////

bool
CGameEngine::InitializeGame()
{
  CurrentMask.red   = 0;
  CurrentMask.green = 0;
  CurrentMask.blue  = 0;
  CurrentMask.alpha = 0;

  TargetMask.red    = 0;
  TargetMask.green  = 0;
  TargetMask.blue   = 0;
  TargetMask.alpha  = 0;

  // load the game.inf file
  char scriptname[520];
  ReadConfigFileString("game.inf", "", "script", scriptname, 512, "");

  // get the game resolution and set it
  int screen_width, screen_height;
  ReadConfigFileInt("game.inf", "", "screen_width", &screen_width, 320);
  ReadConfigFileInt("game.inf", "", "screen_height", &screen_height, 240);
  SwitchResolution(screen_width, screen_height);

  // load the default font, window style, and arrows
  Font.Load(m_SystemFiles->font);
  WindowStyle.Load(m_SystemFiles->window_style);
  Arrow = SLoadImage(m_SystemFiles->arrow);
  UpArrow = SLoadImage(m_SystemFiles->up_arrow);
  DownArrow = SLoadImage(m_SystemFiles->down_arrow);

  ME_InitParty();

  // scripts are in the "scripts" directory
  char scriptpath[520];
  sprintf(scriptpath, "scripts/%s", scriptname);

  // allocate the game script
  GameScript = m_ScriptServer->Load(scriptpath);
  if (GameScript == NULL)
  {
    char ErrorMessage[520];
    sprintf(ErrorMessage, "Error: Could not load game script '%s'", scriptname);
    ExitWithMessage(ErrorMessage);
    return false;
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ShutdownGame()
{
  // if arrow somehow got destroyed, and nobody loaded a new one...
  if (Arrow)
    DestroyImage(Arrow);

  ME_ResetParty();

  SetMap("");
  ChangeSong("");

  SwitchResolution(320, 240);

  m_ScriptServer->Free(GameScript);
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::RunGame()
{
  if (GameScript->ExecuteFunction("game") == false)
    ExitWithMessage("Error: Game script did not contain the function 'game'");
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ShowErrorScreen()
{
  // reset the font to the system one
  Font.Load(m_SystemFiles->font);

  // If an error occured, show the error screen
  if (ShouldExitEngine && strlen(ExitEngineMessage) > 0)
  {
    WhileAnyKeyPressed();

    do
    {
      ClearScreen();
      Font.DrawTextBox(0, 0, GetScreenWidth(), GetScreenHeight(), 0, ExitEngineMessage);
      FlipScreen();
      RefreshInput();
    }
    while (!AnyKeyPressed());

    ShouldExitEngine = false;
  }

  // wait until a key is not pressed
  WhileAnyKeyPressed();
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_ProcessFrameHooks()
{
  for (int i = 0; i < FrameHooks.size(); i++)
    if (++FrameHooks[i].current >= FrameHooks[i].update_rate)
    {
      GameScript->ExecuteFunction(FrameHooks[i].function.c_str());
      FrameHooks[i].current = 0;
    }  
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_ProcessInput()
{
  RefreshInput();

  if (KeyPressed(KEY_F1) && !FPSKeyPressed)
    ShowFPS = !ShowFPS;
  
  FPSKeyPressed = KeyPressed(KEY_F1);

  // Store backups of the current coordinates
  int oldmapx = Party.location.mapx;
  int oldmapy = Party.location.mapy;

  if (KeyPressed(KEY_LEFT))
    Party.location.mapx--;
  if (KeyPressed(KEY_RIGHT))
    Party.location.mapx++;
  if (KeyPressed(KEY_UP))
    Party.location.mapy--;
  if (KeyPressed(KEY_DOWN))
    Party.location.mapy++;
  if (KeyPressed(KEY_ESCAPE))
    ShouldExitEngine = true;

  int& mx = Party.location.mapx;
  int& my = Party.location.mapy;
  int& ox = oldmapx;
  int& oy = oldmapy;

  if (mx > ox) // East
  {
    if (my > oy)  Party.location.direction = 3; else // Southeast
    if (my == oy) Party.location.direction = 2; else // East
    if (my < oy)  Party.location.direction = 1;      // Northeast
  }
  else if (mx < ox) // West
  {
    if (my > oy)  Party.location.direction = 5; else // Southwest
    if (my == oy) Party.location.direction = 6; else // West
    if (my < oy)  Party.location.direction = 7;      // Northwest
  }
  else
  {
    if (my > oy) Party.location.direction = 4; else // South
    if (my < oy) Party.location.direction = 0;      // North
  }

  // obstruction checks
  if (Party.numcharacters > 0)  // people have to be in the party to have obstructions
  {
    const sSpriteset& spriteset = Party.characters[0].spriteset->GetSpriteset();
    int layer = Party.location.maplayer;

    // if the spriteset is currently obstructed, we must find a way to move it back
    if (ME_Obstructed(spriteset, mx, my, layer))
    {
      // motion is horizontal
      if (mx != ox && my == oy)
      {
        if (!ME_Obstructed(spriteset, mx, my - 1, layer))
          my--;
        else if (!ME_Obstructed(spriteset, mx, my + 1, layer))
          my++;
        else
        {
          mx = ox;
          my = oy;
        }
      }
      // motion is vertical
      else if (mx == ox && my != oy)
      {
        if (!ME_Obstructed(spriteset, mx - 1, my, layer))
          mx--;
        else if (!ME_Obstructed(spriteset, mx + 1, my, layer))
          mx++;
        else
        {
          mx = ox;
          my = oy;
        }
      }
      // motion is diagonal
      else if (mx != ox && my != oy)
      {
        if (!ME_Obstructed(spriteset, mx, oy, layer))
          my = oy;
        else if (!ME_Obstructed(spriteset, ox, my, layer))
          mx = ox;
        else
        {
          mx = ox;
          my = oy;
        }
      }
    }
  }

  // update train and walkframes
  if (mx != ox || my != oy)
  {
    // update train
    for (int i = Party.numcharacters * 16 - 1; i >= 0; i--)
    {
      if (i == 0)
      {
        Party.train[i].mapx      = Party.location.mapx;
        Party.train[i].mapy      = Party.location.mapy;
        Party.train[i].maplayer  = Party.location.maplayer;
        Party.train[i].direction = Party.location.direction;
      }
      else
        Party.train[i] = Party.train[i - 1];

      if (i % 16 == 0)
      {
        Party.characters[i / 16].mapx      = Party.train[i].mapx;
        Party.characters[i / 16].mapy      = Party.train[i].mapy;
        Party.characters[i / 16].maplayer  = Party.train[i].maplayer;
        Party.characters[i / 16].direction = Party.train[i].direction;
      }
    }

    // update walkframes
    for (int i = 0; i < Party.numcharacters; i++)
    {
      PARTY::PARTYCHARACTER* character = Party.characters + i;
      character->nextswitch--;
      if (character->nextswitch < 0)
      {
        character->walkframe = (character->walkframe + 1) % character->spriteset->GetNumFrames(character->direction);

        character->nextswitch = character->spriteset->GetFrameDelay(
          character->direction,
          character->walkframe);
      }
    }
  }

  if (CameraTracking)
  {
    Camera.x = Party.location.mapx;
    Camera.y = Party.location.mapy;
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_CheckWarps()
{
  const sMap& map = Map.GetMap();
  for (int i = 0; i < map.GetNumEntities(); i++)
  {
    if (map.GetEntity(i).GetEntityType() != sEntity::WARP)
      continue;

    int tile_width = map.GetTileset().GetTileWidth();
    int tile_height = map.GetTileset().GetTileHeight();

    const sWarpEntity& warp = (const sWarpEntity&)map.GetEntity(i);
    if (Party.location.mapx / tile_width == warp.GetX() / tile_width &&
        Party.location.mapy / tile_height == warp.GetY() / tile_height &&
        Party.location.maplayer == warp.GetLayer())
    {
      ME_RenderScreen();
      FadeOut(warp.GetFade());

      SetMap(warp.GetDestinationMap());

      ME_RenderScreen();
      FadeIn(warp.GetFade());
    }
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_CheckTriggers()
{
  bool on_trigger = false;

  const sMap& map = Map.GetMap();
  for (int i = 0; i < map.GetNumEntities(); i++)
  {
    if (map.GetEntity(i).GetEntityType() != sEntity::TRIGGER)
      continue;

    int tile_width = map.GetTileset().GetTileWidth();
    int tile_height = map.GetTileset().GetTileHeight();

    const sTriggerEntity& trigger = (const sTriggerEntity&)map.GetEntity(i);
    if (Party.location.mapx / tile_width == trigger.GetX() / tile_width &&
        Party.location.mapy / tile_height == trigger.GetY() / tile_height &&
        Party.location.maplayer == trigger.GetLayer())
    {
      if (!OnTrigger)
      {
        if (MapScript)
          MapScript->ExecuteFunction(trigger.GetFunction());
      }

      on_trigger = true;
    }
  }

  OnTrigger = on_trigger;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_CheckDoodads()
{
  // check DOOR activation
  for (int i = 0; i < Doodads.size(); i++)
    if (Doodads[i].doodad->GetActivationMethod() == sDoodadEntity::DOOR)
    {
      int ox = 1;
      int oy = 1;

      // ox and oy should be (the size of the base rectangle) / 2
      if (Party.numcharacters > 0)
      {
        const sSpriteset& ss = Party.characters[0].spriteset->GetSpriteset();
        int width = abs(ss.GetBaseX1() - ss.GetBaseX2());
        int height = abs(ss.GetBaseY1() - ss.GetBaseY2());
        ox = width / 2 + 2;  // I don't know why there needs to be a +2 on here
        oy = height / 2 + 2;
      }

      // check obstructions
      int mapx = Party.location.mapx;
      int mapy = Party.location.mapy;
      if (ME_ObstructedDoodad(i, mapx + ox, mapy + 0)  ||
          ME_ObstructedDoodad(i, mapx + ox, mapy + oy) ||
          ME_ObstructedDoodad(i, mapx + 0,  mapy + oy) ||
          ME_ObstructedDoodad(i, mapx - ox, mapy + oy) ||
          ME_ObstructedDoodad(i, mapx - ox, mapy + 0)  ||
          ME_ObstructedDoodad(i, mapx - ox, mapy - oy) ||
          ME_ObstructedDoodad(i, mapx + 0,  mapy - oy))
      {
        // activate the doodad
        CurrentDoodad = i;
        if (MapScript)
          MapScript->ExecuteFunction(Doodads[i].doodad->GetFunction());
        CurrentDoodad = -1;
      }
    }
}

////////////////////////////////////////////////////////////////////////////////

inline int min(int a, int b)
{
  return (a < b ? a : b);
}

inline int max(int a, int b)
{
  return (a > b ? a : b);
}

bool
CGameEngine::ME_Obstructed(const sSpriteset& ss, int mapx, int mapy, int maplayer)
{
  const sTileset& ts = Map.GetTileset();
  
  int base_min_x = min(ss.GetBaseX1(), ss.GetBaseX2());
  int base_max_x = max(ss.GetBaseX1(), ss.GetBaseX2());
  int base_min_y = min(ss.GetBaseY1(), ss.GetBaseY2());
  int base_max_y = max(ss.GetBaseY1(), ss.GetBaseY2());

  int hotspot_x = (base_max_x + base_min_x) / 2;
  int hotspot_y = (base_max_y + base_min_y) / 2;

  for (int ix = base_min_x; ix <= base_max_x; ix++)
    for (int iy = base_min_y; iy <= base_max_y; iy++)
    {
      int map_x = mapx + ix - hotspot_x;
      int map_y = mapy + iy - hotspot_y;

      int tx = map_x / ts.GetTileWidth();
      int ty = map_y / ts.GetTileHeight();
      const sLayer& layer = Map.GetMap().GetLayer(maplayer);
      if (tx > 0 && ty > 0 && tx < layer.GetWidth() && ty < layer.GetHeight())
      {
        const sTile& tile = ts.GetTile(layer.GetTile(tx, ty));

        int tile_offset_x = map_x % ts.GetTileWidth();
        int tile_offset_y = map_y % ts.GetTileHeight();

        if (tile.IsObstructed(tile_offset_x, tile_offset_y))
          return true;
      }

      // check if the area is obstructed by doodads
      if (ME_ObstructedDoodads(map_x, map_y))
        return true;
    }

  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool
CGameEngine::ME_ObstructedDoodad(int doodad, int map_x, int map_y)
{
  // now check if map_x and map_y are between the spriteset's base
  const sSpriteset& ss = Doodads[doodad].spriteset->GetSpriteset();
  int width  = abs(ss.GetBaseX1() - ss.GetBaseX2());
  int height = abs(ss.GetBaseY1() - ss.GetBaseY2());

  int x = Doodads[doodad].doodad->GetX();
  int y = Doodads[doodad].doodad->GetY();

  return (
    map_x >= x - width / 2 &&
    map_x <= x + width / 2 &&
    map_y >= y - height / 2 &&
    map_y <= y + height / 2
  );
}

////////////////////////////////////////////////////////////////////////////////

bool
CGameEngine::ME_ObstructedDoodads(int map_x, int map_y)
{
  for (int i = 0; i < Doodads.size(); i++)
    if (Doodads[i].doodad->IsObstructive())
      if (ME_ObstructedDoodad(i, map_x, map_y))
        return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_InitColorMask()
{
  PreviousMask.red   = 0;
  PreviousMask.green = 0;
  PreviousMask.blue  = 0;
  PreviousMask.alpha = 0;

  CurrentMask.red   = 0;
  CurrentMask.green = 0;
  CurrentMask.blue  = 0;
  CurrentMask.alpha = 0;

  TargetMask.red   = 0;
  TargetMask.green = 0;
  TargetMask.blue  = 0;
  TargetMask.alpha = 0;

  FadeTime        = 0;
  CurrentFadeTime = 0;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_UpdateColorMask()
{
  // if FadeTime is 0, we're not updating the mask
  if (FadeTime == 0)
    return;

  CurrentMask.red   = (PreviousMask.red   * (FadeTime - CurrentFadeTime)) + (TargetMask.red   * CurrentFadeTime) / FadeTime;
  CurrentMask.green = (PreviousMask.green * (FadeTime - CurrentFadeTime)) + (TargetMask.green * CurrentFadeTime) / FadeTime;
  CurrentMask.blue  = (PreviousMask.blue  * (FadeTime - CurrentFadeTime)) + (TargetMask.blue  * CurrentFadeTime) / FadeTime;
  CurrentMask.alpha = (PreviousMask.alpha * (FadeTime - CurrentFadeTime)) + (TargetMask.alpha * CurrentFadeTime) / FadeTime;

  CurrentFadeTime++;
  if (CurrentFadeTime > FadeTime - 1)
  {
    PreviousMask = TargetMask;
    CurrentMask = TargetMask;
    FadeTime = 0;
    CurrentFadeTime = 0;
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_RenderScreen()
{
  ClearScreen();

  for (int i = 0; i < Map.GetMap().GetNumLayers(); i++)
    ME_DrawLayer(i);
  
  ApplyColorMask(CurrentMask);
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_DrawLayer(int layer)
{
  ME_PreDrawObjects(layer);

  // get center of screen
  int cx, cy;
  ME_GetCenterCoordinates(&cx, &cy);

  int tile_width  = Map.GetTileset().GetTileWidth();
  int tile_height = Map.GetTileset().GetTileHeight();

  // get tile coordinates and offset coordinates
  int tx = cx / tile_width;
  int ty = cy / tile_height;
  int ox = cx % tile_width;
  int oy = cy % tile_height;

  int half_width = GetScreenWidth() / 2;
  int half_height = GetScreenHeight() / 2;

  // draw layer!
  for (int ix = -half_width / tile_width - 2; ix < half_width / tile_width + 2; ix++)
    for (int iy = -half_height / tile_height - 2; iy < half_height / tile_height + 2; iy++)
    {
      if ((tx + ix) >= 0 &&
          (ty + iy) >= 0 &&
          (tx + ix) < Map.GetMap().GetLayer(layer).GetWidth() &&
          (ty + iy) < Map.GetMap().GetLayer(layer).GetHeight())
      {
        int tile = Map.GetMap().GetLayer(layer).GetTile(tx + ix, ty + iy);
        BlitImage(Map.GetTileImage(tile),
                  ix * tile_width  + half_width  - ox,
                  iy * tile_height + half_height - oy);
      }
    }

  ME_PostDrawObjects(layer);
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_PreDrawObjects(int layer)
{
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_PostDrawObjects(int layer)
{
  CRenderSort rs;

  // add party characters
  for (int i = 0; i < Party.numcharacters; i++)
  {
    PARTY::PARTYCHARACTER* c = Party.characters + i;
    if (c->maplayer == layer)
    {
      // calculate the offset of the hotspot in the sprite
      const sSpriteset& spriteset = c->spriteset->GetSpriteset();
      int offset_x = abs(spriteset.GetBaseX1() + spriteset.GetBaseX2()) / 2;
      int offset_y = abs(spriteset.GetBaseY1() + spriteset.GetBaseY2()) / 2;

      // calculate the screen location of the object
      int base_x, base_y;
      MapToScreen(c->mapx, c->mapy, &base_x, &base_y);

      IMAGE image = c->spriteset->GetFrame(c->direction, c->walkframe);
      rs.AddObject(base_x, base_y, offset_x, offset_y, image);
    }
  }

  // add doodads
  for (int i = 0; i < Doodads.size(); i++)
    if (Doodads[i].doodad->GetLayer() == layer)
    {
      // calculate the offset of the hotspot in the sprite
      const sSpriteset& spriteset = Doodads[i].spriteset->GetSpriteset();
      int offset_x = abs(spriteset.GetBaseX1() + spriteset.GetBaseX2()) / 2;
      int offset_y = abs(spriteset.GetBaseY1() + spriteset.GetBaseY2()) / 2;

      // calculate the screen location of the object
      int base_x, base_y;
      MapToScreen(Doodads[i].doodad->GetX(), Doodads[i].doodad->GetY(), &base_x, &base_y);

      IMAGE image = Doodads[i].spriteset->GetFrame(0, Doodads[i].current_frame);
      rs.AddObject(base_x, base_y, offset_x, offset_y, image);
    }
  
  rs.DrawObjects();
}

////////////////////////////////////////////////////////////////////////////////

// In:  map coordinates (in pixels)
// Out: screen coordinates (in pixels)
void
CGameEngine::MapToScreen(int mapx, int mapy, int* screenx, int* screeny)
{
  *screenx = GetScreenWidth() / 2;
  *screeny = GetScreenHeight() / 2;

  int cx, cy;
  ME_GetCenterCoordinates(&cx, &cy);

  *screenx -= (cx - mapx);
  *screeny -= (cy - mapy);
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_InitParty()
{
  Party.numcharacters = 0;
  Party.characters    = NULL;
  Party.train         = NULL;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_ResetParty()
{
  for (int i = 0; i < Party.numcharacters; i++)
    m_SpritesetServer->Free(Party.characters[i].spriteset);

  Party.numcharacters = 0;
  Party.characters = (PARTY::PARTYCHARACTER*)realloc(Party.characters, 0);

  Party.train = (PARTY::PARTYCOORDINATES*)realloc(Party.train, 0);

  Party.location.direction = 0;
  Party.location.mapx      = 0;
  Party.location.mapy      = 0;
  Party.location.maplayer  = 0;
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_AddDoodads()
{
  sMap& map = Map.GetMap();
  for (int i = 0; i < map.GetNumEntities(); i++)
    if (map.GetEntity(i).GetEntityType() == sEntity::DOODAD)
    {
      DOODAD d;
      d.current_frame = 0;
      d.doodad = (sDoodadEntity*)&map.GetEntity(i);
      d.spriteset = m_SpritesetServer->Load(d.doodad->GetSpritesetFile());
      if (d.spriteset == NULL)
      {
        char message[1024];
        sprintf(message, "Error: Cannot load spriteset '%s'", d.doodad->GetSpritesetFile());
        ExitWithMessage(message);
        return;
      }
      Doodads.push_back(d);
    }
}

////////////////////////////////////////////////////////////////////////////////

void
CGameEngine::ME_RemoveDoodads()
{
  for (int i = 0; i < Doodads.size(); i++)
    m_SpritesetServer->Free(Doodads[i].spriteset);
  Doodads.clear();
}

////////////////////////////////////////////////////////////////////////////////

// Returns the map coordinates (in pixels, not tiles) that are the center of the screen
void
CGameEngine::ME_GetCenterCoordinates(int* cx, int* cy)
{
  *cx = Camera.x;
  *cy = Camera.y;

  int tile_width  = Map.GetTileset().GetTileWidth();
  int tile_height = Map.GetTileset().GetTileHeight();
  int half_width  = GetScreenWidth() / 2;
  int half_height = GetScreenHeight() / 2;
  if (*cx < half_width)
    *cx = half_width;
  if (*cy < half_height)
    *cy = half_height;

  if (*cx > Map.GetMap().GetLayer(Party.location.maplayer).GetWidth() * tile_width - half_width)
    *cx = Map.GetMap().GetLayer(Party.location.maplayer).GetWidth() * tile_width - half_width;

  if (*cy > Map.GetMap().GetLayer(Party.location.maplayer).GetHeight() * tile_height - half_height)
    *cy = Map.GetMap().GetLayer(Party.location.maplayer).GetHeight() * tile_height - half_height;
}

////////////////////////////////////////////////////////////////////////////////
