#include "TilesetView.hpp"
#include "NumberDialog.hpp"
#include "TilePropertiesDialog.hpp"
#include "x++.hpp"
#include "resource.h"


const int UPDATE_TILEVIEW_TIMER = 200;


static int s_iTilesetViewID = 1000;


BEGIN_MESSAGE_MAP(CTilesetView, CWnd)

  ON_WM_PAINT()
  ON_WM_SIZE()
  ON_WM_VSCROLL()
  ON_WM_LBUTTONDOWN()
  ON_WM_MOUSEMOVE()
  ON_WM_RBUTTONUP()

  ON_COMMAND(ID_TILESETVIEW_INSERTTILE,  OnInsertTile)
  ON_COMMAND(ID_TILESETVIEW_APPENDTILE,  OnAppendTile)
  ON_COMMAND(ID_TILESETVIEW_DELETETILE,  OnDeleteTile)
  ON_COMMAND(ID_TILESETVIEW_PROPERTIES,  OnTileProperties)
  ON_COMMAND(ID_TILESETVIEW_INSERTTILES, OnInsertTiles)
  ON_COMMAND(ID_TILESETVIEW_APPENDTILES, OnAppendTiles)
  ON_COMMAND(ID_TILESETVIEW_DELETETILES, OnDeleteTiles)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CTilesetView::CTilesetView()
: m_StatusBar(NULL)
, m_Handler(NULL)

, m_Tileset(NULL)

, m_TopRow(0)
, m_SelectedTile(0)

, m_BlitTile(NULL)

, m_MenuShown(false)
{
}

////////////////////////////////////////////////////////////////////////////////

CTilesetView::~CTilesetView()
{
  delete m_BlitTile;
}

////////////////////////////////////////////////////////////////////////////////

BOOL
CTilesetView::Create(CStatusBar* status_bar, CTilesetViewHandler* handler, CWnd* parent, sTileset* tileset)
{
  m_StatusBar = status_bar;
  m_Handler = handler;
  m_Tileset = tileset;
  
  m_BlitTile = new CDIBSection(m_Tileset->GetTileWidth(), m_Tileset->GetTileHeight(), 32);

  BOOL retval = CWnd::Create(
    AfxRegisterWndClass(0, LoadCursor(NULL, IDC_ARROW), NULL, NULL),
    "",
    WS_CHILD | WS_VISIBLE | WS_VSCROLL,
    CRect(0, 0, 0, 0),
    parent,
    s_iTilesetViewID++);

  UpdateScrollBar();

  return retval;
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetView::TileChanged(int tile)
{
  RECT client_rect;
  GetClientRect(&client_rect);
  int num_tiles_x = client_rect.right / m_Tileset->GetTileWidth();
  if (num_tiles_x == 0)
    return;

  int col = tile % num_tiles_x;
  int row = tile / num_tiles_x;
  
  int x = col * m_Tileset->GetTileWidth();
  int y = (row - m_TopRow) * m_Tileset->GetTileHeight();

  RECT rect;
  SetRect(&rect, x, y, x + m_Tileset->GetTileWidth(), y + m_Tileset->GetTileHeight());
  InvalidateRect(&rect);
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetView::TilesetChanged()
{
  delete m_BlitTile;
  m_BlitTile = new CDIBSection(m_Tileset->GetTileWidth(), m_Tileset->GetTileHeight(), 32);

  Invalidate();
  UpdateScrollBar();
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetView::SetSelectedTile(int tile)
{
  m_SelectedTile = tile;
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

int
CTilesetView::GetSelectedTile() const
{
  return m_SelectedTile;
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetView::UpdateScrollBar()
{
  SCROLLINFO si;
  si.cbSize = sizeof(si);
  si.fMask  = SIF_ALL;
  si.nMin   = 0;

  if (GetPageSize() < GetNumRows())
  {
    si.nMax   = GetNumRows() - 1;
    si.nPage  = GetPageSize();
    si.nPos   = m_TopRow;
  }
  else
  {
    si.nMax   = 0xFFFF;
    si.nPage  = 0xFFFE;
    si.nPos   = 0;
  }

  SetScrollInfo(SB_VERT, &si);
}

////////////////////////////////////////////////////////////////////////////////

int
CTilesetView::GetPageSize()
{
  RECT ClientRect;
  GetClientRect(&ClientRect);
  return ClientRect.bottom / m_Tileset->GetTileHeight();
}

////////////////////////////////////////////////////////////////////////////////

int
CTilesetView::GetNumRows()
{
  RECT client_rect;
  GetClientRect(&client_rect);
  int num_tiles_x = client_rect.right / m_Tileset->GetTileWidth();

  if (num_tiles_x == 0)
    return -1;
  else
    return (m_Tileset->GetNumTiles() + num_tiles_x - 1) / num_tiles_x;
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnPaint()
{
  CPaintDC dc(this);
  
  RECT client_rect;
  GetClientRect(&client_rect);

  for (int iy = 0; iy < client_rect.bottom / m_Tileset->GetTileHeight() + 1; iy++)
    for (int ix = 0; ix < client_rect.right / m_Tileset->GetTileWidth() + 1; ix++)
    {
      RECT Rect = {
        ix * m_Tileset->GetTileWidth(),
        iy * m_Tileset->GetTileHeight(),
        (ix + 1) * m_Tileset->GetTileWidth(),
        (iy + 1) * m_Tileset->GetTileHeight(),
      };
      if (dc.RectVisible(&Rect) == FALSE)
        continue;
      
      int num_tiles_x = client_rect.right / m_Tileset->GetTileWidth();

      int it = (iy + m_TopRow) * (client_rect.right / m_Tileset->GetTileWidth()) + ix;
      if (ix < num_tiles_x && it < m_Tileset->GetNumTiles())
      {
        // draw the tile
        // fill the DIB section
        BGRA* pixels = (BGRA*)m_BlitTile->GetPixels();
        
        // make a checkerboard
        for (int ix = 0; ix < m_Tileset->GetTileWidth(); ix++)
          for (int iy = 0; iy < m_Tileset->GetTileHeight(); iy++)
          {
            if ((ix / 8 + iy / 8) % 2)
              pixels[iy * m_Tileset->GetTileWidth() + ix] = bgraWhite;
            else
            {
              BGRA light_blue = { 0xFF, 0xC0, 0xC0, 0xFF };
              pixels[iy * m_Tileset->GetTileWidth() + ix] = light_blue;
            }
          }        

        // draw the tile into it
        RGBA* tilepixels = m_Tileset->GetTile(it).GetPixels();
        for (int ix = 0; ix < m_Tileset->GetTileWidth(); ix++)
          for (int iy = 0; iy < m_Tileset->GetTileHeight(); iy++)
          {
            int alpha = tilepixels[iy * m_Tileset->GetTileWidth() + ix].alpha;
            pixels[iy * m_Tileset->GetTileWidth() + ix].red   = (tilepixels[iy * m_Tileset->GetTileWidth() + ix].red   * alpha + pixels[iy * m_Tileset->GetTileWidth() + ix].red   * (255 - alpha)) / 256;
            pixels[iy * m_Tileset->GetTileWidth() + ix].green = (tilepixels[iy * m_Tileset->GetTileWidth() + ix].green * alpha + pixels[iy * m_Tileset->GetTileWidth() + ix].green * (255 - alpha)) / 256;
            pixels[iy * m_Tileset->GetTileWidth() + ix].blue  = (tilepixels[iy * m_Tileset->GetTileWidth() + ix].blue  * alpha + pixels[iy * m_Tileset->GetTileWidth() + ix].blue  * (255 - alpha)) / 256;
          }
        
        // blit the tile
        CDC* tile = CDC::FromHandle(m_BlitTile->GetDC());
        dc.BitBlt(Rect.left, Rect.top, Rect.right - Rect.left, Rect.bottom - Rect.top, tile, 0, 0, SRCCOPY);

        // if the tile is selected, draw a pink rectangle around it
        if (it == m_SelectedTile)
        {
          HBRUSH newbrush = (HBRUSH)GetStockObject(NULL_BRUSH);
          CBrush* oldbrush = dc.SelectObject(CBrush::FromHandle(newbrush));
          HPEN newpen = (HPEN)CreatePen(PS_SOLID, 1, RGB(0xFF, 0x00, 0xFF));
          CPen* oldpen = dc.SelectObject(CPen::FromHandle(newpen));

          dc.Rectangle(&Rect);

          dc.SelectObject(oldbrush);
          DeleteObject(newbrush);
          dc.SelectObject(oldpen);
          DeleteObject(newpen);
        }

      }
      else
      {
        // draw black rectangle
        dc.FillRect(&Rect, CBrush::FromHandle((HBRUSH)GetStockObject(BLACK_BRUSH)));
      }
      
    }

}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnSize(UINT type, int cx, int cy)
{
  if (cx > 0)
  {
    // if the current top row is greater than the total number of rows minus the page size
    if (m_TopRow > GetNumRows() - GetPageSize())
    {
      // move the top row up
      m_TopRow = GetNumRows() - GetPageSize();
      if (m_TopRow < 0)
        m_TopRow = 0;
      Invalidate();
    }
  }

  // reflect the changes
  UpdateScrollBar();
  Invalidate();

  CWnd::OnSize(type, cx, cy);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnVScroll(UINT code, UINT pos, CScrollBar* scroll_bar)
{
  switch (code)
  {
    case SB_LINEDOWN:   m_TopRow++;                break;
    case SB_LINEUP:     m_TopRow--;                break;
    case SB_PAGEDOWN:   m_TopRow += GetPageSize(); break;
    case SB_PAGEUP:     m_TopRow -= GetPageSize(); break;
    case SB_THUMBTRACK: m_TopRow = (int)pos;       break;
  }

  // validate the values
  if (m_TopRow < 0)
    m_TopRow = 0;
  if (m_TopRow > GetNumRows() - GetPageSize())
    m_TopRow = GetNumRows() - GetPageSize();

  UpdateScrollBar();
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnLButtonDown(UINT flags, CPoint point)
{
  if (m_MenuShown)
    return;

  RECT client_rect;
  GetClientRect(&client_rect);
  int num_tiles_x = client_rect.right / m_Tileset->GetTileWidth();

  int col = point.x / m_Tileset->GetTileWidth();
  int row = point.y / m_Tileset->GetTileHeight();

  int tile = (m_TopRow + row) * num_tiles_x + col;

  if (tile >= 0 && tile < m_Tileset->GetNumTiles())
    m_SelectedTile = tile;

  Invalidate();

  // the selected tile changed, so tell the parent window
  m_Handler->TV_SelectedTileChanged(m_SelectedTile);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnMouseMove(UINT flags, CPoint point)
{
  RECT client_rect;
  GetClientRect(&client_rect);
  int num_tiles_x = client_rect.right / m_Tileset->GetTileWidth();

  int x = point.x / m_Tileset->GetTileWidth();
  int y = point.y / m_Tileset->GetTileHeight();

  int tile = (m_TopRow + y) * num_tiles_x + x;

  if (tile <= m_Tileset->GetNumTiles() -1)
  {
    CString tilenum;
    tilenum.Format("Tile (%i/%i)", tile, m_Tileset->GetNumTiles());
    if (m_StatusBar)
      m_StatusBar->SetWindowText(tilenum);
  }
  else
    if (m_StatusBar)
      m_StatusBar->SetWindowText("");

}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnRButtonUp(UINT flags, CPoint point)
{
  if (m_MenuShown)
    return;

  // select the tile
  OnLButtonDown(flags, point);

  // show pop-up menu
  ClientToScreen(&point);
  
  HMENU menu_ = LoadMenu(AfxGetApp()->m_hInstance, MAKEINTRESOURCE(IDR_TILESETVIEW));
  HMENU menu = GetSubMenu(menu_, 0);

  m_MenuShown = true;
  TrackPopupMenu(menu, TPM_LEFTALIGN | TPM_TOPALIGN, point.x, point.y, 0, m_hWnd, NULL);
  m_MenuShown = false;

  DestroyMenu(menu_);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnInsertTile()
{
  m_Tileset->InsertTiles(m_SelectedTile, 1);
  m_Handler->TV_TilesetChanged();
  UpdateScrollBar();
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnAppendTile()
{
  m_Tileset->AppendTiles(1);
  m_Handler->TV_TilesetChanged();
  UpdateScrollBar();
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnDeleteTile()
{
  m_Tileset->DeleteTiles(m_SelectedTile, 1);
  m_Handler->TV_TilesetChanged();
  UpdateScrollBar();
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnTileProperties()
{
  CTilePropertiesDialog dialog(m_Tileset, m_SelectedTile);
  if (dialog.DoModal() == IDOK)
  {
    m_Handler->TV_TilesetChanged();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnInsertTiles()
{
  CNumberDialog dialog("Insert Tiles", "Number of Tiles", 1, 1, 255);
  if (dialog.DoModal() == IDOK)
  {
    m_Tileset->InsertTiles(m_SelectedTile, dialog.GetValue());
    m_Handler->TV_TilesetChanged();
    UpdateScrollBar();
    Invalidate();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnAppendTiles()
{
  CNumberDialog dialog("Append Tiles", "Number of Tiles", 1, 1, 255);
  if (dialog.DoModal() == IDOK)
  {
    m_Tileset->AppendTiles(dialog.GetValue());
    m_Handler->TV_TilesetChanged();
    UpdateScrollBar();
    Invalidate();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetView::OnDeleteTiles()
{
  CNumberDialog dialog("Delete Tiles", "Number of Tiles", 1, 1, m_Tileset->GetNumTiles() - m_SelectedTile - 1);
  if (dialog.DoModal() == IDOK)
  {
    if (m_SelectedTile >= m_Tileset->GetNumTiles())
      m_SelectedTile = m_Tileset->GetNumTiles() - 1;

    m_Tileset->DeleteTiles(m_SelectedTile, dialog.GetValue());
    m_Handler->TV_SelectedTileChanged(m_SelectedTile);
    m_Handler->TV_TilesetChanged();
    UpdateScrollBar();
    Invalidate();
  }
}

////////////////////////////////////////////////////////////////////////////////
