#include "TilePreviewPalette.hpp"
#include "StateServer.hpp"

#define ID_TILE_PLAY  100015
#define ID_TILE_STOP  100016
#define ID_TILE_TIMER 100017

BEGIN_MESSAGE_MAP(CTilePreviewPalette, CMiniFrameWnd)

  ON_WM_CLOSE()
  ON_WM_SIZE()
  ON_WM_PAINT()
  ON_WM_TIMER()

  ON_COMMAND(ID_TILE_PLAY, OnPlay)
  ON_COMMAND(ID_TILE_STOP, OnStop)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CTilePreviewPalette::CTilePreviewPalette(CWnd* parent, sTileset* tileset)
: m_Created(false)
, m_AspectRatio(1)
, m_TileNum(0)
, m_AnimTileNum(0)
, m_DelayCounter(0)
, m_TileDelay(0)
{
  m_Tileset = tileset;
  m_DrawBitmap = new CDIBSection(tileset->GetTileWidth(), tileset->GetTileHeight(), 32);

  int left   = GetStateServer()->GetInt("TilePreviewPalette:left",   -1);
  int top    = GetStateServer()->GetInt("TilePreviewPalette:top",    -1);
  int right  = GetStateServer()->GetInt("TilePreviewPalette:right",  -1);
  int bottom = GetStateServer()->GetInt("TilePreviewPalette:bottom", -1);

  CRect start_rect(left, top, right, bottom);

  if (left == -1 && top == -1 && right == -1 && bottom == -1)
    start_rect = CRect(0, 0, 80, 80);

  Create(
    AfxRegisterWndClass(0, LoadCursor(NULL, MAKEINTRESOURCE(IDC_ARROW)), NULL, NULL),
    "tile preview",
    WS_POPUP | WS_CAPTION | WS_VISIBLE | WS_SIZEBOX | MFS_SYNCACTIVE | MFS_THICKFRAME | MFS_BLOCKSYSMENU,
    start_rect,
    parent);

  CFont* pFont = CFont::FromHandle((HFONT)GetStockObject(DEFAULT_GUI_FONT));
  m_PlayButton.Create("Play", WS_CHILD | WS_VISIBLE, CRect(0,0,10,10), this, ID_TILE_PLAY);
  m_PlayButton.SetFont(pFont);
  m_StopButton.Create("Stop", WS_CHILD | WS_VISIBLE, CRect(10,10,10,10), this, ID_TILE_STOP);
  m_StopButton.SetFont(pFont);
  m_StopButton.EnableWindow(false);
  m_Created = true;
  
  m_TileDelay = m_Tileset->GetTile(m_AnimTileNum).GetDelay();

  RECT rect;
  GetClientRect(&rect);
  OnSize(0, rect.right, rect.bottom);
}

////////////////////////////////////////////////////////////////////////////////

CTilePreviewPalette::~CTilePreviewPalette()
{
  delete m_DrawBitmap;
}

////////////////////////////////////////////////////////////////////////////////

void
CTilePreviewPalette::SetTile(int tile)
{
  if (m_TileNum != tile)
  {
    m_TileNum = tile;
    m_AnimTileNum = tile;
    m_TileDelay = m_Tileset->GetTile(tile).GetDelay();
    if (!m_TileDelay) m_TileDelay = 1;
    m_DelayCounter = 0;

    Invalidate();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilePreviewPalette::OnPlay()
{
  SetTimer(ID_TILE_TIMER, 17, NULL);

  m_PlayButton.EnableWindow(false);
  m_StopButton.EnableWindow(true);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilePreviewPalette::OnStop()
{
  KillTimer(ID_TILE_TIMER);

  m_PlayButton.EnableWindow(true);
  m_StopButton.EnableWindow(false);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilePreviewPalette::OnClose()
{
  RECT rect;
  GetWindowRect(&rect);
  GetStateServer()->SetInt("TilePreviewPalette:left",   rect.left);
  GetStateServer()->SetInt("TilePreviewPalette:top",    rect.top);
  GetStateServer()->SetInt("TilePreviewPalette:right",  rect.right);
  GetStateServer()->SetInt("TilePreviewPalette:bottom", rect.bottom);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilePreviewPalette::OnSize(UINT type, int cx, int cy)
{
  if (m_PlayButton.m_hWnd)
  {
    m_PlayButton.MoveWindow(CRect(0, cy - 30, cx / 2, cy));
    m_StopButton.MoveWindow(CRect(cx / 2, cy - 30, cx, cy));
  }

  int AR_x, AR_y;
  int PrevAR = m_AspectRatio;
  AR_x = cx / m_Tileset->GetTileWidth();
  AR_y = (cy - 30) / m_Tileset->GetTileHeight();
  m_AspectRatio = (AR_x < AR_y ? AR_x : AR_y);
  if (!m_AspectRatio) m_AspectRatio = 1;

  if (PrevAR != m_AspectRatio)
    Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilePreviewPalette::OnPaint()
{
  CPaintDC dc(this);
  
  RECT rect;
  CBrush SysBrush;
  GetClientRect(&rect);
  SysBrush.CreateSysColorBrush(NULL);

  if ((rect.right - rect.left) > (m_Tileset->GetTileWidth() * m_AspectRatio))
  {
    RECT window = rect;
    window.bottom -= 30;

    // left
    window.left = 0;
    window.right = ((rect.right - rect.left) - (m_Tileset->GetTileWidth() * m_AspectRatio)) / 2;
    dc.FillRect(&window, &SysBrush);

    // right
    window.left = ((rect.right - rect.left) + (m_Tileset->GetTileWidth() * m_AspectRatio)) / 2;
    window.right = rect.right - rect.left;
    dc.FillRect(&window, &SysBrush);
  }
  if ((rect.bottom - rect.top - 30) > (m_Tileset->GetTileHeight() * m_AspectRatio))
  {
    RECT window = rect;
    window.left = ((rect.right - rect.left) - (m_Tileset->GetTileWidth() * m_AspectRatio)) / 2;
    window.right = ((rect.right - rect.left) + (m_Tileset->GetTileWidth() * m_AspectRatio)) / 2;

    // top
    window.top = 0;
    window.bottom = ((rect.bottom - rect.top - 30) - (m_Tileset->GetTileHeight() * m_AspectRatio)) / 2;
    dc.FillRect(&window, &SysBrush);

    // bottom
    window.top = ((rect.bottom - rect.top - 30) + (m_Tileset->GetTileHeight() * m_AspectRatio)) / 2;
    window.bottom = rect.bottom - rect.top - 30;
    dc.FillRect(&window, &SysBrush);
  }
  SysBrush.DeleteObject();

  int bmp_width = m_Tileset->GetTileWidth();
  for (int j=0; j<m_Tileset->GetTileHeight(); j++)
    for (int i=0; i<m_Tileset->GetTileWidth(); i++)
    {
      BGRA* dest = (BGRA*)m_DrawBitmap->GetPixels();
      RGBA* src = m_Tileset->GetTile(m_AnimTileNum).GetPixels();
      int alpha = src[j * bmp_width + i].alpha;

      dest[j * bmp_width + i].red = (src[j * bmp_width + i].red * alpha + src[j * bmp_width + i].blue * (255 - alpha)) / 256;
      dest[j * bmp_width + i].green = (src[j * bmp_width + i].green * alpha + src[j * bmp_width + i].green * (255 - alpha)) / 256;
      dest[j * bmp_width + i].blue = (src[j * bmp_width + i].blue * alpha + src[j * bmp_width + i].blue * (255 - alpha)) / 256;
    }

  //dc.FillRect(&rect, &SysBrush);
  //dc.BitBlt(0,0, rect.right - rect.left, rect.bottom - rect.top - 30, CDC::FromHandle(m_DrawBitmap->GetDC()), 0, 0, SRCCOPY);
  dc.StretchBlt(
    ((rect.right - rect.left) -  (m_Tileset->GetTileWidth() * m_AspectRatio)) / 2, 
    ((rect.bottom - rect.top) - (m_Tileset->GetTileHeight() * m_AspectRatio) - 30) / 2, 
    m_Tileset->GetTileWidth() * m_AspectRatio, 
    m_Tileset->GetTileHeight() * m_AspectRatio, 
    CDC::FromHandle(m_DrawBitmap->GetDC()), 
    0, 0, m_DrawBitmap->GetWidth(), m_DrawBitmap->GetHeight(), SRCCOPY);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilePreviewPalette::OnTimer(UINT nIDEvent)
{
  m_DelayCounter++;

  if (m_DelayCounter >= m_TileDelay)
  {
    if (m_Tileset->GetTile(m_AnimTileNum).IsAnimated())
    {
      m_AnimTileNum = m_Tileset->GetTile(m_AnimTileNum).GetNextTile();
      m_TileDelay = m_Tileset->GetTile(m_AnimTileNum).GetDelay();
      if (!m_TileDelay) m_TileDelay = 1;
      m_DelayCounter = 0;

      Invalidate();
    }
  }
}

////////////////////////////////////////////////////////////////////////////////