/////////////////////////////////////////////////////////////////////////////
// Copyright (C) 1998 by Juraj Rojko jrojko@twist.cz
// All rights reserved
//
// VBScriptEditorView.h : interface of the CVBScriptEditorView class
//
/////////////////////////////////////////////////////////////////////////////

/* 
   16th April 2000
   Code cleaned up and modified by Darklich for use with SDE 
   (Sphere Development Environment)
*/


#include "ScriptView.hpp"
#include "..\common\SphereScript.hpp"
#include "resource.h"

static int s_ScriptViewID = 900;


#define DEFINE_COLOR(name, bold, color) \
  CScriptView::SymbolColor CScriptView::name = { bold, color };

DEFINE_COLOR(m_icComment,  false, RGB(0,   128, 0  ))
DEFINE_COLOR(m_icNumber,   false, RGB(255, 0,   255))
DEFINE_COLOR(m_icString,   false, RGB(128, 128, 128))
DEFINE_COLOR(m_icKeyword,  false, RGB(0,   0,   255))
DEFINE_COLOR(m_icFunction, false, RGB(0,   0,   128))

const UINT MsgFindReplace = ::RegisterWindowMessage(FINDMSGSTRING);

BEGIN_MESSAGE_MAP(CScriptView, CRichEditCtrl)
  ON_WM_SETFOCUS()
  ON_WM_KEYUP()
  ON_WM_KEYDOWN()
  ON_WM_CHAR()
  ON_WM_LBUTTONDOWN()
  ON_WM_LBUTTONDBLCLK()

  ON_REGISTERED_MESSAGE(MsgFindReplace, OnFindReplaceMsg)
END_MESSAGE_MAP()

///////////////////////////////////////////////////////////////////////////////

CScriptView::CScriptView()
: m_SearchDialog(NULL)
{
    m_InForcedChange = false;
    m_ChangeType = ctUndo;
    m_OldSel.cpMin = m_OldSel.cpMax = 0;
    m_LineCount = 0;
}

///////////////////////////////////////////////////////////////////////////////

CScriptView::~CScriptView()
{
}

///////////////////////////////////////////////////////////////////////////////

BOOL
CScriptView::Create(CStatusBar* status_bar, CScriptViewHandler* handler, CWnd* parent_window)
{
  m_Handler = handler;
  m_StatusBar = status_bar;

  BOOL Val;
  Val = CRichEditCtrl::Create(WS_CHILD | WS_VISIBLE | ES_LEFT | ES_MULTILINE | ES_NOHIDESEL | ES_AUTOHSCROLL | ES_AUTOVSCROLL | WS_HSCROLL | WS_VSCROLL,
        CRect(0,0,0,0),
        parent_window,
        s_ScriptViewID++);

  if (Val)
  {
    CFont theText;
    theText.Attach(GetStockObject(ANSI_FIXED_FONT));
    SetFont(&theText);
  }

  return Val;
}

///////////////////////////////////////////////////////////////////////////////

void 
CScriptView::FormatAll()
{
  FormatTextRange(0, GetTextLength());
}

///////////////////////////////////////////////////////////////////////////////

void
CScriptView::ScriptFindWord()
{
  if (m_SearchDialog == NULL)
  {
    m_SearchDialog = new CFindReplaceDialog;
    m_SearchDialog->Create(true, NULL, NULL, FR_DOWN, this);
  }
}

///////////////////////////////////////////////////////////////////////////////

void
CScriptView::ScriptFindReplaceWord()
{
  if (m_SearchDialog == NULL)
  {
    m_SearchDialog = new CFindReplaceDialog;
    m_SearchDialog->Create(false, NULL, NULL, FR_DOWN, this);
  }
}

///////////////////////////////////////////////////////////////////////////////

afx_msg void
CScriptView::OnSetFocus(CWnd* pOldWnd)
{
  char ControlKey = GetAsyncKeyState(VK_CONTROL) & 2;
  char ShiftKey   = GetAsyncKeyState(VK_SHIFT) & 2;

  if (!ControlKey) m_ControlPressed = false;
  if (!ShiftKey)   m_ShiftPressed = false;

  CRichEditCtrl::OnSetFocus(pOldWnd);
}

///////////////////////////////////////////////////////////////////////////////

afx_msg void
CScriptView::OnKeyUp(UINT nKey, UINT nRepCnt, UINT nFlags)
{
  CRichEditCtrl::OnKeyUp(nKey, nRepCnt, nFlags);

  switch(nKey)
  {

  case VK_UP:
  case VK_DOWN:
  case VK_RIGHT:
  case VK_LEFT:
    if (!m_ShiftPressed)
    {
      m_ChangeType = ctMove;
      OnChange();
      UpdateLineInfo();
    }
    break;


  case VK_RETURN:
  {
      UpdateLineInfo();
    }
    break;

  case VK_CONTROL:
    m_ControlPressed = false;
    break;

  case VK_SHIFT:
    m_ShiftPressed = false;
    break;

  case VK_DELETE: 
    m_ChangeType = ctDelete; 
    OnChange();
    m_Handler->SV_ScriptChanged();
    UpdateLineInfo();
    break;

  case VK_BACK:   
    m_ChangeType = ctBack;
    OnChange();
    m_Handler->SV_ScriptChanged();
    UpdateLineInfo();
    break;

  case VK_PRIOR:  //VK_PAGE_UP:
  case VK_NEXT:   //VK_PAGE_DOWN:
  case VK_HOME:
  case VK_END:
    m_ChangeType = ctMove;
    OnChange();
    UpdateLineInfo();
    break;
  }
  }

///////////////////////////////////////////////////////////////////////////////

afx_msg void
CScriptView::OnKeyDown(UINT nKey, UINT nRepCnt, UINT nFlags)
{ 
  CRichEditCtrl::OnKeyDown(nKey, nRepCnt, nFlags);

  switch (nKey)
  {
  case VK_UP:
  case VK_DOWN:
  case VK_RIGHT:
  case VK_LEFT:
    if (!m_ShiftPressed)
    {
      m_ChangeType = ctMove;
      OnChange();
      
      char buf[80];
      sprintf(buf, "Line %i / %i", LineFromChar(-1) + 1, GetLineCount());
      m_StatusBar->SetWindowText(buf);
    }
    break;

  case 'A':
  case 'B':
  case 'C':
  case 'D':
  case 'E':
  case 'F':
  case 'G':
  case 'H':
  case 'I':
  case 'J':
  case 'K':
  case 'L':
  case 'M':
  case 'N':
  case 'O':
  case 'P':
  case 'Q':
  case 'R':
  case 'S':
  case 'T':
  case 'U':
  case 'V':
  case 'W':
  case 'X':
  case 'Y':
  case 'Z':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
  case '0':
  case VK_RETURN:
  case VK_SPACE:
    if (!m_ControlPressed)
      m_Handler->SV_ScriptChanged();
    break;

  case VK_CONTROL:
    m_ControlPressed = true;
    break;

  case VK_SHIFT:
    m_ShiftPressed = true;
    break;

  default:
    break;
  }
}

///////////////////////////////////////////////////////////////////////////////

afx_msg void
CScriptView::OnChar(UINT nKey, UINT nRepCnt, UINT nFlags)
{

  if (nKey == VK_TAB)
  {
    ::SendMessage(this->m_hWnd, WM_CHAR, VK_SPACE, 0);
    ::SendMessage(this->m_hWnd, WM_CHAR, VK_SPACE, 0);    
    OnChange();
    m_Handler->SV_ScriptChanged();
    return;
  }

  m_ChangeType = ctReplSel;
  CRichEditCtrl::OnChar(nKey, nRepCnt, nFlags);
  OnChange();
}

///////////////////////////////////////////////////////////////////////////////

afx_msg void
CScriptView::OnLButtonDown(UINT nFlags, CPoint point)
{
  CRichEditCtrl::OnLButtonUp(nFlags, point);
  UpdateLineInfo();
}

///////////////////////////////////////////////////////////////////////////////

afx_msg void 
CScriptView::OnLButtonDblClk(UINT nFlags, CPoint point)
{
  CRichEditCtrl::OnLButtonDblClk(nFlags, point);
  UpdateLineInfo();
}

///////////////////////////////////////////////////////////////////////////////

LRESULT
CScriptView::OnFindReplaceMsg(WPARAM wParam, LPARAM lParam)
{
  if (m_SearchDialog->IsTerminating())
    m_SearchDialog = NULL;
  else if (m_SearchDialog->FindNext())
  {
    //MessageBox("Find!");
    OnFindText();
  }
  else if (m_SearchDialog->ReplaceCurrent())
    MessageBox("Replace!");

  return 0;
}


///////////////////////////////////////////////////////////////////////////////

void
CScriptView::UpdateLineInfo()
{
  char buf[80];
  sprintf(buf, "Line %i / %i", LineFromChar(-1) + 1, GetLineCount());
  m_StatusBar->SetWindowText(buf);
}

///////////////////////////////////////////////////////////////////////////////

void 
CScriptView::OnChange() 
{
  CHARRANGE CurSel; 
  GetSel(CurSel);

  if (m_ChangeType == ctMove && CurSel.cpMin == CurSel.cpMax) {
    // cut was canceled, so this is paste operation
    m_ChangeType = ctPaste;
  }

  switch(m_ChangeType) {
  case ctReplSel:// old=(x,y) -> cur=(x+len,x+len)
  case ctPaste:  // old=(x,y) -> cur=(x+len,x+len)
    FormatTextLines(CurSel.cpMin, CurSel.cpMax);
    break;
  case ctDelete: // old=(x,y) -> cur=(x,x)
  case ctBack:   // old=(x,y) -> cur=(x,x), newline del => old=(x,x+1) -> cur=(x-1,x-1)
  //case ctCut:    // old=(x,y) -> cur=(x,x)
    FormatTextLines(CurSel.cpMin, CurSel.cpMax);
    break;
  case ctUndo:   // old=(?,?) -> cur=(x,y)
    FormatTextLines(CurSel.cpMin, CurSel.cpMax);
    break;
  case ctMove:   // old=(x,x+len) -> cur=(y-len,y) | cur=(y,y+len)
    FormatTextLines(CurSel.cpMin, CurSel.cpMax);
    if (CurSel.cpMin > m_OldSel.cpMin) // move after
      FormatTextLines(m_OldSel.cpMin, m_OldSel.cpMin);
    else // move before
      FormatTextLines(m_OldSel.cpMax, m_OldSel.cpMax);
    break;
  default:
    FormatAll();
    break;
  }

  //undo action does not call OnProtected, so make it default
  m_ChangeType = ctUndo;
}

///////////////////////////////////////////////////////////////////////////////

void 
CScriptView::SetFormatRange(int nStart, int nEnd, BOOL bBold, COLORREF clr)
{
  if (nStart >= nEnd)
    return;

  SetSel(nStart, nEnd);

  DWORD dwEffects = bBold?CFE_BOLD:0;

  CHARFORMAT cfm;
  cfm.cbSize = sizeof(cfm);
    GetSelectionCharFormat(cfm);
  
  if ((cfm.dwMask & CFM_COLOR)  && cfm.crTextColor == clr && 
    (cfm.dwMask & CFM_BOLD) && (cfm.dwEffects & CFE_BOLD) == dwEffects)
    return;
  
  cfm.dwEffects = dwEffects;
  cfm.crTextColor = clr;
  cfm.dwMask = CFM_BOLD | CFM_COLOR;

  SetSelectionCharFormat(cfm);
}

///////////////////////////////////////////////////////////////////////////////

void 
CScriptView::FormatTextLines(int nLineStart, int nLineEnd)
{
  long nStart = LineIndex(LineFromChar(nLineStart));
  long nEnd = LineIndex(LineFromChar(nLineEnd));
  nEnd += LineLength(nLineEnd);

  FormatTextRange(nStart, nEnd);
}

///////////////////////////////////////////////////////////////////////////////

void 
CScriptView::FormatTextRange(int nStart, int nEnd)
{
    if (nStart >= nEnd)
      return;

    CHARRANGE crOldSel;

    GetSel(crOldSel);
    HideSelection(TRUE, FALSE);

    SetSel(nStart, nEnd);
    TCHAR *pBuffer = new TCHAR[nEnd - nStart + 1];
    long nLen = GetSelText(pBuffer);

    pBuffer[nLen] = 0;

    TCHAR *pStart, *pPtr;
    SymbolColor ic;
    pStart = pPtr = pBuffer;

    TCHAR* pSymbolStart = NULL;

    while (*pPtr != 0) {
      TCHAR ch = *pPtr;

  
      if (ch == '/' && pPtr[1] == '/') { // Process // comment
        pSymbolStart = pPtr;
        do { ch = *(++pPtr); } while (ch != 0 && ch != '\r' && ch != '\n');
        ic = m_icComment;

      } else if (ch == '/' && pPtr[1] == '*') { // Process /* */ comment
        pSymbolStart = pPtr++;
        do { ch = *(++pPtr);
             if (ch == '*' && pPtr[1] != 0)
               if (pPtr[1] == '/') 
               { pPtr+=2; break; }
           } while(ch != 0);
        ic = m_icComment;

      } else if (ch == '"') { // Process strings
        pSymbolStart = pPtr;
        do { ch = *(++pPtr); } while (ch != 0 && ch != '"');
        if (ch == '"') pPtr++;
        ic = m_icString;

      } else if (_istdigit(ch)) { // Process numbers
        pSymbolStart = pPtr;
        _tcstod(pSymbolStart, &pPtr);
        ic = m_icNumber;

      } else if (_istalpha(ch) || ch == '_') { // Process keywords
        pSymbolStart = pPtr;
        do { ch = *(++pPtr); } while (_istalnum(ch) || ch == '_');
        *pPtr = 0;
        if (SS_IsSystemFunction(pSymbolStart))
        { 
          ic = m_icFunction;
        } else {
          if (SS_IsKeyword(pSymbolStart)) 
            ic = m_icKeyword;
          else 
            pSymbolStart = NULL;
        }
        *pPtr = ch;

      } else {
        pPtr++;
      }

      if (pSymbolStart != NULL) {
        SetFormatRange(nStart + pStart - pBuffer, nStart + pSymbolStart - pBuffer, FALSE, RGB(0,0,0));
        SetFormatRange(nStart + pSymbolStart - pBuffer, nStart + pPtr - pBuffer, ic.bold, ic.color);
        pStart = pPtr;
        pSymbolStart = 0;
      } else if (*pPtr == 0)
        SetFormatRange(nStart + pStart - pBuffer, nStart + pPtr - pBuffer, FALSE, RGB(0,0,0));
    }

  delete [] pBuffer;

  SetSel(crOldSel);
  HideSelection(FALSE, FALSE);
}

///////////////////////////////////////////////////////////////////////////////

void
CScriptView::OnFindText()
{
  CString Word;
  CHARRANGE crSearchRegion;
  CHARRANGE crOldSel;
  long Location;
  
  GetSel(crSearchRegion);
  GetSel(crOldSel);
  crSearchRegion.cpMax = -1;

  SetSel(crSearchRegion);
  Word = GetSelText();

  Location = Word.Find("int");
  if (Location > 0)
  {
    crOldSel.cpMin += Location;
    crOldSel.cpMax = crOldSel.cpMin + 3;
  }
  SetSel(crOldSel);
}

///////////////////////////////////////////////////////////////////////////////

