#include <windows.h>
#include <stdio.h>
#include <string.h>
#include "Project.hpp"
#include "configfile.h"
#include "types.h"
#include "x++.hpp"


////////////////////////////////////////////////////////////////////////////////

CProject::CProject()
: m_Directory(newstr(""))
, m_Filename(newstr(""))

, m_GameTitle(newstr(""))
, m_GameScript(newstr(""))

, m_ScreenWidth(320)
, m_ScreenHeight(240)
{
  for (int i = 0; i < 10; i++)
  {
    m_Groups[i].num_entries = 0;
    m_Groups[i].entries = NULL;
  }
}

////////////////////////////////////////////////////////////////////////////////

CProject::~CProject()
{
  Destroy();
}

////////////////////////////////////////////////////////////////////////////////

bool
CProject::Create(const char* games_directory, const char* project_name)
{
  Destroy();

  // create the project
  if (SetCurrentDirectory(games_directory) == FALSE)
    return false;
  
  // if creating the directory failed, it may already exist
  CreateDirectory(project_name, NULL);
  
  // wait to see if SetCurrentDirectory() fails
  if (SetCurrentDirectory(project_name) == FALSE)
    return false;

  // set the project directory
  char path[MAX_PATH];
  if (GetCurrentDirectory(MAX_PATH, path) == FALSE)
    return false;
  m_Directory = newstr(path);
    
  // set the project filename
  m_Filename = new char[strlen(path) + strlen(project_name) + 20];
  sprintf(m_Filename, "%s\\game.inf", path);

  // set default values in project
  m_GameTitle = newstr("");
  m_GameScript = newstr("");

  m_ScreenWidth = 320;
  m_ScreenHeight = 240;

  RefreshItems();
  return Save();
}

////////////////////////////////////////////////////////////////////////////////

bool
CProject::Open(const char* filename)
{
  Destroy();

  // set the game directory
  m_Directory = newstr(filename);
  *strrchr(m_Directory, '\\') = 0;

  // set the game filename
  m_Filename = newstr(m_Directory + strlen(m_Directory) + 1);

  // load the game.inf
  CONFIG config;
  LoadConfig(&config, m_Filename);

  // game title
  char title[520];
  ReadConfigString(&config, "", "name", title, 512, "");
  m_GameTitle = newstr(title);

  // script
  char script[520];
  ReadConfigString(&config, "", "script", script, 512, "");

  if (strrchr(script, '.'))
    strcpy(strrchr(script, '.'), ".ss");
  m_GameScript = newstr(script);

  // screen dimensions
  ReadConfigInt(&config, "", "screen_width",  &m_ScreenWidth, 320);
  ReadConfigInt(&config, "", "screen_height", &m_ScreenHeight, 240);

  DestroyConfig(&config);

  RefreshItems();

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CProject::Save() const
{
  SetCurrentDirectory(m_Directory);
  
  CONFIG config;
  LoadConfig(&config, m_Filename);

  // title
  WriteConfigString(&config, "", "name", m_GameTitle);

  // script
  char* script_path = new char[strlen(m_GameScript) + 10];
  strcpy(script_path, m_GameScript);
  if (strrchr(script_path, '.'))
    strcpy(strrchr(script_path, '.'), ".ssx");

  WriteConfigString(&config, "", "script", script_path);
  delete[] script_path;

  // screen dimensions
  WriteConfigInt(&config, "", "screen_width",  m_ScreenWidth);
  WriteConfigInt(&config, "", "screen_height", m_ScreenHeight);

  SaveConfig(&config, m_Filename);
  DestroyConfig(&config);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetDirectory() const
{
  return m_Directory;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetGameSubDirectory() const
{
  if (strrchr(m_Directory, '\\'))
    return strrchr(m_Directory, '\\') + 1;
  else
    return "";
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetGameTitle() const
{
  return m_GameTitle;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetGameScript() const
{
  return m_GameScript;
}

////////////////////////////////////////////////////////////////////////////////

int
CProject::GetScreenWidth() const
{
  return m_ScreenWidth;
}

////////////////////////////////////////////////////////////////////////////////

int
CProject::GetScreenHeight() const
{
  return m_ScreenHeight;
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::SetGameTitle(const char* game_title)
{
  delete[] m_GameTitle;
  m_GameTitle = newstr(game_title);
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::SetGameScript(const char* game_script)
{
  delete[] m_GameScript;
  m_GameScript = newstr(game_script);
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::SetScreenWidth(int width)
{
  m_ScreenWidth = width;
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::SetScreenHeight(int height)
{
  m_ScreenHeight = height;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetGroupDirectory(int grouptype)
{
  switch (grouptype)
  {
    case GT_MAPS:         return "maps";
    case GT_SPRITESETS:   return "spritesets";
    case GT_SCRIPTS:      return "scripts";
    case GT_SOUNDS:       return "sounds";
    case GT_MUSIC:        return "music";
    case GT_FONTS:        return "fonts";
    case GT_WINDOWSTYLES: return "windowstyles";
    case GT_IMAGES:       return "images";
    case GT_ANIMATIONS:   return "anim";
    default:              return NULL;
  }
}

////////////////////////////////////////////////////////////////////////////////

static const char* GetFilterList(int grouptype)
{
  switch (grouptype)
  {
    case GT_MAPS:         return "1\0*.rmp";
    case GT_SPRITESETS:   return "1\0*.rss";
    case GT_SCRIPTS:      return "2\0*.ss\0*.ssl";
    case GT_SOUNDS:       return "1\0*.wav";
    case GT_MUSIC:        return "6\0*.mid\0*.it\0*.xm\0*.s3m\0*.mod\0*.mp3";  // *.mid takes care of *.midi too :(
    case GT_FONTS:        return "1\0*.rfn";
    case GT_WINDOWSTYLES: return "1\0*.rws";
    case GT_IMAGES:       return "3\0*.png\0*.pcx\0*.bmp";
    case GT_ANIMATIONS:   return "2\0*.fli\0*.flc"; // Win32 thinks *.fli is the same as *.flic
    default:              return NULL;
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::RefreshItems()
{
  // empty the old lists
  for (int i = 0; i < NUM_GROUP_TYPES; i++)
  {
    for (int j = 0; j < m_Groups[i].num_entries; j++)
      delete[] m_Groups[i].entries[j];

    m_Groups[i].num_entries = 0;
    m_Groups[i].entries = (char**)realloc(m_Groups[i].entries, 0);
  }

  // store the old directory
  char old_directory[MAX_PATH];
  GetCurrentDirectory(MAX_PATH, old_directory);

  for (int i = 0; i < NUM_GROUP_TYPES; i++)
  {
    SetCurrentDirectory(m_Directory);
    
    if (!SetCurrentDirectory(GetGroupDirectory(i)))
      continue;

    // find all files in the group and add them to the group
    const char* filter_list = GetFilterList(i);
    int num_filters = atoi(filter_list);
    filter_list += strlen(filter_list) + 1;

    // go through each filter and add it to the project
    for (int j = 0; j < num_filters; j++)
    {
      WIN32_FIND_DATA ffd;
      HANDLE h = FindFirstFile(filter_list, &ffd);
      if (h == INVALID_HANDLE_VALUE)
      {
        // go to the next filter
        filter_list += strlen(filter_list) + 1;
        continue;
      }

      do {

        if (!(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
          AddItem(i, ffd.cFileName);

      } while (FindNextFile(h, &ffd));

      FindClose(h);

      // go to the next filter
      filter_list += strlen(filter_list) + 1;
    }

  }

  // restore the old directory
  SetCurrentDirectory(old_directory);
}

////////////////////////////////////////////////////////////////////////////////

int
CProject::GetItemCount(EGroupType group_type) const
{
  return m_Groups[group_type].num_entries;
}

////////////////////////////////////////////////////////////////////////////////

const char*
CProject::GetItem(EGroupType group_type, int i) const
{
  return m_Groups[group_type].entries[i];
}

////////////////////////////////////////////////////////////////////////////////

bool
CProject::HasItem(EGroupType group_type, const char* item) const
{
  for (int i = 0; i < GetItemCount(group_type); i++)
    if (strcmp(item, GetItem(group_type, i)) == 0)
      return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::AddItem(int grouptype, const char* filename)
{
  Group* group = m_Groups + grouptype;
  group->entries = (char**)realloc(group->entries, sizeof(char*) * (group->num_entries + 1));
  group->entries[group->num_entries] = newstr(filename);
  group->num_entries++;
}

////////////////////////////////////////////////////////////////////////////////

void
CProject::Destroy()
{
  delete[] m_Directory;
  m_Directory = NULL;

  delete[] m_Filename;
  m_Filename = NULL;

  delete[] m_GameTitle;
  m_GameTitle = NULL;

  m_GameScript = 0;
  m_ScreenWidth = 0;
  m_ScreenHeight = 0;

  for (int i = 0; i < NUM_GROUP_TYPES; i++)
  {
    for (int j = 0; j < m_Groups[i].num_entries; j++)
      delete[] m_Groups[i].entries[j];

    m_Groups[i].num_entries = 0;
    m_Groups[i].entries = (char**)realloc(m_Groups[i].entries, 0);
  }
}

////////////////////////////////////////////////////////////////////////////////
