// system
#include <afxdlgs.h>
#include <htmlhelp.h>

// core
#include "WindowCommands.hpp"
#include "WindowAttributes.hpp"
#include "FileSystem.hpp"
#include "StateServer.hpp"
#include "resource.h"

// document windows
#include "MainWindow.hpp"
#include "ProjectWindow.hpp"
#include "MapWindow.hpp"
#include "SpritesetWindow.hpp"
#include "MusicWindow.hpp"
#include "SoundWindow.hpp"
#include "ScriptWindow.hpp"
#include "ImageWindow.hpp"
#include "AnimationWindow.hpp"
#include "WindowStyleWindow.hpp"
#include "FontWindow.hpp"

// dialogs
#include "NewProjectDialog.hpp"
#include "OptionsDialog.hpp"
#include "GameSettingsDialog.hpp"
#include "FileDialogs.hpp"
#include "ResizeDialog.hpp"

// common
#include "folderdialog.h"
#include "../common/Map.hpp"
#include "x++.hpp"


const char szBarState[] = "SDE_BarState";

static UINT indicators[] =
{
	ID_SEPARATOR,           // status line indicator
	ID_INDICATOR_CAPS,
	ID_INDICATOR_NUM,
};


BEGIN_MESSAGE_MAP(CMainWindow, CMDIFrameWnd)

  ON_WM_CLOSE()

  // generic file open
  ON_COMMAND(ID_FILE_OPEN, OnFileOpen)

  // project
  ON_COMMAND(ID_FILE_NEW_PROJECT,  OnFileNewProject)
  ON_COMMAND(ID_FILE_OPEN_PROJECT, OnFileOpenProject)
  ON_COMMAND(ID_FILE_CLOSEPROJECT, OnFileCloseProject)

  // file | new
  ON_COMMAND(ID_FILE_NEW_MAP,         OnFileNewMap)
  ON_COMMAND(ID_FILE_NEW_SPRITESET,   OnFileNewSpriteset)
  ON_COMMAND(ID_FILE_NEW_SCRIPT,      OnFileNewScript)
  ON_COMMAND(ID_FILE_NEW_FONT,        OnFileNewFont)
  ON_COMMAND(ID_FILE_NEW_WINDOWSTYLE, OnFileNewWindowStyle)
  ON_COMMAND(ID_FILE_NEW_IMAGE,       OnFileNewImage)

  // file | open
  ON_COMMAND(ID_FILE_OPEN_MAP,         OnFileOpenMap)
  ON_COMMAND(ID_FILE_OPEN_SPRITESET,   OnFileOpenSpriteset)
  ON_COMMAND(ID_FILE_OPEN_SCRIPT,      OnFileOpenScript)
  ON_COMMAND(ID_FILE_OPEN_SOUND,       OnFileOpenSound)
  ON_COMMAND(ID_FILE_OPEN_MUSIC,       OnFileOpenMusic)
  ON_COMMAND(ID_FILE_OPEN_FONT,        OnFileOpenFont)
  ON_COMMAND(ID_FILE_OPEN_WINDOWSTYLE, OnFileOpenWindowStyle)
  ON_COMMAND(ID_FILE_OPEN_IMAGE,       OnFileOpenImage)
  ON_COMMAND(ID_FILE_OPEN_ANIMATION,   OnFileOpenAnimation)

  // file | import
  ON_COMMAND(ID_FILE_IMPORT_IMAGETOMAPTILESET, OnFileImportImageToMap)
  ON_COMMAND(ID_FILE_IMPORT_BITMAPTORWS, OnFileImportBitmapToRWS)
  ON_COMMAND(ID_FILE_IMPORT_BITMAPTORSS, OnFileImportBitmapToRSS)

  ON_COMMAND(ID_FILE_OPTIONS, OnFileOptions)
  ON_COMMAND(ID_FILE_EXIT,    OnClose)

  // insert
  ON_COMMAND(ID_PROJECT_INSERT_MAP,         OnProjectInsertMap)
  ON_COMMAND(ID_PROJECT_INSERT_SPRITESET,   OnProjectInsertSpriteset)
  ON_COMMAND(ID_PROJECT_INSERT_SCRIPT,      OnProjectInsertScript)
  ON_COMMAND(ID_PROJECT_INSERT_SOUND,       OnProjectInsertSound)
  ON_COMMAND(ID_PROJECT_INSERT_MUSIC,       OnProjectInsertMusic)
  ON_COMMAND(ID_PROJECT_INSERT_FONT,        OnProjectInsertFont)
  ON_COMMAND(ID_PROJECT_INSERT_WINDOWSTYLE, OnProjectInsertWindowStyle)
  ON_COMMAND(ID_PROJECT_INSERT_IMAGE,       OnProjectInsertImage)
  ON_COMMAND(ID_PROJECT_INSERT_ANIMATION,   OnProjectInsertAnimation)

  ON_COMMAND(ID_PROJECT_REFRESH,         OnProjectRefresh)
  ON_COMMAND(ID_PROJECT_RUNSPHERE,       OnProjectRunSphere)
  ON_COMMAND(ID_PROJECT_CONFIGURESPHERE, OnProjectConfigureSphere)

  ON_COMMAND(ID_HELP_CONTENTS,       OnHelpContents)
  ON_COMMAND(ID_HELP_ABOUT,          OnHelpAbout)

  ON_UPDATE_COMMAND_UI(ID_FILE_CLOSEPROJECT,             OnUpdateProjectCommand)

  ON_UPDATE_COMMAND_UI(ID_FILE_SAVE,       OnUpdateSaveCommand)
  ON_UPDATE_COMMAND_UI(ID_FILE_SAVEAS,     OnUpdateSaveCommand)
  ON_UPDATE_COMMAND_UI(ID_FILE_SAVECOPYAS, OnUpdateSaveCommand)

  ON_UPDATE_COMMAND_UI(ID_PROJECT_RUNSPHERE, OnUpdateProjectCommand)

  // document window messages
  ON_MESSAGE(WM_DW_CLOSING,       OnDocumentWindowClosing)
  ON_MESSAGE(WM_SET_CHILD_MENU,   OnSetChildMenu)
  ON_MESSAGE(WM_CLEAR_CHILD_MENU, OnClearChildMenu)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CMainWindow::CMainWindow()
: m_ProjectOpen(false)
, m_ProjectWindow(NULL)
, m_ChildMenuResource(-1)
{
}

////////////////////////////////////////////////////////////////////////////////

BOOL
CMainWindow::Create()
{
  // create the window
  RECT startupRect;
  startupRect.left   = GetStateServer()->GetInt("Startup:left");
  startupRect.top    = GetStateServer()->GetInt("Startup:top");
  startupRect.right  = GetStateServer()->GetInt("Startup:right");
  startupRect.bottom = GetStateServer()->GetInt("Startup:bottom");
  if (startupRect.right == 0)  // detect if this is the first time it was run
    startupRect = rectDefault;    
    
  CMDIFrameWnd::Create(
    AfxRegisterWndClass(0, NULL, NULL, AfxGetApp()->LoadIcon(IDI_SDE)),
    "Sphere Development Environment",
    WS_OVERLAPPEDWINDOW,
    startupRect,
    NULL,
    MAKEINTRESOURCE(IDR_MAIN));

  LoadAccelTable(MAKEINTRESOURCE(IDR_ACCELERATOR));

  // create the toolbar
  m_ToolBar.CreateEx(
    this,
    TBSTYLE_FLAT,
    WS_CHILD | WS_VISIBLE | CBRS_SIZE_DYNAMIC | CBRS_TOP | CBRS_GRIPPER | CBRS_FLYBY);
  m_ToolBar.SetWindowText("Main");
  m_ToolBar.LoadToolBar(IDR_TOOLBAR);
  m_ToolBar.EnableDocking(CBRS_ALIGN_ANY);

  // create the statusbar
  m_StatusBar.Create(this);
  m_StatusBar.SetIndicators(indicators, sizeof(indicators)/sizeof(UINT));
  m_StatusBar.SetBarStyle(m_StatusBar.GetBarStyle() | CBRS_FLYBY | CBRS_TOOLTIPS);

  // enable docking
  EnableDocking(CBRS_ALIGN_ANY);

  DockControlBar(&m_ToolBar, AFX_IDW_DOCKBAR_TOP);
  
  // load the command bar state
  LoadBarState(szBarState);

  // show the window
  int startupState = GetStateServer()->GetInt("WindowState");
  if (startupState == 0)
    ShowWindow(SW_SHOW);
  else if (startupState == 1)
    ShowWindow(SW_MAXIMIZE);
  else if (startupState == 2)
    ShowWindow(SW_MINIMIZE);
  UpdateWindow();

  UpdateMenu();

  return TRUE;
}

////////////////////////////////////////////////////////////////////////////////

void
CMainWindow::CreateProject(const char* projectname, const char* gametitle)
{
  CloseProject();

  char games_directory[MAX_PATH];
  GetGamesDirectory(games_directory);

  // this may fail, but we don't care
  CreateDirectory(games_directory, NULL);
  
  if (!m_Project.Create(games_directory, projectname))
  {
    MessageBox("Error: Could not create project");
    return;
  }

  m_Project.SetGameTitle(gametitle);
  m_ProjectOpen = true;
  m_ProjectWindow = new CProjectWindow(this, &m_Project);
  m_ProjectWindow->Create();

  UpdateMenu();
}

////////////////////////////////////////////////////////////////////////////////

void
CMainWindow::OpenProject(const char* filename)
{
  CloseProject();
  
  if (m_Project.Open(filename) == false)
  {
    MessageBox("Could not open project");
    return;
  }

  m_ProjectOpen = true;
  m_ProjectWindow = new CProjectWindow(this, &m_Project);
  m_ProjectWindow->Create();

  UpdateMenu();
}

////////////////////////////////////////////////////////////////////////////////

void
CMainWindow::CloseProject()
{
  if (m_ProjectOpen)
  {
    m_ProjectOpen = false;

    if (m_ProjectWindow)
    {
      m_ProjectWindow->DestroyWindow();
      m_ProjectWindow = NULL;
    }

    UpdateMenu();
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CMainWindow::OpenGameSettings()
{
  CGameSettingsDialog(&m_Project).DoModal();
}

////////////////////////////////////////////////////////////////////////////////

void
CMainWindow::OpenDocumentWindow(EGroupType grouptype, const char* filename)
{
  CDocumentWindow* window = NULL;
  switch (grouptype)
  {
    case GT_MAPS:         window = new CMapWindow(&m_StatusBar, filename); break;
    case GT_SPRITESETS:   window = new CSpritesetWindow(filename);         break;
    case GT_SCRIPTS:      window = new CScriptWindow(&m_StatusBar, filename);            break;
    case GT_SOUNDS:       window = new CSoundWindow(filename);             break;
    case GT_MUSIC:        window = new CMusicWindow(filename);             break;
    case GT_FONTS:        window = new CFontWindow(filename);              break;
    case GT_WINDOWSTYLES: window = new CWindowStyleWindow(filename);       break;
    case GT_IMAGES:       window = new CImageWindow(filename);             break;
    case GT_ANIMATIONS:   window = new CAnimationWindow(filename);         break;
  }

  m_DocumentWindows.Add(window);
}

////////////////////////////////////////////////////////////////////////////////

bool
CMainWindow::AddToDirectory(const char* pathname, const char* sub_directory)
{
  // if file isn't already in the subdirectory
  if (!CheckDirectory(pathname, sub_directory))
  {
    // ask the user if we can copy it
    char message[1024];
    sprintf(message, "The file must be copied into the game '%s' directory.  Is this okay?", sub_directory);
    if (MessageBox(message, NULL, MB_YESNO) == IDNO)
      return false;

    char szDestination[MAX_PATH];
    strcpy(szDestination, m_Project.GetDirectory());
    strcat(szDestination, "\\");
    strcat(szDestination, sub_directory);

    // create the directory
    CreateDirectory(szDestination, NULL);

    // append the filename
    strcat(szDestination, "\\");
    strcat(szDestination, strrchr(pathname, '\\') + 1);

    // copy it
    if (CopyFile(pathname, szDestination, TRUE) == FALSE)
    {
      if (MessageBox("File appears to already exist, overwrite?", NULL, MB_YESNO) == IDNO)
        return false;

      if (CopyFile(pathname, szDestination, FALSE) == FALSE)
      {
        MessageBox("Error: Could not copy file");
        return false;
      }
    }
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CMainWindow::CheckDirectory(const char* filename, const char* sub_directory)
{
  // remove the file name from szFileName
  char szDirectory[MAX_PATH];
  strcpy(szDirectory, filename);
  if (*strrchr(szDirectory, '\\'))
    *strrchr(szDirectory, '\\') = 0;

  // add the sub directory to the project directory
  char szProjectDirectory[MAX_PATH];
  strcpy(szProjectDirectory, m_Project.GetDirectory());
  strcat(szProjectDirectory, "\\");
  strcat(szProjectDirectory, sub_directory);

  // compare the path with the project directory + szSubDirectory
  strlwr(szDirectory);
  strlwr(szProjectDirectory);
  return (strcmp(szDirectory, szProjectDirectory) == 0);
}

////////////////////////////////////////////////////////////////////////////////

void
CMainWindow::InsertProjectFile(CFileDialog* file_dialog, EGroupType grouptype)
{
  // determine filetype directory
  char path[MAX_PATH];
  sprintf(path, "%s\\%s", m_Project.GetDirectory(), m_Project.GetGroupDirectory(grouptype));
  if (!PathExists(path))
  {
    char szMessage[80];
    sprintf(szMessage, "'%s' subdirectory does not exist, create?", m_Project.GetGroupDirectory(grouptype));
    if (MessageBox(szMessage, NULL, MB_YESNO) == IDNO)
      return;

    // create the directory
    if (!CreateDirectory(path, NULL))
    {
      char szMessage[MAX_PATH + 80];
      sprintf(szMessage, "Error: Could not create directory '%s'", path);
      MessageBox(szMessage);
      return;
    }
  }

  // start the file dialog in the correct directory
  SetCurrentDirectory(path);
  if (file_dialog->DoModal() == IDOK)
  {
    // we've got a full file path
    CString sPathName = file_dialog->GetPathName();

    char path_name[MAX_PATH];
    strcpy(path_name, sPathName);
    *strrchr(path_name, '\\') = 0;

    if (strcmp_ci(path, path_name) == 0)
    {
      // if the file is in the same directory, and the file already exists
      if (FileExists(sPathName))
      {
        MessageBox("File is already in project");
        return;
      }
      else
      {
        // create an empty file that will not be valid
        fclose(fopen(sPathName, "wb"));
      }
    }
    else
    {
      // if file exists, use AddToDirectory() in case we need to copy it
      AddToDirectory(sPathName, CProject::GetGroupDirectory(grouptype));
    }

    // save the project and update the view
    m_Project.RefreshItems();
    m_ProjectWindow->Update();
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CMainWindow::GetGamesDirectory(char games_directory[MAX_PATH])
{
  GetStateServer()->GetString("SphereDirectory", games_directory, MAX_PATH);
  if (games_directory[strlen(games_directory) - 1] != '\\')
    strcat(games_directory, "\\");
  strcat(games_directory, "games");
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::UpdateMenu()
{
  int iWindowMenu = 2;

  // destroy the old menu
  if (GetMenu())
    GetMenu()->DestroyMenu();

  // create the new menu
  HINSTANCE hInstance = AfxGetApp()->m_hInstance;
  HMENU hNewMenu = LoadMenu(hInstance, MAKEINTRESOURCE(IDR_MAIN));

  // if a project is open, add the project menu
  if (m_ProjectOpen)
  {
    HMENU hProjectMenu = LoadMenu(hInstance, MAKEINTRESOURCE(IDR_PROJECT));

    char szPopupTitle[80];
    GetMenuString(hProjectMenu, 0, szPopupTitle, 80, MF_BYPOSITION);

    InsertMenu(hNewMenu,
               1,
               MF_POPUP | MF_BYPOSITION | MF_STRING,
               (UINT_PTR)GetSubMenu(hProjectMenu, 0),
               szPopupTitle);

    iWindowMenu++;
  }

  // if a child menu is set, add it
  if (m_ChildMenuResource != -1)
  {
    HMENU hChildMenu = LoadMenu(hInstance, MAKEINTRESOURCE(m_ChildMenuResource));

    char szPopupTitle[80];
    GetMenuString(hChildMenu, 0, szPopupTitle, 80, MF_BYPOSITION);
    
    InsertMenu(hNewMenu,
               iWindowMenu,
               MF_POPUP | MF_BYPOSITION | MF_STRING,
               (UINT_PTR)GetSubMenu(hChildMenu, 0),
               szPopupTitle);

    iWindowMenu++;
  }

  // set the new menu
  CMenu* pNewMenu = CMenu::FromHandle(hNewMenu);
  MDISetMenu(pNewMenu, pNewMenu->GetSubMenu(iWindowMenu));
  DrawMenuBar();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnClose()
{
  // ask if the child windows should be destroyed
  for (int i = 0; i < m_DocumentWindows.GetSize(); i++)
  {
    if (m_DocumentWindows[i]->Close())
      m_DocumentWindows[i]->DestroyWindow();
    else
      return;
  }

  // save the command bar state
  SaveBarState(szBarState);

  // close the project
  CloseProject();

  // store the window state (0 = normal, 1 = maximized, 2 = minimized)
  if (IsIconic())
    GetStateServer()->SetInt("WindowState", 2);
  else if (IsZoomed())
    GetStateServer()->SetInt("WindowState", 1);
  else
    GetStateServer()->SetInt("WindowState", 0);

  // store the window coordinates
  if (GetStateServer()->GetInt("WindowState") == 0)
  {
    RECT rect;
    GetWindowRect(&rect);
    GetStateServer()->SetInt("Startup:left",   rect.left);
    GetStateServer()->SetInt("Startup:top",    rect.top);
    GetStateServer()->SetInt("Startup:right",  rect.right);
    GetStateServer()->SetInt("Startup:bottom", rect.bottom);
  }

  // finally, destroy the window
  DestroyWindow();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileOpen()
{
  char games_directory[MAX_PATH];
  GetGamesDirectory(games_directory);

  SetCurrentDirectory(games_directory);
  CFileDialog FileDialog(TRUE,
                         "inf",
                         NULL,
                         OFN_FILEMUSTEXIST | OFN_HIDEREADONLY | OFN_ALLOWMULTISELECT,
                         "All Game Files|game.inf;*.rmp;*.rss;*.midi;*.mid;*.it;*.xm;*.s3m;*.mod;*.mp3;*.wav;*.raw;*.ss;*.ssl;*.bmp;*.png;*.pcx;*.flic;*.flc;*.fli;*.rws;*.rfn|" \
                         "Sphere Game Files (game.inf)|game.inf|" \
                         "Sphere Map Files (*.rmp)|*.rmp|" \
                         "Sphere Spriteset Files (*.rss)|*.rss|" \
                         "Music Files (*.midi,*.mid,*.it,*.xm,*.s3m,*.mod,*.mp3,*.wav)|*.midi;*.mid;*.it;*.xm;*.s3m;*.mod;*.mp3;*.wav|" \
                         "Sound Files (*.wav,*.raw)|*.wav;*.raw|" \
                         "SphereScript Files (*.ss;*.ssl)|*.ss;*.ssl|" \
                         "Sphere Image Files (*.png,*.bmp,*.pcx)|*.i32;*.png;*.bmp;*.pcx|" \
                         "AutoDesk Animation Files (*.flic,*.flc,*.fli)|*.flic;*.flc;*.fli|" \
                         "Sphere WindowStyle Files (*.rws)|*.rws|" \
                         "Sphere Font Files (*.rfn)|*.rfn||");
  
  if (FileDialog.DoModal() == IDOK)
  {
    POSITION pos;
    pos = FileDialog.GetStartPosition();

    while (pos != NULL)
    {
      CString thePath;
      thePath = FileDialog.GetNextPathName(pos);

      if (!stricmp(thePath.Right(8), "game.inf"))
        OpenProject(thePath);
      else if (!stricmp(thePath.Right(4), ".rmp"))
        m_DocumentWindows.Add(new CMapWindow(&m_StatusBar, thePath));
      else if (!stricmp(thePath.Right(4), ".rss"))
        m_DocumentWindows.Add(new CSpritesetWindow(thePath));
      else if (!stricmp(thePath.Right(4), ".mid") ||
               !stricmp(thePath.Right(5), ".midi") ||
               !stricmp(thePath.Right(3), ".it") ||
               !stricmp(thePath.Right(3), ".xm") ||
               !stricmp(thePath.Right(4), ".s3m") ||
               !stricmp(thePath.Right(4), ".mod") ||
               !stricmp(thePath.Right(4), ".mp3"))
        m_DocumentWindows.Add(new CMusicWindow(thePath));
      else if (!stricmp(thePath.Right(4), ".wav") ||
               !stricmp(thePath.Right(4), ".raw"))
        m_DocumentWindows.Add(new CSoundWindow(thePath));
      else if (!stricmp(thePath.Right(3), ".ss"))
        m_DocumentWindows.Add(new CScriptWindow(&m_StatusBar, thePath));
      else if (!stricmp(thePath.Right(4), ".png") ||
               !stricmp(thePath.Right(4), ".bmp") ||
               !stricmp(thePath.Right(4), ".pcx"))
        m_DocumentWindows.Add(new CImageWindow(thePath));
      else if (!stricmp(thePath.Right(5), ".flic") ||
               !stricmp(thePath.Right(4), ".flc") ||
               !stricmp(thePath.Right(4), ".fli"))
        m_DocumentWindows.Add(new CAnimationWindow(thePath));
      else if (!stricmp(thePath.Right(4), ".rws"))
        m_DocumentWindows.Add(new CWindowStyleWindow(thePath));
      else if (!stricmp(thePath.Right(4), ".rfn"))
        m_DocumentWindows.Add(new CFontWindow(thePath));
      else
        MessageBox("Unknown Document Type");
    }
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileNewProject()
{
  CNewProjectDialog newprojectdialog(this);
    
  if (newprojectdialog.DoModal() == IDOK)
  {
    char projectname[32];
    char gametitle[32];

    strncpy(projectname, newprojectdialog.GetProjectName(), 32);
    strncpy(gametitle, newprojectdialog.GetGameTitle(), 32);

    CreateProject(projectname, gametitle);
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileOpenProject()
{
  char games_directory[MAX_PATH];
  GetGamesDirectory(games_directory);

  SetCurrentDirectory(games_directory);
  CFileDialog FileDialog(TRUE,
                         "inf",
                         "game.inf",
                         OFN_FILEMUSTEXIST | OFN_HIDEREADONLY,
                         "Sphere Game Files (game.inf)|game.inf||");
  if (FileDialog.DoModal() == IDOK)
    OpenProject(FileDialog.GetPathName());
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileCloseProject()
{
  CloseProject();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileNewMap()
{
  m_DocumentWindows.Add(new CMapWindow(&m_StatusBar));
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileNewScript()
{
  m_DocumentWindows.Add(new CScriptWindow(&m_StatusBar));
}

////////////////////////////////////////////////////////////////////////////////

#define FILE_NEW_HANDLER(name)                  \
afx_msg void                                    \
CMainWindow::OnFileNew##name##()                \
{                                               \
  m_DocumentWindows.Add(new C##name##Window()); \
}

FILE_NEW_HANDLER(Spriteset)
FILE_NEW_HANDLER(Font)
FILE_NEW_HANDLER(WindowStyle)
FILE_NEW_HANDLER(Image)

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileOpenMap()
{
  CMapFileDialog Dialog(FDM_OPEN, true, true);
  if (Dialog.DoModal() == IDOK)
  {
    POSITION pos;
    pos = Dialog.GetStartPosition();

    while (pos != NULL)
      m_DocumentWindows.Add(new CMapWindow(&m_StatusBar, Dialog.GetNextPathName(pos)));
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileOpenScript()
{
  CScriptFileDialog Dialog(FDM_OPEN, true, true);
  if (Dialog.DoModal() == IDOK)
  {
    POSITION pos;
    pos = Dialog.GetStartPosition();

    while (pos != NULL)
      m_DocumentWindows.Add(new CScriptWindow(&m_StatusBar, Dialog.GetNextPathName(pos)));
  }
}

////////////////////////////////////////////////////////////////////////////////

#define FILE_OPEN_HANDLER(name)                                       \
afx_msg void                                                          \
CMainWindow::OnFileOpen##name##()                                     \
{                                                                     \
  C##name##FileDialog Dialog(FDM_OPEN, true, true);                   \
  if (Dialog.DoModal() == IDOK)                                       \
  {                                                                   \
    POSITION pos;                                                     \
    pos = Dialog.GetStartPosition();                                  \
                                                                      \
    while (pos != NULL)                                               \
      m_DocumentWindows.Add(new C##name##Window(Dialog.GetNextPathName(pos))); \
  }                                                                   \
}

FILE_OPEN_HANDLER(Spriteset)
FILE_OPEN_HANDLER(Sound)
FILE_OPEN_HANDLER(Music)
FILE_OPEN_HANDLER(Font)
FILE_OPEN_HANDLER(WindowStyle)
FILE_OPEN_HANDLER(Image)
FILE_OPEN_HANDLER(Animation)

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileImportImageToMap()
{
  // get name of image
  CImageFileDialog FileDialog(FDM_OPEN);
  if (FileDialog.DoModal() != IDOK)
    return;

  CString filename = FileDialog.GetPathName();

  CResizeDialog resize_dialog("Tile Dimensions", 16, 16);
  if (resize_dialog.DoModal() != IDOK)
    return;

  // load image
  CImage32 image;
  if (image.Load(filename) == false)
  {
    MessageBox("Error: Could not load image '" + FileDialog.GetFileName() + "'");
    return;
  }

  // build map from image
  sMap map;
  if (map.BuildFromImage(image, resize_dialog.GetWidth(), resize_dialog.GetHeight()) == false)
  {
    MessageBox("Error: Could not build map from image");
    return;
  }

  char* fn = new char[strlen(filename) + 10];
  strcpy(fn, filename);

  strcpy(strrchr(fn, '.'), ".rmp");
  map.Save(fn);

  delete[] fn;

  MessageBox("Conversion successful");
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileImportBitmapToRWS()
{
  CImageFileDialog InFileDialog(FDM_OPEN);
  if (InFileDialog.DoModal() == IDCANCEL)
    return;

  CWindowStyleFileDialog OutFileDialog(FDM_SAVE);
  if (OutFileDialog.DoModal() == IDCANCEL)
    return;

  sWindowStyle ws;
  if (ws.Import(InFileDialog.GetPathName(), rgbaMagenta) == false)
  {
    MessageBox("Can't Import file, either file is invalid \nor not a 3x3 bitmap", "Error");
    return;
  }

  if (ws.Save(OutFileDialog.GetPathName()) == false)
  {
    MessageBox("Can't Save file!");
    return;
  }

  MessageBox("Import Successful!");
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileImportBitmapToRSS()
{
  CImageFileDialog InFileDialog(FDM_OPEN);
  if (InFileDialog.DoModal() != IDOK)
    return;

  CSpritesetFileDialog OutFileDialog(FDM_SAVE);
  if (OutFileDialog.DoModal() != IDOK)
    return;

  CResizeDialog ResizeDialog("Frame Size", 16, 32);
  ResizeDialog.SetRange(1, 4096, 1, 4096);
  if (ResizeDialog.DoModal() != IDOK)
    return;

  sSpriteset sprite;
  if (sprite.Import_BMP(InFileDialog.GetPathName(), ResizeDialog.GetWidth(), ResizeDialog.GetHeight(), rgbaMagenta) == false)
  {
    MessageBox("Can't Import file", "Error");
    return;
  }

  if (sprite.Save(OutFileDialog.GetPathName()) == false)
  {
    MessageBox("Can't Save file!");
    return;
  }

  MessageBox("Import Successful!");
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnFileOptions()
{
  COptionsDialog().DoModal();
}

////////////////////////////////////////////////////////////////////////////////

#define PROJECT_INSERT_HANDLER(type, group_type)                               \
afx_msg void                                                                   \
CMainWindow::OnProjectInsert##type()                                           \
{                                                                              \
  InsertProjectFile(&C##type##FileDialog(FDM_OPEN, false), group_type);        \
}

PROJECT_INSERT_HANDLER(Map,         GT_MAPS)
PROJECT_INSERT_HANDLER(Spriteset,   GT_SPRITESETS)
PROJECT_INSERT_HANDLER(Script,      GT_SCRIPTS)
PROJECT_INSERT_HANDLER(Sound,       GT_SOUNDS)
PROJECT_INSERT_HANDLER(Music,       GT_MUSIC)
PROJECT_INSERT_HANDLER(Font,        GT_FONTS)
PROJECT_INSERT_HANDLER(WindowStyle, GT_WINDOWSTYLES)
PROJECT_INSERT_HANDLER(Image,       GT_IMAGES)
PROJECT_INSERT_HANDLER(Animation,   GT_ANIMATIONS)

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnProjectRefresh()
{
  m_ProjectWindow->Update();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnProjectRunSphere()
{
  char szCommandLine[MAX_PATH + 80];
  GetStateServer()->GetString("SphereDirectory", szCommandLine, MAX_PATH);
  strcat(szCommandLine, "\\engine.exe -game ");
  strcat(szCommandLine, m_Project.GetGameSubDirectory());

  STARTUPINFO si;
  memset(&si, 0, sizeof(si));
  si.cb = sizeof(si);

  PROCESS_INFORMATION pi;
  
  char sphere_directory[MAX_PATH];
  GetStateServer()->GetString("SphereDirectory", sphere_directory, MAX_PATH);

  BOOL retval = CreateProcess(
    NULL,                  // lpApplicationName
    szCommandLine,         // lpCommandLine
    NULL,                  // lpProcessAttributes
    NULL,                  // lpThreadAttributes
    FALSE,                 // bInheritHandles
    0,                     // dwCreationFlags
    NULL,                  // lpEnvironment
    sphere_directory,      // lpCurrentDirectory
    &si,                   // lpStartupInfo
    &pi);                  // lpProcessInformation
  if (retval == FALSE)
    MessageBox("Error: Could not execute Sphere engine");
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnProjectConfigureSphere()
{
  char szCommandLine[MAX_PATH + 80];
  GetStateServer()->GetString("SphereDirectory", szCommandLine, MAX_PATH);
  strcat(szCommandLine, "\\config.exe");

  STARTUPINFO si;
  memset(&si, 0, sizeof(si));
  si.cb = sizeof(si);

  PROCESS_INFORMATION pi;
  
  char sphere_directory[MAX_PATH];
  GetStateServer()->GetString("SphereDirectory", sphere_directory, MAX_PATH);

  BOOL retval = CreateProcess(
    NULL,                  // lpApplicationName
    szCommandLine,         // lpCommandLine
    NULL,                  // lpProcessAttributes
    NULL,                  // lpThreadAttributes
    FALSE,                 // bInheritHandles
    0,                     // dwCreationFlags
    NULL,                  // lpEnvironment
    sphere_directory,      // lpCurrentDirectory
    &si,                   // lpStartupInfo
    &pi);                  // lpProcessInformation
  if (retval == FALSE)
    MessageBox("Error: Could not configure Sphere");
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnHelpContents()
{
  HtmlHelp(m_hWnd, "help\\sde.chm", HH_DISPLAY_TOC, 0);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnHelpAbout()
{
  char szAboutText[520];
  sprintf(szAboutText,
    "SDE\n"
    "Sphere Development Environment\n"
    "Chad Austin (c) 1999\n"
    "Additional code and icons by Darklich\n\n"
    "%s\n"
    "%s\n",
    __DATE__,
    __TIME__);

  MessageBox(szAboutText, "About");
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnUpdateProjectCommand(CCmdUI* cmdui)
{
  if (m_ProjectOpen)
    cmdui->Enable(TRUE);
  else
    cmdui->Enable(FALSE);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CMainWindow::OnUpdateSaveCommand(CCmdUI* cmdui)
{
  CWnd* pWindow = MDIGetActive();
  if (pWindow == NULL)
    cmdui->Enable(FALSE);
  else
  {
    if (GetWindowLong(pWindow->m_hWnd, GWL_USERDATA) & WA_SAVEABLE)
      cmdui->Enable(TRUE);
    else
      cmdui->Enable(FALSE);
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg LRESULT
CMainWindow::OnDocumentWindowClosing(WPARAM wparam, LPARAM lparam)
{
  CDocumentWindow* window = (CDocumentWindow*)lparam;

  // remove window from list
  for (int i = 0; i < m_DocumentWindows.GetSize(); i++)
    if (m_DocumentWindows[i] == window)
    {
      m_DocumentWindows.RemoveAt(i);
      return 0;
    }

  return 0;
}

////////////////////////////////////////////////////////////////////////////////

afx_msg LRESULT
CMainWindow::OnSetChildMenu(WPARAM wparam, LPARAM lparam)
{
  m_ChildMenuResource = wparam;
  UpdateMenu();
  return 0;
}

////////////////////////////////////////////////////////////////////////////////

afx_msg LRESULT
CMainWindow::OnClearChildMenu(WPARAM wparam, LPARAM lparam)
{
  m_ChildMenuResource = -1;
  UpdateMenu();
  return 0;
}

////////////////////////////////////////////////////////////////////////////////
