#include "LayerView.hpp"
#include "LayerPropertiesDialog.hpp"
#include "../common/Map.hpp"
#include "menu.h"
#include "resource.h"


const int LAYER_BUTTON_WIDTH  = 100;
const int LAYER_BUTTON_HEIGHT = 20;


static int s_iLayerViewID = 5000;


BEGIN_MESSAGE_MAP(CLayerView, CWnd)

  ON_WM_PAINT()
  ON_WM_LBUTTONDOWN()
  ON_WM_RBUTTONUP()
  ON_WM_MOUSEMOVE()

  ON_COMMAND(ID_LAYERVIEW_VISIBLE,     OnToggleVisible)
  ON_COMMAND(ID_LAYERVIEW_BASELAYER,   OnSetBaseLayer)
  ON_COMMAND(ID_LAYERVIEW_INSERTLAYER, OnInsertLayer)
  ON_COMMAND(ID_LAYERVIEW_DELETELAYER, OnDeleteLayer)
  ON_COMMAND(ID_LAYERVIEW_PROPERTIES,  OnLayerProperties)
  ON_COMMAND(ID_LAYERVIEW_MOVEDOWN,    OnLayerMoveDown)
  ON_COMMAND(ID_LAYERVIEW_MOVEUP,      OnLayerMoveUp)
  ON_COMMAND(ID_LAYERVIEW_FILL,        OnLayerFill)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CLayerView::CLayerView()
: m_pMap(NULL)
, m_iSelectedLayer(0)
, m_iSelectedTile(0)
{
}

////////////////////////////////////////////////////////////////////////////////

CLayerView::~CLayerView()
{
}

////////////////////////////////////////////////////////////////////////////////

BOOL
CLayerView::Create(CStatusBar* pStatusBar, CLayerViewHandler* pHandler, CWnd* pParentWindow, sMap* pMap)
{
  m_pStatusBar = pStatusBar;
  m_pHandler = pHandler;
  m_pMap = pMap;

  return CWnd::Create(
    AfxRegisterWndClass(0, LoadCursor(NULL, IDC_ARROW), NULL, LoadIcon(NULL, IDI_APPLICATION)),
    "LayerView",
    WS_CHILD | WS_VISIBLE | WS_VSCROLL,
    CRect(0, 0, 0, 0),
    pParentWindow,
    s_iLayerViewID++);
}

////////////////////////////////////////////////////////////////////////////////

int
CLayerView::GetSelectedLayer() const
{
  return m_iSelectedLayer;
}

////////////////////////////////////////////////////////////////////////////////

void
CLayerView::SetSelectedTile(int iTile)
{
  m_iSelectedTile = iTile;
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnPaint()
{
  RECT ClientRect;
  GetClientRect(&ClientRect);

  CPaintDC dc(this);

  for (int i = 0; i < ClientRect.bottom / LAYER_BUTTON_HEIGHT + 1; i++)
  {
    RECT Rect =
    {
      0,
      i * LAYER_BUTTON_HEIGHT,
      LAYER_BUTTON_WIDTH,
      (i + 1) * LAYER_BUTTON_HEIGHT
    };
    
    // visibility check
    if (!dc.RectVisible(&Rect))
      continue;

    if (i < m_pMap->GetNumLayers())
    {
      // draw layers upside down!!
      int layer = m_pMap->GetNumLayers() - i - 1;

      // draw the button
      dc.FillRect(&Rect, CBrush::FromHandle((HBRUSH)(COLOR_MENU + 1)));

      if (layer == m_iSelectedLayer)
      {
        // draw it pushed in
        dc.Draw3dRect(&Rect, RGB(0x00, 0x00, 0x00), RGB(0xFF, 0xFF, 0xFF));
        InflateRect(&Rect, -1, -1);
        dc.Draw3dRect(&Rect, RGB(0x40, 0x40, 0x40), RGB(0xD0, 0xD0, 0xD0));
      }
      else
      {
        // draw it popped out
        dc.Draw3dRect(&Rect, RGB(0xFF, 0xFF, 0xFF), RGB(0x00, 0x00, 0x00));
        InflateRect(&Rect, -1, -1);
        dc.Draw3dRect(&Rect, RGB(0xD0, 0xD0, 0xD0), RGB(0x40, 0x40, 0x40));
      }

      // draw the text
      CFont* OldFont = (CFont*)dc.SelectStockObject(DEFAULT_GUI_FONT);
      dc.SetTextColor(RGB(0x00, 0x00, 0x00));
      dc.SetBkMode(TRANSPARENT);

      dc.DrawText(
        m_pMap->GetLayer(layer).GetName(),
        -1,
        &Rect,
        DT_SINGLELINE | DT_CENTER | DT_VCENTER);

      dc.SelectObject(OldFont);
    }
    else
    {
      // draw an empty rectangle
      dc.FillRect(&Rect, CBrush::FromHandle((HBRUSH)(COLOR_APPWORKSPACE + 1)));
    }
  }

}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnLButtonDown(UINT uFlags, CPoint Point)
{
  if (Point.x >= 0 &&
      Point.x < LAYER_BUTTON_WIDTH &&
      Point.y >= 0 &&
      Point.y < LAYER_BUTTON_HEIGHT * m_pMap->GetNumLayers())
  {
    // layers are displayed upside down
    m_iSelectedLayer = (m_pMap->GetNumLayers() * LAYER_BUTTON_HEIGHT - Point.y) / LAYER_BUTTON_HEIGHT;
    Invalidate();

    // tell parent that the layer changed
    m_pHandler->LV_SelectedLayerChanged(m_iSelectedLayer);
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnRButtonUp(UINT uFlags, CPoint Point)
{
  HMENU _menu = LoadMenu(AfxGetApp()->m_hInstance, MAKEINTRESOURCE(IDR_LAYERVIEW));
  HMENU menu = GetSubMenu(_menu, 0);

  // simulate a left click to set the selected tile
  OnLButtonDown(uFlags, Point);

  // make sure the right things are greyed and stuff
  if (Point.x >= 0 &&
      Point.x < LAYER_BUTTON_WIDTH &&
      Point.y >= 0 &&
      Point.y < LAYER_BUTTON_HEIGHT * m_pMap->GetNumLayers())
  {
    // check the base layer if necessary
    if (m_pMap->GetStartLayer() == m_iSelectedLayer)
      CheckMenuItem(menu, ID_LAYERVIEW_BASELAYER, MF_CHECKED);

    if (m_pMap->GetNumLayers() > 1)
      EnableItem(menu, ID_LAYERVIEW_DELETELAYER);
    else
      DisableItem(menu, ID_LAYERVIEW_DELETELAYER);

    EnableItem(menu, ID_LAYERVIEW_PROPERTIES);

    // validate move up and move down commands
    int layer = Point.y / LAYER_BUTTON_HEIGHT;

    if (layer > 0)
      EnableItem(menu, ID_LAYERVIEW_MOVEUP);
    else
      DisableItem(menu, ID_LAYERVIEW_MOVEUP);

    if (layer < m_pMap->GetNumLayers() - 1)
      EnableItem(menu, ID_LAYERVIEW_MOVEDOWN);
    else
      DisableItem(menu, ID_LAYERVIEW_MOVEDOWN);

    ClientToScreen(&Point);
    TrackPopupMenu(menu, TPM_LEFTALIGN | TPM_TOPALIGN, Point.x, Point.y, 0, m_hWnd, NULL);
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnMouseMove(UINT nFlags, CPoint Point)
{
 int layer = Point.y / LAYER_BUTTON_HEIGHT;
 
 if (layer < m_pMap->GetNumLayers())
 {
   CString layername;
   layername.Format("Layer \"%s\"", m_pMap->GetLayer(m_pMap->GetNumLayers() - layer - 1).GetName());
   m_pStatusBar->SetWindowText(layername);
 }
 else
   m_pStatusBar->SetWindowText("");
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnToggleVisible()
{
  // FILL THIS IN
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnSetBaseLayer()
{
  m_pMap->SetStartLayer(m_iSelectedLayer);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnInsertLayer()
{
  m_pMap->InsertLayer(0, sLayer(64, 64));
  m_pMap->GetLayer(0).SetName("Unnamed Layer");

  Invalidate();
  m_pHandler->LV_MapChanged();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnDeleteLayer()
{
  m_pMap->DeleteLayer(m_iSelectedLayer);
  m_pHandler->LV_MapChanged();

  if (m_iSelectedLayer > 0)
  {
    m_iSelectedLayer--;
    m_pHandler->LV_SelectedLayerChanged(m_iSelectedLayer);
  }

  // make sure the base is a valid layer...
  if (m_pMap->GetStartLayer() >= m_pMap->GetNumLayers())
    m_pMap->SetStartLayer(m_pMap->GetNumLayers() - 1);

  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnLayerProperties()
{
  // show the dialog box
  CLayerPropertiesDialog Dialog(&m_pMap->GetLayer(m_iSelectedLayer));
  if (Dialog.DoModal() == IDOK)
  {
    Invalidate();
    m_pHandler->LV_MapChanged();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnLayerMoveUp()
{
  m_pMap->SwapLayers(m_iSelectedLayer, m_iSelectedLayer + 1);
  m_iSelectedLayer++;

  Invalidate();
  m_pHandler->LV_MapChanged();
  m_pHandler->LV_SelectedLayerChanged(m_iSelectedLayer);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnLayerMoveDown()
{
  m_pMap->SwapLayers(m_iSelectedLayer, m_iSelectedLayer - 1);
  m_iSelectedLayer--;

  Invalidate();
  m_pHandler->LV_MapChanged();
  m_pHandler->LV_SelectedLayerChanged(m_iSelectedLayer);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CLayerView::OnLayerFill()
{
  if (MessageBox("Are you sure?", "Fill Layer", MB_YESNO) == IDNO)
    return;

  sLayer& layer = m_pMap->GetLayer(m_iSelectedLayer);

  // fill the current map
  for (int x = 0; x < layer.GetWidth(); x++)
    for (int y = 0; y < layer.GetHeight(); y++)
      layer.SetTile(x, y, m_iSelectedTile);
    
  m_pHandler->LV_MapChanged();
}

////////////////////////////////////////////////////////////////////////////////
