#include "FontWindow.hpp"
#include "FileDialogs.hpp"
#include "ResizeDialog.hpp"
#include "x++.hpp"
#include "resource.h"


#define IDC_FONTSCROLL 900

#define MIN_CHARACTER 32
#define MAX_CHARACTER 126

#define DEFAULT_CHARACTER_WIDTH  8
#define DEFAULT_CHARACTER_HEIGHT 12

#define ALPHA_WIDTH      32
#define SCROLLBAR_HEIGHT 16


BEGIN_MESSAGE_MAP(CFontWindow, CSaveableDocumentWindow)

  ON_WM_SIZE()
  ON_WM_HSCROLL()

  ON_COMMAND(ID_FONT_RESIZE,     OnFontResize)
  ON_COMMAND(ID_FONT_SIMPLIFY,   OnFontSimplify)

  ON_COMMAND(ID_FONT_SLIDELEFT,  OnFontSlideLeft)
  ON_COMMAND(ID_FONT_SLIDEUP,    OnFontSlideUp)
  ON_COMMAND(ID_FONT_SLIDERIGHT, OnFontSlideRight)
  ON_COMMAND(ID_FONT_SLIDEDOWN,  OnFontSlideDown)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CFontWindow::CFontWindow()
: CSaveableDocumentWindow("", IDR_FONT)
, m_CurrentCharacter(MIN_CHARACTER)
, m_CurrentAlpha(0)
, m_Created(false)
{
  m_Font.SetNumCharacters(256);
  for (int i = 0; i < 256; i++)
    m_Font.GetCharacter(i).Resize(DEFAULT_CHARACTER_WIDTH, DEFAULT_CHARACTER_HEIGHT);

  SetModified(true);
  SetSaved(false);

  Create();
}

////////////////////////////////////////////////////////////////////////////////

CFontWindow::CFontWindow(const char* font)
: CSaveableDocumentWindow(font, IDR_FONT)
, m_CurrentCharacter(MIN_CHARACTER)
, m_CurrentAlpha(0)
, m_Created(false)
{
  if (m_Font.Load(font))
  {
    SetModified(false);
  }
  else
  {
    AfxGetApp()->m_pMainWnd->MessageBox("Could not load font, creating new");
    m_Font.SetNumCharacters(256);
    for (int i = 0; i < 256; i++)
      m_Font.GetCharacter(i).Resize(DEFAULT_CHARACTER_WIDTH, DEFAULT_CHARACTER_HEIGHT);

    SetModified(true);
  }

  SetSaved(true);

  Create();
}

////////////////////////////////////////////////////////////////////////////////

CFontWindow::~CFontWindow()
{
}

////////////////////////////////////////////////////////////////////////////////

RGBA
CFontWindow::GetFontColor(int pixel)
{
  RGBA rgba = { pixel, pixel, pixel, 255 };
  return rgba;
}

////////////////////////////////////////////////////////////////////////////////

void
CFontWindow::Create()
{
  // create window
  CSaveableDocumentWindow::Create(GetDefaultWindowClass());
  
  // create children
  m_ImageView.Create(this, this);
  m_AlphaView.Create(this, this);

  m_ScrollBar.Create(WS_CHILD | WS_VISIBLE | SBS_HORZ, CRect(0, 0, 0, 0), this, IDC_FONTSCROLL);
  m_ScrollBar.SetScrollRange(MIN_CHARACTER, MAX_CHARACTER);
  m_ScrollBar.SetScrollPos(m_CurrentCharacter);

  m_Created = true;

  // make sure everything is the right size
  RECT ClientRect;
  GetClientRect(&ClientRect);
  OnSize(0, ClientRect.right, ClientRect.bottom);

  SetImage();
  m_AlphaView.SetAlpha(m_CurrentAlpha);

  UpdateWindowTitle();
}

////////////////////////////////////////////////////////////////////////////////

void
CFontWindow::UpdateWindowTitle()
{
  char title[520];
  strcpy(title, GetDocumentTitle());

  char append[520];
  sprintf(append, " - %c - %d/%d",
    (char)m_CurrentCharacter,
    m_CurrentCharacter,
    m_Font.GetNumCharacters());
  strcat(title, append);

  SetCaption(title);
}

////////////////////////////////////////////////////////////////////////////////

void
CFontWindow::SetImage()
{
  sFontCharacter& c = m_Font.GetCharacter(m_CurrentCharacter);
  RGBA* pixels = new RGBA[c.GetWidth() * c.GetHeight()];
  for (int i = 0; i < c.GetWidth() * c.GetHeight(); i++)
  {
    pixels[i] = GetFontColor(c.GetPixels()[i]);
  }
  m_ImageView.SetImage(c.GetWidth(), c.GetHeight(), pixels);
  delete[] pixels;

  m_ScrollBar.SetScrollPos(m_CurrentCharacter);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CFontWindow::OnSize(UINT type, int cx, int cy)
{
  if (m_Created)
  {
    m_ImageView.MoveWindow(0, 0, cx - ALPHA_WIDTH, cy - SCROLLBAR_HEIGHT);
    m_AlphaView.MoveWindow(cx - ALPHA_WIDTH, 0, ALPHA_WIDTH, cy - SCROLLBAR_HEIGHT);
    m_ScrollBar.MoveWindow(0, cy - SCROLLBAR_HEIGHT, cx, SCROLLBAR_HEIGHT);
  }

  CSaveableDocumentWindow::OnSize(type, cx, cy);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CFontWindow::OnHScroll(UINT sbcode, UINT pos, CScrollBar* scroll_bar)
{
  // scroll bar
  if (scroll_bar->m_hWnd == m_ScrollBar.m_hWnd)
  {
    switch (sbcode)
    {
      case SB_LEFT:
        m_CurrentCharacter = 0;
        break;

      case SB_RIGHT:
        m_CurrentCharacter = m_Font.GetNumCharacters() - 1;
        break;

      case SB_LINELEFT:
      case SB_PAGELEFT:
        m_CurrentCharacter--;
        break;

      case SB_LINERIGHT:
      case SB_PAGERIGHT:
        m_CurrentCharacter++;
        break;

      case SB_THUMBPOSITION:
      case SB_THUMBTRACK:
        m_CurrentCharacter = pos;
        break;

      default:
        return;
     }

    if (m_CurrentCharacter < MIN_CHARACTER)
      m_CurrentCharacter = MIN_CHARACTER;
    if (m_CurrentCharacter > MAX_CHARACTER)
      m_CurrentCharacter = MAX_CHARACTER;

    SetImage();
    UpdateWindowTitle();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CFontWindow::OnFontResize()
{
  sFontCharacter& c = m_Font.GetCharacter(m_CurrentCharacter);
  CResizeDialog Dialog("Resize Font Character", c.GetWidth(), c.GetHeight());
  if (Dialog.DoModal() == IDOK)
  {
    c.Resize(Dialog.GetWidth(), Dialog.GetHeight());
    SetModified(true);
    SetImage();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CFontWindow::OnFontSimplify()
{
  if (MessageBox("This will convert the entire font to black and white.\nIs this okay?", NULL, MB_YESNO) == IDYES)
  {
    for (int i = 0; i < m_Font.GetNumCharacters(); i++)
    {
      sFontCharacter& c = m_Font.GetCharacter(i);
      for (int j = 0; j < c.GetWidth() * c.GetHeight(); j++)
      {
        if (c.GetPixels()[j] < 128)
          c.GetPixels()[j] = 0;
        else
          c.GetPixels()[j] = 255;
      }
    }

    SetModified(true);
    SetImage();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CFontWindow::OnFontSlideLeft()
{
  m_Font.GetCharacter(m_CurrentCharacter).SlideLeft();

  SetModified(true);
  SetImage();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CFontWindow::OnFontSlideUp()
{
  m_Font.GetCharacter(m_CurrentCharacter).SlideUp();

  SetModified(true);
  SetImage();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CFontWindow::OnFontSlideRight()
{
  m_Font.GetCharacter(m_CurrentCharacter).SlideRight();

  SetModified(true);
  SetImage();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CFontWindow::OnFontSlideDown()
{
  m_Font.GetCharacter(m_CurrentCharacter).SlideDown();

  SetModified(true);
  SetImage();
}

////////////////////////////////////////////////////////////////////////////////

bool
CFontWindow::GetSavePath(char* path)
{
  CFontFileDialog Dialog(FDM_SAVE);
  if (Dialog.DoModal() != IDOK)
    return false;

  strcpy(path, Dialog.GetPathName());
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CFontWindow::SaveDocument(const char* path)
{
  return m_Font.Save(path);
}

////////////////////////////////////////////////////////////////////////////////

void
CFontWindow::IV_ImageChanged()
{
  sFontCharacter& c = m_Font.GetCharacter(m_CurrentCharacter);
  for (int iy = 0; iy < c.GetHeight(); iy++)
    for (int ix = 0; ix < c.GetWidth(); ix++)
      c.GetPixels()[iy * c.GetWidth() + ix] = m_ImageView.GetPixels()[iy * c.GetWidth() + ix].red;

  SetModified(true);
  UpdateWindowTitle();
}

////////////////////////////////////////////////////////////////////////////////

void
CFontWindow::IV_ColorChanged(RGBA color)
{
  m_CurrentAlpha = color.red;
  m_AlphaView.SetAlpha(m_CurrentAlpha);
}

////////////////////////////////////////////////////////////////////////////////

void
CFontWindow::AV_AlphaChanged(byte alpha)
{
  m_CurrentAlpha = alpha;
  RGBA c = { alpha, alpha, alpha, 255 };
  m_ImageView.SetColor(c);
}

////////////////////////////////////////////////////////////////////////////////
