#include "AudioSetupDialog.hpp"
#include <mmsystem.h>
#include <stdio.h>
#include "Audio.hpp"
#include "StateServer.hpp"


static int s_InitializeCount;

static bool s_AudioReady = false;
static bool s_MidiReady = false;

// Fmod internal configuration
static unsigned long FSOUND_BaseMode;
static bool          StereoMode;

////////////////////////////////////////////////////////////////////////////////

static bool
InitializeAudio()
{
  if (++s_InitializeCount > 1)
    return true;

  int driver        = GetStateServer()->GetInt("AudioDriver",       0);
  int mixer         = GetStateServer()->GetInt("AudioMixer",        0);
  int master_volume = GetStateServer()->GetInt("AudioMasterVolume", 255);
  int quality       = GetStateServer()->GetInt("AudioSampleRate",   44100);
  int stereo        = GetStateServer()->GetInt("AudioStereo",       1);
  int bit_rate      = GetStateServer()->GetInt("AudioBitRate",      16);

  // set up the mixers
  FSOUND_MIXERTYPES iCurrentMixer;
  switch(mixer)
  {
    case 0: iCurrentMixer = FSOUND_MIXER_AUTODETECT;         break;
    case 1: iCurrentMixer = FSOUND_MIXER_QUALITY_AUTODETECT; break;
    case 2: iCurrentMixer = FSOUND_MIXER_BLENDMODE;          break;
    case 3: iCurrentMixer = FSOUND_MIXER_MMXP5;              break;
    case 4: iCurrentMixer = FSOUND_MIXER_MMXP6;              break;
    case 5: iCurrentMixer = FSOUND_MIXER_QUALITY_FPU;        break;
    case 6: iCurrentMixer = FSOUND_MIXER_QUALITY_MMXP5;      break;
    case 7: iCurrentMixer = FSOUND_MIXER_QUALITY_MMXP6;      break;
    // in case some one screws around the sound settings
    default: iCurrentMixer = FSOUND_MIXER_AUTODETECT;        break;
  }

  if (driver == 0)
    FSOUND_SetOutput(FSOUND_OUTPUT_DSOUND);
  else if (driver == 1)
    FSOUND_SetOutput(FSOUND_OUTPUT_WINMM);

  if (!FSOUND_SetMixer(iCurrentMixer) ||
      !(FSOUND_SetSFXMasterVolume(master_volume), true) ||  // SetSFXMasterVolume returns true so we make it so the expression calls the function and returns true
      !FSOUND_Init(quality, 64, 0))
  {
    s_AudioReady = false;
    return false;
  }

  // stereo mode?
  StereoMode = stereo ? true : false;

  // set up the base sound playback modes
  FSOUND_BaseMode = FSOUND_2D | FSOUND_UNSIGNED;
  if (StereoMode)
    FSOUND_BaseMode |= FSOUND_STEREO;
  else
    FSOUND_BaseMode |= FSOUND_MONO;

  if (bit_rate == 16)
    FSOUND_BaseMode |= FSOUND_16BITS;
  else
    FSOUND_BaseMode |= FSOUND_8BITS;

  s_AudioReady = true;
  
  #ifdef DMUSIC_LUV
    if (FAILED(CoInitialize(NULL)))
      return false;
  #endif

  s_MidiReady = true;
  return true;
}

////////////////////////////////////////////////////////////////////////////////

static bool
ShutdownAudio()
{
  if (--s_InitializeCount > 0)
    return true;    

  FSOUND_Close();
  s_AudioReady = false;

  #ifdef DMUSIC_LUV
    if (s_bMidiReady)
    {
      CoUninitialize();
      s_MidiReady = false;
    }
  #endif

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
ConfigureAudio()
{
  CAudioSetupDialog AudioSetupDialog;
  if (AudioSetupDialog.DoModal() != IDOK)
    return false;
  return true;
}

////////////////////////////////////////////////////////////////////////////////

CModule::CModule()
: m_Loaded(false)
, m_SongVolume(0)
, m_Channel(-1)
{
  InitializeAudio();
}

////////////////////////////////////////////////////////////////////////////////

CModule::~CModule()
{
  if (!s_MidiReady)
  {
    ShutdownAudio();
    return;
  }

  if (m_Loaded)
    if (m_MusicType == MUSIC_MIDI)
    {
      #ifdef DMUSIC_LUV
        m_Music.m_Midi->Stop(NULL, NULL, 0, 0);
        m_MidiSegment->Release();
        m_Music.m_Midi->CloseDown();
        m_Music.m_Midi->Release();
        m_MidiLoader->Release();
        m_DirectMusic->Release();
        //DestroyWindow(m_Music.m_mciWindow);
      #else
        mciSendString("stop midi", NULL, NULL, NULL);
        mciSendString("close midi", NULL, NULL, NULL);
      #endif
    }
    else if (m_MusicType == MUSIC_MOD)
    {
      FMUSIC_FreeSong(m_Music.m_Module);
    }
    else if (m_MusicType == MUSIC_MP3 || m_MusicType == MUSIC_WAV)
    {
      if (m_Music.m_Stream != NULL)
      {
        FSOUND_Stream_Stop(m_Music.m_Stream);
        FSOUND_Stream_Close(m_Music.m_Stream);
      }
    }

  ShutdownAudio();
}

////////////////////////////////////////////////////////////////////////////////

bool
CModule::Load(const char* filename)
{
  if (!s_AudioReady)
    return true;

  int x = strlen(filename);

  if (stricmp(filename + x - 4, ".mid") == 0 ||
      stricmp(filename + x - 5, ".midi") == 0  )
    m_MusicType = MUSIC_MIDI;
  else if (stricmp(filename + x - 4, ".mp3") == 0 ||
           stricmp(filename + x - 4, ".mp2") == 0   )
    m_MusicType = MUSIC_MP3;
  else if (stricmp(filename + x - 4, ".wav") == 0)
    m_MusicType = MUSIC_WAV;
  else if (stricmp(filename + x - 3, ".it" ) == 0 ||
           stricmp(filename + x - 3, ".xm" ) == 0 ||
           stricmp(filename + x - 4, ".s3m") == 0 ||
           stricmp(filename + x - 4, ".mod") == 0   )
    m_MusicType = MUSIC_MOD;
  else
    return false;
           


  // if it was a midi...
  if (m_MusicType == MUSIC_MIDI)
  {
    #if DMUSIC_LUV
      if (!s_bMidiReady)
        return true;

      if (FAILED(CoCreateInstance(CLSID_DirectMusicPerformance, NULL, CLSCTX_INPROC, IID_IDirectMusicPerformance2, (void**)&m_Music.m_Midi)))
        return false;

      if (FAILED(CoCreateInstance(CLSID_DirectMusicLoader, NULL, CLSCTX_INPROC, IID_IDirectMusicLoader, (void**)&m_MidiLoader)))
      {
        m_Music.m_Midi->Release();
        return false;
      }

      DMUS_OBJECTDESC ObjDesc;
      WCHAR wcFilename[MAX_PATH];

      ObjDesc.dwSize = sizeof(DMUS_OBJECTDESC);
      ObjDesc.dwValidData = DMUS_OBJ_CLASS | DMUS_OBJ_FILENAME | DMUS_OBJ_FULLPATH;
      ObjDesc.guidClass = CLSID_DirectMusicSegment;
      MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED, filename, -1, wcFilename, MAX_PATH);
      wcscpy(ObjDesc.wszFileName, wcFilename);
      //wcscpy(ObjDesc.wszFileName, L"c:\\temp\\COSTA.mid");

      if (FAILED(m_MidiLoader->GetObject(&ObjDesc, IID_IDirectMusicSegment2, (void**)&m_MidiSegment)))
      {
        m_MidiLoader->Release();
        m_Music.m_Midi->Release();
        return false;
      }
    
      m_DirectMusic = NULL;
      m_Music.m_Midi->Init(&m_DirectMusic, NULL, NULL);
      m_Music.m_Midi->AddPort(NULL);
      m_MidiSegment->SetParam(GUID_StandardMIDIFile, -1, 0, 0, (void*)m_Music.m_Midi);
      m_MidiSegment->SetParam(GUID_Download, -1, 0, 0, (void*)m_Music.m_Midi);
      m_MidiSegment->SetRepeats(-1);
      //m_Music.m_Midi->PlaySegment(m_MidiSegment, 0, 0, &m_MidiSegmentState);
    #else
      char buf[MAX_PATH+100];
      sprintf(buf, "open \"%s\" type sequencer alias midi", filename);
      mciSendString(buf, NULL, NULL, NULL);
      if(mciSendString("play midi", NULL, NULL, NULL))
        return false;
      //if(mciSendString("pause midi", NULL, NULL, NULL))
        //return false;
    #endif

    m_Loaded = true;
  }
  // if it was a mp3 ;)
  else if (m_MusicType == MUSIC_MP3)
  {
    m_Music.m_Stream = FSOUND_Stream_OpenMpeg((char *)filename, FSOUND_BaseMode | FSOUND_LOOP_NORMAL | FSOUND_STREAMABLE);
    if (m_Music.m_Stream == NULL)
      return false;
    
    m_Channel = FSOUND_Stream_Play(FSOUND_FREE, m_Music.m_Stream);
    if (m_Channel == -1)
      return false;

    FSOUND_Stream_SetPaused(m_Music.m_Stream, true);

    m_Loaded = true;
  }
  // Wav playback.
  else if (m_MusicType == MUSIC_WAV)
  {
    m_Music.m_Stream = FSOUND_Stream_OpenWav((char *)filename, FSOUND_BaseMode | FSOUND_LOOP_NORMAL | FSOUND_STREAMABLE);
    if (m_Music.m_Stream == NULL)
      return false;

    m_Channel = FSOUND_Stream_Play(FSOUND_FREE, m_Music.m_Stream);
    if (m_Channel == -1)
      return false;

    FSOUND_Stream_SetPaused(m_Music.m_Stream, true);

    m_Loaded = true;
  }
  // if it was a mod, s3m, xm, or it
  else if (m_MusicType == MUSIC_MOD)
  {
    m_Music.m_Module = FMUSIC_LoadSong((char *)filename);
    if (m_Music.m_Module == NULL)
      return false;

    m_Loaded = true;
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CModule::Play()
{
  if (!s_AudioReady)
    return false;

  if (!m_Loaded)
    return false;

  if (m_MusicType == MUSIC_MIDI)
  {
    if (!IsPlaying())

    #ifdef DMUSIC_LUV
      m_Music.m_Midi->PlaySegment(m_MidiSegment, 0, 0, &m_MidiSegmentState);
    #else
      mciSendString("resume midi", NULL, NULL, NULL);
    #endif
      
    else
      return false;
  }
  else if (m_MusicType == MUSIC_MOD)
  {
    if (!IsPlaying())
    {
      FMUSIC_PlaySong(m_Music.m_Module);
      if (StereoMode)
        FMUSIC_SetPanSeperation(m_Music.m_Module, 1);
      else
        FMUSIC_SetPanSeperation(m_Music.m_Module, 0);
    }
    else
      return false;
  }
  else if (m_MusicType == MUSIC_MP3 || m_MusicType == MUSIC_WAV)
  {
    if (!IsPlaying())
      FSOUND_Stream_SetPaused(m_Music.m_Stream, false);
    else
      return false;
  }
  
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CModule::Stop()
{
  if (!s_AudioReady)
    return false;

  if (!m_Loaded)
    return false;

  if (m_MusicType == MUSIC_MIDI)
  {
    #ifdef DMUSIC_LUV
      m_Music.m_Midi->Stop(NULL, NULL, 0, 0);
    #else
      mciSendString("pause midi", NULL, NULL, NULL);
    #endif
  }
  else if (m_MusicType == MUSIC_MP3 || m_MusicType == MUSIC_WAV)
  {
    FSOUND_Stream_SetPaused(m_Music.m_Stream, true);
  }
  else if (m_MusicType == MUSIC_MOD)
  {
    FMUSIC_StopSong(m_Music.m_Module);
    m_Channel = -1;
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

unsigned int
CModule::GetVolume()
{
  if (!s_AudioReady)
    return 0;

  if (m_MusicType == MUSIC_MIDI)
  {
/*
    bool MidiIsPlaying = false;
    HMIDIOUT midimix;
    //LPDWORD dwStereoVolume;
    unsigned long dwStereoVolume;
    unsigned int Volume;

    //dwStereoVolume = (LPDWORD)malloc(sizeof(LPDWORD));

    // We pause the music if we're playing
    if (IsPlaying())
    { MidiIsPlaying = true; Stop(); }

    midiOutOpen(&midimix, MIDI_MAPPER, NULL, NULL, CALLBACK_NULL);
    midiOutGetVolume(midimix, (LPDWORD)dwStereoVolume);
    midiOutClose(midimix);

    // We unpause it afterwards.
    if (MidiIsPlaying)
    Play();

    Volume = (unsigned int)(100 * ((float)LOWORD(dwStereoVolume)/(float)0xffff));

    //free(dwStereoVolume);
    return Volume;
*/
    // the above is still not working
    // so I use this
    //SetVolume(65535/2);
    return (65535 / 2);
  }
  else
  {
    //return (65535 / 2);
    return 65535;
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CModule::SetVolume(unsigned int Volume)
{
  if (!s_AudioReady)
    return;

  if (m_MusicType == MUSIC_MIDI)
  {
/*
    bool MidiIsPlaying = false;
    HMIDIOUT midimix;
    DWORD dwStereoVolume;

    m_iSongVolume = Volume;

    dwStereoVolume = MAKELPARAM(Volume, Volume);

    // We pause the music if we're playing
    if (IsPlaying())
    { MidiIsPlaying = true; Stop(); }

    midiOutOpen(&midimix, MIDI_MAPPER, NULL, NULL, CALLBACK_NULL);
    midiOutSetVolume(midimix, dwStereoVolume);
    midiOutClose(midimix);

    // We unpause it afterwards.
    if (MidiIsPlaying)
      Play();
*/
  }
  else if (m_MusicType == MUSIC_MP3)
  {
    m_SongVolume = Volume / 257;
    FSOUND_SetVolume(m_Channel, m_SongVolume);
    
  }
  else if (m_MusicType == MUSIC_MOD)
  {
    m_SongVolume = Volume / 256;
    FMUSIC_SetMasterVolume(m_Music.m_Module, m_SongVolume);
  }
}

////////////////////////////////////////////////////////////////////////////////

unsigned int 
CModule::GetSongLength()
{
  if (!s_AudioReady)
    return 0;

  if (m_MusicType == MUSIC_MIDI)
  {
    #ifdef DMUSIC_LUV
      MUSIC_TIME* theTime;
      theTime = new MUSIC_TIME;
      m_MidiSegment->GetLength(theTime);
      return *theTime;
    #else
      char buf[80];
      mciSendString("status midi length", buf, 80, NULL);
      return atoi(buf);
    #endif
  }
  else if (m_MusicType == MUSIC_MP3 || m_MusicType == MUSIC_WAV)
  {
    return (FSOUND_Stream_GetLength(m_Music.m_Stream));
  }
  else // m_MusicType == MUSIC_MOD
  {
    return (FMUSIC_GetNumOrders(m_Music.m_Module));
  }
}

////////////////////////////////////////////////////////////////////////////////
unsigned int 
CModule::GetSongPos()
{
  if (!s_AudioReady)
    return 0;

  if (m_MusicType == MUSIC_MIDI)
  {
    #ifdef DMUSIC_LUV
      MUSIC_TIME *theTime;
      theTime = new MUSIC_TIME;
      m_MidiSegmentState->GetSeek(theTime);
      //m_Music.m_Midi->GetTime(NULL, theTime);
      return *theTime;
    #else
      char buf[80];
      mciSendString("status midi position", buf, 80, NULL);
      return atoi(buf);
    #endif
  }
  else if (m_MusicType == MUSIC_MP3 || m_MusicType == MUSIC_WAV)
  {
    return (FSOUND_Stream_GetPosition(m_Music.m_Stream));
  }
  else // m_MusicType == MUSIC_MOD
  {
    return (FMUSIC_GetOrder(m_Music.m_Module));
  }
}

////////////////////////////////////////////////////////////////////////////////

void 
CModule::SetSongPos(unsigned int Pos)
{
  if (!s_AudioReady)
    return;

  if (m_MusicType == MUSIC_MP3 || m_MusicType == MUSIC_WAV)
    FSOUND_Stream_SetPosition(m_Music.m_Stream, Pos);

  if (m_MusicType == MUSIC_MOD)
    FMUSIC_SetOrder(m_Music.m_Module, Pos);

  if (m_MusicType == MUSIC_MIDI)
  {
    #ifndef DMUSIC_LUV
      char buf[80];
      sprintf(buf, "seek midi to %i", Pos);
      mciSendString(buf, NULL, NULL, NULL);
      mciSendString("play midi", NULL, NULL, NULL);
    #endif
  }

}

////////////////////////////////////////////////////////////////////////////////

bool
CModule::IsPlaying() const
{
  if (!s_AudioReady)
    return false;

  if (m_MusicType == MUSIC_MIDI)
  {
    // I know there's a bunch of other modes, but I only really care about
    // if it was playing.
    #ifdef DMUSIC_LUV
      if (m_Music.m_Midi->IsPlaying(m_MidiSegment, m_MidiSegmentState) == S_OK)
        return true;
    #else
      char buf[7];
      mciSendString("status midi mode", buf, 8, NULL);
      if (!strcmp(buf, "playing"))
        return true;
    #endif

      else
        return false;
  }
  else if (m_MusicType == MUSIC_MP3 || m_MusicType == MUSIC_WAV)
  {
    signed char scState;

    scState = FSOUND_Stream_GetPaused(m_Music.m_Stream);
    if (scState == TRUE)
      return false;
    if (scState == FALSE)
      return true;
  }
  else if (m_MusicType == MUSIC_MOD)
  {
    signed char scState;

    scState = FMUSIC_IsPlaying(m_Music.m_Module);
    if (scState == FALSE)
      return false;
    if (scState == TRUE)
      return true;
  }
  
  // This is here to stop that compiler warning. O_o
  return false; 
}

////////////////////////////////////////////////////////////////////////////////

char*
CModule::ModuleGetSongName()
{
  if (!s_AudioReady)
    return "Unknown";

  return (FMUSIC_GetName(m_Music.m_Module));
}

////////////////////////////////////////////////////////////////////////////////

CSample::CSample()
: m_Loaded(false)
, m_Channel(0)
, m_SampleRate(0)
, m_Sample(NULL)
, m_SampleVolume(127)
{
  InitializeAudio();
}

////////////////////////////////////////////////////////////////////////////////

CSample::~CSample()
{
  if (!s_AudioReady)
    return;

  FSOUND_Sample_Free(m_Sample);

  ShutdownAudio();
}

////////////////////////////////////////////////////////////////////////////////

bool
CSample::Load(const char* filename)
{
  if (!s_AudioReady)
    return true;
  
  long x;
  x = strlen(filename);

  if (stricmp(filename + x - 4, ".wav") == 0)
    m_SoundType = SOUND_WAV;
  else if (stricmp(filename + x - 4, ".raw") == 0)
    m_SoundType = SOUND_RAW;

  if (m_SoundType == SOUND_WAV)
  {
    m_Sample = FSOUND_Sample_LoadWav(FSOUND_FREE, (char *)filename, FSOUND_BaseMode | FSOUND_LOOP_OFF);
    if (m_Sample == NULL)
      return false;
  }
  else if (m_SoundType == SOUND_RAW)
  {
    FSOUND_Sample_LoadRaw(FSOUND_FREE, (char *)filename, FSOUND_BaseMode | FSOUND_LOOP_OFF);
    if (m_Sample == NULL)
      return false;
  }

  m_Loaded = true;
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CSample::Play()
{
  if (!s_AudioReady)
    return false;

  if (!m_Loaded)
    return false;

  m_Channel = FSOUND_PlaySound(FSOUND_FREE, m_Sample);
  if (m_Channel == -1)
  {
    return false;
  }
  SetVolume(m_SampleVolume);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CSample::Stop()
{
  if (!s_AudioReady)
    return false;

  if (!m_Loaded)
    return false;

  FSOUND_StopSound(m_Channel);
  m_Channel = -1;

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CSample::IsPlaying() const
{
  if (!s_AudioReady)
    return false;
  
  if (m_Sample)
  {
    signed char scStatus = FSOUND_IsPlaying(m_Channel);
    if (scStatus == TRUE)
      return true;
    else if (scStatus == FALSE)
      return false;
  }

  return false;
}

////////////////////////////////////////////////////////////////////////////////

void
CSample::SetVolume(unsigned int Volume)
{
  if (!s_AudioReady)
    return;

  m_SampleVolume = Volume;
  FSOUND_SetVolume(m_Channel, Volume);
}

////////////////////////////////////////////////////////////////////////////////
