#include <string.h>
#include "Spriteset.hpp"
#include "packed.h"
#include "x++.hpp"



////////////////////////////////////////////////////////////////////////////////

sSpriteset::sSpriteset()
: m_FrameWidth(0)
, m_FrameHeight(0)

, m_NumDirections(0)
, m_Directions(NULL)

, m_BaseX1(0)
, m_BaseY1(0)
, m_BaseX2(0)
, m_BaseY2(0)
{
}

////////////////////////////////////////////////////////////////////////////////

sSpriteset::~sSpriteset()
{
  Destroy();
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::Create(int num_directions, int num_frames)
{
  m_FrameWidth = 16;
  m_FrameHeight = 32;

  m_NumDirections = num_directions;
  m_Directions = new Direction[num_directions];
  for (int i = 0; i < num_directions; i++)
  {
    m_Directions[i].num_frames = num_frames;
    m_Directions[i].frames = new sSprite[num_frames];

    for (int j = 0; j < num_frames; j++)
    {
      m_Directions[i].frames[j].Resize(m_FrameWidth, m_FrameHeight);
      memset(m_Directions[i].frames[j].GetPixels(), 0, 16 * 32 * sizeof(RGBA));
      m_Directions[i].frames[j].SetDelay(8);
    }

  }
}

////////////////////////////////////////////////////////////////////////////////

PACKED_STRUCT(SPRITESET_HEADER)
  byte signature[4];
  word version;
  word num_frames;
  word frame_width;
  word frame_height;
  word num_directions;
  word base_x1;
  word base_y1;
  word base_x2;
  word base_y2;
  byte reserved[106];
END_STRUCT(SPRITESET_HEADER)

PACKED_STRUCT(DIRECTION_HEADER)
  word num_frames;
  byte reserved[62];
END_STRUCT(DIRECTION_HEADER)

PACKED_STRUCT(FRAME_HEADER)
  word width;
  word height;
  word delay;
  byte reserved[26];
END_STRUCT(FRAME_HEADER)

ASSERT_STRUCT_SIZE(SPRITESET_HEADER, 128)
ASSERT_STRUCT_SIZE(DIRECTION_HEADER, 64)
ASSERT_STRUCT_SIZE(FRAME_HEADER,     32)

////////////////////////////////////////////////////////////////////////////////

bool
sSpriteset::Load(const char* filename)
{
  // open file
  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  // read the header
  SPRITESET_HEADER header;
  fread(&header, 1, sizeof(header), file);

  // validate header
  if (memcmp(header.signature, ".rss", 4) != 0 ||
      (header.version != 1 && header.version != 2))
  {
    fclose(file);
    return false;
  }

  Destroy();

  m_FrameWidth  = header.frame_width;
  m_FrameHeight = header.frame_height;

  m_BaseX1 = header.base_x1;
  m_BaseY1 = header.base_y1;
  m_BaseX2 = header.base_x2;
  m_BaseY2 = header.base_y2;

  if (header.version == 1)  // VERSION ONE
  {
    m_NumDirections = 8;
    m_Directions = new Direction[8];
    for (int i = 0; i < 8; i++)
    {
      m_Directions[i].num_frames = 8;
      m_Directions[i].frames = new sSprite[8];
      for (int j = 0; j < 8; j++)
      {
        m_Directions[i].frames[j].Resize(header.frame_width, header.frame_height);
        m_Directions[i].frames[j].SetDelay(8);
        fread(m_Directions[i].frames[j].GetPixels(), sizeof(RGBA), header.frame_width * header.frame_height, file);
      }
    }
  }
  else                      // VERSION TWO
  {
    m_NumDirections = header.num_directions;
    m_Directions = new Direction[header.num_directions];

    for (int i = 0; i < m_NumDirections; i++)
    {
      // read the direction header
      DIRECTION_HEADER direction_header;
      fread(&direction_header, 1, sizeof(direction_header), file);

      m_Directions[i].num_frames = direction_header.num_frames;
      m_Directions[i].frames = new sSprite[direction_header.num_frames];

      for (int j = 0; j < direction_header.num_frames; j++)
      {
        // read the frame header
        FRAME_HEADER frame_header;
        fread(&frame_header, 1, sizeof(frame_header), file);

        // some backwards compatibility hacking
        if (m_FrameWidth == 0 || m_FrameHeight == 0)
        {
          m_FrameWidth  = frame_header.width;
          m_FrameHeight = frame_header.height;
        }

        m_Directions[i].frames[j].SetDelay(frame_header.delay);
        m_Directions[i].frames[j].Resize(m_FrameWidth, m_FrameHeight);
        fread(m_Directions[i].frames[j].GetPixels(), sizeof(RGBA), m_FrameWidth * m_FrameHeight, file);
      }
    }
  }

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sSpriteset::Save(const char* filename) const
{
  // open file
  FILE* file = fopen(filename, "wb");
  if (file == NULL)
    return false;

  // fill header
  SPRITESET_HEADER header;
  memset(&header, 0, sizeof(header));
  memcpy(header.signature, ".rss", 4);
  header.version        = 2;
  header.frame_width    = m_FrameWidth;
  header.frame_height   = m_FrameHeight;
  header.num_directions = m_NumDirections;
  header.base_x1        = m_BaseX1;
  header.base_y1        = m_BaseY1;
  header.base_x2        = m_BaseX2;
  header.base_y2        = m_BaseY2;

  // write header
  fwrite(&header, 1, sizeof(header), file);

  // write all of the directions
  for (int i = 0; i < m_NumDirections; i++)
  {
    // fill direction header
    DIRECTION_HEADER direction_header;
    memset(&direction_header, 0, sizeof(direction_header));
    direction_header.num_frames = m_Directions[i].num_frames;

    // write it
    fwrite(&direction_header, 1, sizeof(direction_header), file);

    // write all of the frames
    for (int j = 0; j < m_Directions[i].num_frames; j++)
    {
      // fill frame header
      FRAME_HEADER frame_header;
      memset(&frame_header, 0, sizeof(frame_header));
      frame_header.delay  = m_Directions[i].frames[j].GetDelay();

      // write it
      fwrite(&frame_header, 1, sizeof(frame_header), file);

      // write the frame
      fwrite(m_Directions[i].frames[j].GetPixels(), sizeof(RGBA), m_FrameWidth * m_FrameHeight, file);
    }
  }

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sSpriteset::Import_BMP(const char* filename, int frame_width, int frame_height, RGBA transparent)
{
  CImage32 image;
  if (!image.Load(filename))
    return false;

  if (image.GetWidth() % frame_width || image.GetHeight() % frame_height)
    return false;

  // replace for the transparent color
  for (int y = 0; y < image.GetHeight(); y++)
    for (int x = 0; x < image.GetWidth(); x++)
      if (image.GetPixels()[y * image.GetWidth() + x].red   == transparent.red &&
          image.GetPixels()[y * image.GetWidth() + x].green == transparent.green &&
          image.GetPixels()[y * image.GetWidth() + x].blue  == transparent.blue)
      {
        image.GetPixels()[y * image.GetWidth() + x].alpha = 0;
      }



  m_FrameWidth = frame_width;
  m_FrameHeight = frame_height;
  int numRows = image.GetHeight() / frame_height;
  int numFrames = image.GetWidth() / frame_width;

  m_NumDirections = numRows;
  m_Directions = new Direction[numRows];
  for (int i = 0; i < numRows; i++)
  {
    m_Directions[i].num_frames = numFrames;
    m_Directions[i].frames = new sSprite[numFrames];

    for (int j = 0; j < numFrames; j++)
    {
      m_Directions[i].frames[j].Resize(m_FrameWidth, m_FrameHeight);
      memset(m_Directions[i].frames[j].GetPixels(), 0, m_FrameWidth * m_FrameHeight * sizeof(RGBA));
      m_Directions[i].frames[j].SetDelay(8);
    }
  }

  // grabs the data in a gigantic loops 'o doom...
  for (int j=0; j<numRows; j++)
    for (int i=0; i<numFrames; i++)
      for (int k=0; k<m_FrameHeight; k++)
        memcpy(m_Directions[j].frames[i].GetPixels() + (k*m_FrameWidth),
               image.GetPixels() + (j*m_FrameHeight*image.GetWidth()) + (i*m_FrameWidth) + (k*image.GetWidth()),
               m_FrameWidth*sizeof(RGBA));

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sSpriteset::Import_CHR(const char* filename, const char* palette_file)
{
  // read the palette
  RGB palette[256];
  FILE* file = fopen(palette_file, "rb");
  if (file == NULL)
    return false;
  fread(palette, 256, sizeof(RGB), file);
  fclose(file);

  // determine which type of .chr it is
  file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  // get the version of the .chr
  byte version;
  version = (byte)fgetc(file);
  fseek(file, 0, SEEK_SET);

  // import the .chr
  bool success;
  if (version == 0)
    success = Import_CHR1(file, palette);
  else
    success = Import_CHR2(file, palette);

  fclose(file);
  return success;
}

////////////////////////////////////////////////////////////////////////////////

int
sSpriteset::GetFrameWidth() const
{
  return m_FrameWidth;
}

////////////////////////////////////////////////////////////////////////////////

int
sSpriteset::GetFrameHeight() const
{
  return m_FrameHeight;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::ResizeFrames(int width, int height)
{
  for (int i = 0; i < m_NumDirections; i++)
    for (int j = 0; j < m_Directions[i].num_frames; j++)
      m_Directions[i].frames[j].Resize(width, height);

  m_FrameWidth  = width;
  m_FrameHeight = height;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::InsertDirection(int direction)
{
  Direction* new_directions = new Direction[m_NumDirections + 1];

  int width = m_Directions[0].frames[0].GetWidth();
  int height = m_Directions[0].frames[0].GetHeight();

  for (int i = 0; i < direction; i++)
    new_directions[i] = m_Directions[i];
  for (int i = direction; i < m_NumDirections; i++)
    new_directions[i + 1] = m_Directions[i];

  new_directions[direction].num_frames = 1;
  new_directions[direction].frames = new sSprite[1];
  new_directions[direction].frames[0].Resize(width, height);
  new_directions[direction].frames[0].SetDelay(8);

  m_NumDirections++;
  delete[] m_Directions;
  m_Directions = new_directions;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::DeleteDirection(int direction)
{
  Direction* new_directions = new Direction[m_NumDirections - 1];

  delete[] m_Directions[direction].frames;

  for (int i = 0; i < direction; i++)
    new_directions[i] = m_Directions[i];
  for (int i = direction; i < m_NumDirections - 1; i++)
    new_directions[i] = m_Directions[i + 1];
  
  m_NumDirections--;
  delete[] m_Directions;
  m_Directions = new_directions;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::AppendDirection()
{
  Direction* new_directions = new Direction[m_NumDirections + 1];

  for (int i = 0; i < m_NumDirections; i++)
    new_directions[i] = m_Directions[i];

  int width = m_Directions[0].frames[0].GetWidth();
  int height = m_Directions[0].frames[0].GetHeight();

  new_directions[m_NumDirections].num_frames = 1;
  new_directions[m_NumDirections].frames = new sSprite[1];
  new_directions[m_NumDirections].frames[0].Resize(width, height);
  new_directions[m_NumDirections].frames[0].SetDelay(8);

  m_NumDirections++;
  delete[] m_Directions;
  m_Directions = new_directions;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::InsertFrame(int direction, int frame)
{
  Direction* d = m_Directions + direction;

  sSprite* new_frames = new sSprite[d->num_frames + 1];
  for (int i = 0; i < frame; i++)
    new_frames[i] = d->frames[i];
  for (int i = frame + 1; i < d->num_frames + 1; i++)
    new_frames[i] = d->frames[i - 1];

  int new_width  = d->frames[frame].GetWidth();
  int new_height = d->frames[frame].GetHeight();
  new_frames[frame].Resize(new_width, new_height);
  new_frames[frame].SetDelay(8);
  
  delete[] d->frames;
  d->frames = new_frames;
  d->num_frames++;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::DeleteFrame(int direction, int frame)
{
  Direction* d = m_Directions + direction;

  sSprite* new_frames = new sSprite[d->num_frames - 1];
  for (int i = 0; i < frame; i++)
    new_frames[i] = d->frames[i];
  for (int i = frame; i < d->num_frames - 1; i++)
    new_frames[i] = d->frames[i + 1];
  
  delete[] d->frames;
  d->frames = new_frames;
  d->num_frames--;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::AppendFrame(int direction)
{
  Direction* d = m_Directions + direction;

  sSprite* new_frames = new sSprite[d->num_frames + 1];
  for (int i = 0; i < d->num_frames; i++)
    new_frames[i] = d->frames[i];

  new_frames[d->num_frames].Resize(d->frames[0].GetWidth(), d->frames[0].GetHeight());
  new_frames[d->num_frames].SetDelay(8);

  delete[] d->frames;
  d->frames = new_frames;
  d->num_frames++;
}

////////////////////////////////////////////////////////////////////////////////

int
sSpriteset::GetNumDirections() const
{
  return m_NumDirections;
}

////////////////////////////////////////////////////////////////////////////////

int
sSpriteset::GetNumFrames(int direction) const
{
  return m_Directions[direction].num_frames;
}

////////////////////////////////////////////////////////////////////////////////

sSprite&
sSpriteset::GetFrame(int direction, int frame)
{
  return m_Directions[direction].frames[frame];
}

////////////////////////////////////////////////////////////////////////////////

const sSprite&
sSpriteset::GetFrame(int direction, int frame) const
{
  return m_Directions[direction].frames[frame];
}

////////////////////////////////////////////////////////////////////////////////

int
sSpriteset::GetBaseX1() const
{
  return m_BaseX1;
}

////////////////////////////////////////////////////////////////////////////////

int
sSpriteset::GetBaseY1() const
{
  return m_BaseY1;
}

////////////////////////////////////////////////////////////////////////////////

int
sSpriteset::GetBaseX2() const
{
  return m_BaseX2;
}

////////////////////////////////////////////////////////////////////////////////

int
sSpriteset::GetBaseY2() const
{
  return m_BaseY2;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::SetBaseX1(int x)
{
  m_BaseX1 = x;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::SetBaseY1(int x)
{
  m_BaseY1 = x;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::SetBaseX2(int x)
{
  m_BaseX2 = x;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::SetBaseY2(int x)
{
  m_BaseY2 = x;
}

////////////////////////////////////////////////////////////////////////////////

void
sSpriteset::Destroy()
{
  for (int i = 0; i < m_NumDirections; i++)
    delete[] m_Directions[i].frames;
  delete[] m_Directions;
  m_NumDirections = 0;
  m_Directions = NULL;
}

////////////////////////////////////////////////////////////////////////////////

bool
sSpriteset::Import_CHR1(FILE* file, RGB palette[256])
{
//  Create(9, 8);
/*
  // create the frames...
  m_FrameWidth = 16;
  m_FrameHeight = 32;

  m_NumDirections = 9;
  m_Directions = new Direction[9];
  for (int i = 0; i < 9; i++)
  {
    m_Directions[i].num_frames = (i == 8 ? 10 : 8);
    m_Directions[i].frames = new sSprite[(i == 8 ? 10 : 8)];
    int k = (i == 8 ? 2 : 0);

    for (int j = 0; j < (8 + k); j++)
    {
      m_Directions[i].frames[j].Resize(m_FrameWidth, m_FrameHeight);
      memset(m_Directions[i].frames[j].GetPixels(), 0, 16 * 32 * sizeof(RGBA));
      m_Directions[i].frames[j].SetDelay(8);
    }
*/
  // allocate .chr frames and read them
  RGBA chr_frames[30][16 * 32];

  for (int i = 0; i < 30; i++)
  {
    for (int y = 0; y < 32; y++)
      for (int x = 0; x < 16; x++)
      {
        byte c = (byte)fgetc(file);

        RGBA* p = chr_frames[i] + (y * 16 + x);
        p->red   = palette[c].red   * 4;
        p->green = palette[c].green * 4;
        p->blue  = palette[c].blue  * 4;
        p->alpha = (byte)(c == 0 ? 0 : 255);
      }
  }

#define CopyFrame(direction, row, frame, column)            \
{                                                           \
  memcpy(m_Directions[direction].frames[frame].GetPixels(), \
         chr_frames[row * 5 + column],                      \
         16 * 32 * sizeof(RGBA));                           \
}

#define CopyRow(direction, row)    \
{                                  \
  CopyFrame(direction, row, 0, 0); \
  CopyFrame(direction, row, 1, 1); \
  CopyFrame(direction, row, 2, 2); \
  CopyFrame(direction, row, 3, 1); \
  CopyFrame(direction, row, 4, 0); \
  CopyFrame(direction, row, 5, 3); \
  CopyFrame(direction, row, 6, 4); \
  CopyFrame(direction, row, 7, 3); \
}

  // copy frames to spriteset
  CopyRow(0, 1);
  CopyRow(1, 2);
  CopyRow(2, 2);
  CopyRow(3, 2);
  CopyRow(4, 0);
  CopyRow(5, 3);
  CopyRow(6, 3);
  CopyRow(7, 3);
  for (int i=0; i<10; i++)
    CopyFrame(8, 4, i, i);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

PACKED_STRUCT(CHR2_HEADER)

  byte version;
  word width;
  word height;
  word hotspot_left;
  word hotspot_top;
  word hotspot_width;
  word hotspot_height;
  word num_frames;

END_STRUCT(CHR2_HEADER)

////////////////////////////////////////////////////////////////////////////////

bool
sSpriteset::Import_CHR2(FILE* file, RGB palette[256])
{
  CHR2_HEADER header;
  fread(&header, sizeof(header), 1, file);

   // because we want to get the extra frames in the set, so I'm building
  // a table, and then parsing it. This is so that I'm marking down the 
  // frames that's used, and the unused shall be in a seperate direction.
  // Finally we then do something about it.
  bool* frameUsed;
  bool  extraFramesUsed;
  RGBA* frames;
  long  dataLength;
  dword walkcodeLength[4];
  char* walkcode[4];
  int*  walkcodeValue[4];
  int* walkcodeType[4];
  int   walkcodeNumArg[4];
  int frameCount[5];
  int i,j,k,l;
  
  if (header.version != 2)
  return false;

  //grab all the data then expand it.
  frameCount[0] = frameCount[1] = frameCount[2] = frameCount[3] = 1;
  frameCount[4] = 0;

  frameUsed = new bool[header.num_frames];
  memset(frameUsed, 0, header.num_frames);
  frames = new RGBA[header.num_frames * header.width * header.height];
  j=0;
    
  fread(&dataLength, 1, 4, file);
  for (i=0; i<dataLength; i++)
  {
    byte value = getc(file);
    
    if (value<255)
    {
      frames[j].red = palette[value].red * 4;
      frames[j].blue = palette[value].blue * 4;
      frames[j].green = palette[value].green * 4;
      frames[j].alpha = (byte)(value == 0 ? 0 : 255);
      j++;
    }
    else if (value == 255)
    {
      byte runlength = getc(file);
      value = getc(file);
      for(k=0; k<runlength; k++)
      {
        frames[j].red = palette[value].red * 4;
        frames[j].blue = palette[value].blue * 4;
        frames[j].green = palette[value].green * 4;
        frames[j].alpha = (byte)(value == 0 ? 0 : 255);
        j++;
      }
      i+=2;
    }
  }

  for (i=0; i<4; i++)
  {
    walkcodeType[i] = new int[1];
    walkcodeValue[i] = new int[1];
    walkcodeType[i][0] = true;
	  fread(&walkcodeValue[i][0], 1, sizeof(long), file);
  }

  // read the movement script
  for (i=0; i<4; i++)
  {
    fread(&walkcodeLength[i], 1, sizeof(dword), file);
    walkcode[i] = new char[walkcodeLength[i]];
    fread(walkcode[i], 1, walkcodeLength[i], file);
    //printf("%s\n", walkcode[i]);
  }

  // translate the movement script
  for (i=0; i<4; i++)
  {
    k=1;
    for (j=0; j<(int)walkcodeLength[i];)
      if (walkcode[i][j] == 'F' || walkcode[i][j] == 'W')
      {
        char command;
        char token[10];
        int* tempwalkcodeType;
        int* tempwalkcodeValue;
        command = walkcode[i][j];
        l=0;
        j++;

        while (j < (int)walkcodeLength[i] && walkcode[i][j] == ' ') j++;
        while (j < (int)walkcodeLength[i] && walkcode[i][j] >= 48 &&
               walkcode[i][j] <= 57)
        {
          token[l] = walkcode[i][j];
          l++;
          j++;
        }
        
        token[l] = 0;
        //printf("%i\n", atoi(token));

        tempwalkcodeType = new int[k + 1];
        tempwalkcodeValue = new int[k + 1];
        memcpy(tempwalkcodeType, walkcodeType[i], k*sizeof(int));
        memcpy(tempwalkcodeValue, walkcodeValue[i], k*sizeof(int));
        tempwalkcodeType[k] = (command == 'F' ? 1 : 0);
        tempwalkcodeValue[k] = atoi(token);
        delete[] walkcodeType[i]; delete[] walkcodeValue[i];
        walkcodeType[i] = tempwalkcodeType;
        walkcodeValue[i] = tempwalkcodeValue;

        if (command == 'F')
        {
        frameUsed[atoi(token)] = true;
        frameCount[i]++;
        }
        k++;
      }
      else
        j++;

    walkcodeNumArg[i] = k;
  }

  // check for extra frames
  for (i=0; i<header.num_frames; i++)
    if (!frameUsed[i])
      frameCount[4]++;
  if (frameCount[4] > 0)
    extraFramesUsed = true;
  else
    extraFramesUsed = false;


  // okay, now build the data the painful way (Create() can't handle this)
  // and transfer ALL the data.
  m_FrameWidth = header.width;
  m_FrameHeight = header.height;

  m_NumDirections = 8 + extraFramesUsed;
  m_Directions = new Direction[8 + extraFramesUsed];
  for (int i = 0; i < 8 + extraFramesUsed; i++)
  {
    switch(i)
    {
      case EAST: 
      case SOUTHEAST: 
      case NORTHEAST:
        l = 0;
        break;

      case WEST:
      case NORTHWEST:
      case SOUTHWEST:
        l = 1;
        break;

      case NORTH: l = 2; break;
      case SOUTH: l = 3; break;
      case 8: l = 4; break;
    }

    m_Directions[i].num_frames = frameCount[l];
    m_Directions[i].frames = new sSprite[frameCount[l]];

    for (int j = 0; j < frameCount[l]; j++)
    {
      m_Directions[i].frames[j].Resize(m_FrameWidth, m_FrameHeight);
      memset(m_Directions[i].frames[j].GetPixels(), 0, m_FrameWidth * m_FrameHeight * sizeof(RGBA));
      m_Directions[i].frames[j].SetDelay(0);
    }

  }

  // left right up down special
#define CopySide(direction, theside)                                    \
{                                                                       \
  k=-1;                                                                 \
  for (i=0; i<walkcodeNumArg[theside]; i++)                             \
    if (walkcodeType[theside][i] == 1)                                  \
    {                                                                   \
      /*printf("F%i ", walkcodeValue[theside][i]);*/                    \
      k++;                                                              \
      if (k < frameCount[theside])                                      \
      memcpy(m_Directions[direction].frames[k].GetPixels(),             \
             frames + (walkcodeValue[theside][i]*header.width*header.height),\
             header.width * header.height * 4);                         \
      /*m_Directions[direction].frames[k].SetDelay(10);  */             \
    }                                                                   \
    else                                                                \
    {                                                                   \
      /*printf("W%i ", walkcodeValue[theside][i]);*/                    \
      m_Directions[direction].frames[k].SetDelay(m_Directions[direction].frames[k].GetDelay() + walkcodeValue[theside][i]);\
    }                                                                   \
    /*printf("\n");*/                                                       \
}

  CopySide(EAST, 1);
  CopySide(NORTHEAST, 1);
  CopySide(SOUTHEAST, 1);
  CopySide(WEST, 0);
  CopySide(NORTHWEST, 0);
  CopySide(SOUTHWEST, 0);
  CopySide(NORTH, 2);
  CopySide(SOUTH, 3);

  j = 0;
  if (extraFramesUsed)
    {
    for (i=0; i<header.num_frames; i++)
      if (!frameUsed[i])
      {
        memcpy(m_Directions[8].frames[j].GetPixels(),
               frames + (i * header.width * header.height),
               header.width * header.height * 4);
        j++;
      }
    }

  delete[] frameUsed;
  delete[] frames;
  for (i=0; i<4; i++)
    {
    delete[] walkcode[i];
    delete[] walkcodeValue[i];
    delete[] walkcodeType[i];
    }

  return true;

}


////////////////////////////////////////////////////////////////////////////////
