#ifndef SS_COMPILER_HPP
#define SS_COMPILER_HPP


#include "SphereScript.hpp"
#include "SS_Tables.hpp"
#include "SS_Internal.hpp"
#include "Buffer.hpp"


class sCompiler
{
public:     // METHODS
  sCompiler();
  ~sCompiler();
 
  void Execute(CBuffer* in, CBuffer* out);

private:    // TYPES
  struct sArrayType
  {
    int    initial_size;
    ssType sub_type;
  };

  struct sStringLiteral
  {
    sString label;
    sString value;
  };

private:    // METHODS
  void InitializeSystemTypes();
  void InitializeSystemFunctions();

  void Tokenize();

  // compile
  void FirstPass();
  void SecondPass();

  // verification functions
  static bool IsAlpha(char c);
  static bool IsNumber(char c);
  static bool IsValidIdentifier(sString s);
  static bool IsKeyword(sString s);
  static bool IsNativeType(sString s);
         bool IsUserType(sString s);
         bool IsDataType(sString s);
         bool IsReturnType(sString s);
         bool IsGlobalIdentifier(sString s);
         bool IsTakenIdentifier(sString s);
         bool IsFunction(sString s);
         bool IsSystemFunction(sString s);
         bool IsExternalFunction(sString s);
         bool IsLValue(sString s);
         bool IsGlobalVariable(sString s);
         bool IsParameter(sString s);
         bool IsLocalVariable(sString s);
  static bool IsAssignmentOperator(sString s);
  static bool IsLogicOperator(sString s);
  static bool IsComparisonOperator(sString s);
  static bool IsMultiplicationOperator(sString s);
  static bool IsAdditionOperator(sString s);
  static bool IsBooleanConstant(sString s);
  static bool IsNumericConstant(sString s);
  static bool IsPositiveInteger(sString s);
  
  // general elements
  sString   cDataType();
  sString   cReturnType();
  sString   cIdentifier();
  sVariable cParameter();
  sString   cMangledVariable(ssType& t);
  int       cPositiveInteger();

  // script elements
  void cProgram();
  void cUseDirective();
  void cImplementDirective();
  void cGlobal();
  void cUserType();
  void cGlobalVariable();
  void cFunction();
  void cBlock();
  void cStatement();
  void cLocalVariable();
  void cReturn();
  void cIf();
  void cFor();
  void cWhile();
  void cDo();
  void cSwitch();

  // expression elements
  ssType cExpression();
  ssType cAssignment();
  ssType cLogicExpression();
  void   cLogic(ssType t);
  ssType cComparisonExpression();
  void   cCompare(ssType t);
  ssType cAdditionExpression();
  void   cAdd(ssType t);
  ssType cMultiplicationExpression();
  void   cMultiply(ssType t);
  ssType cTerm();
  ssType cStringLiteral();
  ssType cBooleanConstant();
  ssType cNumericConstant();

  // general

  void Error(sString message);

  void Emit(sString s = "");
  void EmitGlobalVariableDefinitions();
  void EmitInitializeFunction();
  void EmitUninitializeFunction();
  void EmitStringLiterals();

  void    Next();
  void    Prev();
  sString Token();
  sString TokenInc();
  void    Match(sString s);
  void    MatchTypes(ssType t1, ssType t2);

  ssType StringToType(sString s);
  int    SizeOf(ssType type);
  ssType TypeOf(sString s);

  sFunction& GetFunction(sString name);

  sString NewLabel();
  void    PostLabel(sString label);

  void BeginScopeLevel();
  void EndScopeLevel();

  void ConstructVariable(sString name, ssType type);
  void DestructVariable(sString name, ssType type);
  void MoveVariable(ssType type, sString dest, sString src);

  void AddDependancy(sString name);
  void EmitDependancies();

  static bool IsUserType(ssType t);
  int         OffsetOfField(ssType t, sString field);
  ssType      TypeOfField(ssType t, sString field);

  static bool IsArrayType(ssType t);
  ssType      AddArrayType(sArrayType a);

  static sString itos(int i);

private:    // DATA
  CBuffer* inbuffer;
  CBuffer* outbuffer;

  enum { NO_PASS, FIRST_PASS, SECOND_PASS } Pass;

  sVector<sToken> Tokens;
  int             CurrentToken;
  sString         CurrentFunction;

  sVector<sFunction>           SystemFunctions;
  sVector<sUserType>           UserTypes;
  sVector<sArrayType>          ArrayTypes;
  sVector<sVariable>           GlobalVariables;
  sVector<sFunction>           Functions;
  sVector<sVariable>           Parameters;
  sVector<sVector<sVariable> > LocalVariableLevels;
  sVector<sStringLiteral>      StringLiterals;
  sVector<sFunction>           RequiredFunctions;
  sVector<sFunction>           ExternalFunctions;

  sVector<sString> Dependancies;

  int LabelIndex;  // used to generate the temporary labels
};


#endif
