#include <stdio.h>
#include <stdlib.h>
#include "Map.hpp"
#include "Layer.hpp"
#include "Entities.hpp"
#include "Tileset.hpp"
#include "Image32.hpp"
#include "x++.hpp"
#include "packed.h"
#include "types.h"



////////////////////////////////////////////////////////////////////////////////

sMap::sMap()
: m_StartX(0)
, m_StartY(0)
, m_StartLayer(0)
, m_StartDirection(0)

, m_MusicFile(newstr(""))
, m_ScriptFile(newstr(""))

, m_NumLayers(0)
, m_Layers(NULL)

, m_NumEntities(0)
, m_Entities(NULL)
{
}

////////////////////////////////////////////////////////////////////////////////

sMap::sMap(int width, int height, int layers)
{
  m_NumLayers = layers;
  m_Layers = new sLayer[m_NumLayers];
  for (int i = 0; i < layers; i++)
    m_Layers[i].SetSize(width, height);

  m_MusicFile = newstr("");
  m_ScriptFile = newstr("");
}

////////////////////////////////////////////////////////////////////////////////

sMap::sMap(const char* filename)
{
  if (!Load(filename))
  {
    m_NumLayers = 0;
    m_Layers = NULL;
    m_MusicFile = newstr("");
    m_ScriptFile = newstr("");
  }
}

////////////////////////////////////////////////////////////////////////////////

sMap::~sMap()
{
  delete[] m_MusicFile;
  delete[] m_ScriptFile;
  delete[] m_Layers;
  delete[] m_Entities;
}

////////////////////////////////////////////////////////////////////////////////

PACKED_STRUCT(MAP_HEADER)
  byte signature[4];
  word version;
  byte obsolete; // type;
  byte num_layers;
  byte base_layer;
  word num_entities;
  word startx;
  word starty;
  byte startlayer;
  byte startdirection;
  word num_strings;
  byte reserved[237];
END_STRUCT(MAP_HEADER)

PACKED_STRUCT(LAYER_HEADER)
  word  width;
  word  height;
  word  flags;
  sbyte parallax_x_mult;
  sbyte parallax_x_div;
  sbyte parallax_y_mult;
  sbyte parallax_y_div;
  sbyte scrolling_x_mult;
  sbyte scrolling_x_div;
  sbyte scrolling_y_mult;
  sbyte scrolling_y_div;
  byte  reserved[16];
END_STRUCT(LAYER_HEADER)

PACKED_STRUCT(ENTITY_HEADER)
  word mapx;
  word mapy;
  word layer;
  word type;
  byte reserved[8];
END_STRUCT(ENTITY_HEADER)

////////////////////////////////////////////////////////////////////////////////

ASSERT_STRUCT_SIZE(MAP_HEADER,    256)
ASSERT_STRUCT_SIZE(LAYER_HEADER,  30)
ASSERT_STRUCT_SIZE(ENTITY_HEADER, 16)

////////////////////////////////////////////////////////////////////////////////

static byte ReadMapByte(FILE* file)
{
  byte b;
  fread(&b, 1, 1, file);
  return b;
}

////////////////////////////////////////////////////////////////////////////////

static word ReadMapWord(FILE* file)
{
  word w;
  fread(&w, 1, 2, file);
  return w;
}

////////////////////////////////////////////////////////////////////////////////

static char* ReadMapString(FILE* file)
{
  word length = ReadMapWord(file);
  char* str = new char[length + 1];
  fread(str, 1, length, file);
  str[length] = 0;
  return str;
}

////////////////////////////////////////////////////////////////////////////////

static void SkipMapBytes(FILE* file, int bytes)
{
  fseek(file, bytes, SEEK_CUR);
}

////////////////////////////////////////////////////////////////////////////////

bool
sMap::Load(const char* filename)
{
  // open the file
  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  // read the header
  MAP_HEADER header;
  fread(&header, 1, sizeof(header), file);

  // make sure it's valid
  if (memcmp(header.signature, ".rmp", 4) != 0 ||
      header.version != 1 ||
      header.num_strings != 3)
  {
    fclose(file);
    return false;
  }

  m_StartX         = header.startx;
  m_StartY         = header.starty;
  m_StartLayer     = header.startlayer;
  m_StartDirection = header.startdirection;

  // read the strings (tileset, music, script)
  char* tileset_file = ReadMapString(file); // OBSOLETE
  delete[] m_MusicFile;
  m_MusicFile = ReadMapString(file);
  delete[] m_ScriptFile;
  m_ScriptFile = ReadMapString(file);

  // delete the old layer array and allocate a new one
  delete[] m_Layers;
  m_Layers = new sLayer[header.num_layers];
  m_NumLayers = header.num_layers;

  // read the layers
  for (int i = 0; i < header.num_layers; i++)
  {
    // read the layer header
    LAYER_HEADER lh;
    fread(&lh, 1, sizeof(lh), file);

    // read the layer name
    char* name = ReadMapString(file);

    // set all layer attributes
    m_Layers[i].SetName(name);
    m_Layers[i].SetSize(lh.width, lh.height);
    m_Layers[i].SetParallaxX(lh.parallax_x_mult, lh.parallax_x_div);
    m_Layers[i].SetParallaxY(lh.parallax_y_mult, lh.parallax_y_div);
    m_Layers[i].SetScrollingX(lh.scrolling_x_mult, lh.scrolling_x_div);
    m_Layers[i].SetScrollingY(lh.scrolling_y_mult, lh.scrolling_y_div);

    delete[] name;

    // read the layer data
    for (int iy = 0; iy < lh.height; iy++)
    {
      for (int ix = 0; ix < lh.width; ix++)
      {
        word tile;
        fread(&tile, 1, 2, file);
        m_Layers[i].SetTile(ix, iy, tile);
      }
    }

  } // end for layer

  // delete the old entities
  delete[] m_Entities;
  m_Entities = NULL;
  m_NumEntities = 0;

  // read entities
  for (int i = 0; i < header.num_entities; i++)
  {
    ENTITY_HEADER eh;
    fread(&eh, 1, sizeof(eh), file);

    sEntity* entity;
    switch (eh.type)
    {
      // WARP
      case 0:
      {
        sWarpEntity* warp = new sWarpEntity;

        // read the warp data
        char* destination_map   = ReadMapString(file);
        sword destination_x     = (sword)ReadMapWord(file);
        sword destination_y     = (sword)ReadMapWord(file);
        sword destination_layer = (sword)ReadMapWord(file);
        word  fade              = ReadMapWord(file);

        // set the warp data
        warp->SetDestinationMap(destination_map);
        warp->SetDestination(destination_x, destination_y, destination_layer);
        warp->SetFade(fade);

        delete[] destination_map;

        entity = warp;
        break;
      }
      
      // PERSON
      case 1:
      {
        sPersonEntity* person = new sPersonEntity;

        // read the person data
        byte  direction     = ReadMapByte(file);
        char* spriteset     = ReadMapString(file);
        byte  dialogue_type = ReadMapByte(file);
        char* dialogue      = ReadMapString(file);
        byte  speed         = ReadMapByte(file);
        word  num_tiles     = ReadMapWord(file);
        word  delay         = ReadMapWord(file);
        SkipMapBytes(file, 32); // reserved

        // set the person data
        switch (direction)
        {
          case 0: person->SetDirection(sPersonEntity::DIRECTION_UP);    break;
          case 2: person->SetDirection(sPersonEntity::DIRECTION_RIGHT); break;
          case 4: person->SetDirection(sPersonEntity::DIRECTION_DOWN);  break;
          case 6: person->SetDirection(sPersonEntity::DIRECTION_LEFT);  break;
        }
        person->SetSpritesetFile(spriteset);
        switch (dialogue_type)
        {
          case 0: person->SetDialogue(sPersonEntity::DIALOGUE_TEXT, dialogue); break;
          case 1: person->SetDialogue(sPersonEntity::DIALOGUE_FILE, dialogue); break;
        }
        person->SetWalkingRate(speed, num_tiles, delay);

        delete[] spriteset;
        delete[] dialogue;

        entity = person;
        break;
      }

      // TRIGGER
      case 2:
      {
        sTriggerEntity* trigger = new sTriggerEntity;

        // read the trigger data
        char* function = ReadMapString(file);

        // set the trigger data
        trigger->SetFunction(function);

        delete[] function;

        entity = trigger;
        break;
      }

      // DOODAD
      case 3:
      {
        sDoodadEntity* doodad = new sDoodadEntity;

        // read the doodad data
        char* doodad_file = ReadMapString(file);
        char* function    = ReadMapString(file);
        byte  obstructive = ReadMapByte(file);
        byte  activation_method = ReadMapByte(file);
        SkipMapBytes(file, 14); // reserved

        // set the doodad data
        doodad->SetSpritesetFile(doodad_file);
        doodad->SetFunction(function);
        doodad->SetObstructive(obstructive != 0);
        doodad->SetActivationMethod(activation_method);

        delete[] doodad_file;
        delete[] function;

        entity = doodad;
        break;
      }

      default:  // unknown
        continue;

    } // end switch

    entity->SetCoordinates(eh.mapx, eh.mapy, eh.layer);

    AddEntity(entity);
  }

  // if no tileset file was specified, it is appended to the map file
  if (strlen(tileset_file) == 0)
  {
    if (!m_Tileset.LoadFromFile(file))
    {
      fclose(file);
      return false;
    }
  }
  else
    m_Tileset.Clear();

  delete[] tileset_file;

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

static void WriteMapByte(FILE* file, byte b)
{
  fwrite(&b, 1, 1, file);
}

////////////////////////////////////////////////////////////////////////////////

static void WriteMapWord(FILE* file, word w)
{
  fwrite(&w, 1, 2, file);
}

////////////////////////////////////////////////////////////////////////////////

static void WriteMapString(FILE* file, const char* string)
{
  word len = strlen(string);
  WriteMapWord(file, len);
  fwrite(string, 1, len, file);
}

////////////////////////////////////////////////////////////////////////////////

bool
sMap::Save(const char* filename) const
{
  FILE* file = fopen(filename, "wb");
  if (file == NULL)
    return false;

  // write the map header
  MAP_HEADER header;
  memset(&header, 0, sizeof(header));
  memcpy(header.signature, ".rmp", 4);
  header.version        = 1;
  header.num_layers     = m_NumLayers;
  header.base_layer     = 0;
  header.num_entities   = m_NumEntities;
  header.startx         = m_StartX;
  header.starty         = m_StartY;
  header.startlayer     = m_StartLayer;
  header.startdirection = m_StartDirection;
  header.num_strings    = 3;
  fwrite(&header, 1, sizeof(header), file);

  // write the strings
  WriteMapString(file, "");  // OBSOLETE
  WriteMapString(file, m_MusicFile);
  WriteMapString(file, m_ScriptFile);
 
  // write layers
  for (int i = 0; i < m_NumLayers; i++)
  {
    // write the header
    LAYER_HEADER lh;
    memset(&lh, 0, sizeof(lh));
    lh.width            = m_Layers[i].GetWidth();
    lh.height           = m_Layers[i].GetHeight();
    lh.flags            = 0;
    lh.parallax_x_mult  = m_Layers[i].GetParallaxXMult();
    lh.parallax_x_div   = m_Layers[i].GetParallaxXDiv();
    lh.parallax_y_mult  = m_Layers[i].GetParallaxYMult();
    lh.parallax_y_div   = m_Layers[i].GetParallaxYDiv();
    lh.scrolling_x_mult = m_Layers[i].GetScrollingXMult();
    lh.scrolling_x_div  = m_Layers[i].GetScrollingXDiv();
    lh.scrolling_y_mult = m_Layers[i].GetScrollingYMult();
    lh.scrolling_y_div  = m_Layers[i].GetScrollingYDiv();
    fwrite(&lh, 1, sizeof(lh), file);

    // write the layer name
    WriteMapString(file, m_Layers[i].GetName());

    // write the layer data
    for (int iy = 0; iy < m_Layers[i].GetHeight(); iy++)
      for (int ix = 0; ix < m_Layers[i].GetWidth(); ix++)
      {
        word w = m_Layers[i].GetTile(ix, iy);
        fwrite(&w, 1, 2, file);
      }
  } // end for layer

  // write entities
  for (int i = 0; i < m_NumEntities; i++)
  {
    // write the header
    ENTITY_HEADER eh;
    memset(&eh, 0, sizeof(eh));
    eh.mapx  = m_Entities[i]->GetX();
    eh.mapy  = m_Entities[i]->GetY();
    eh.layer = m_Entities[i]->GetLayer();
    switch (m_Entities[i]->GetEntityType())
    {
      case sEntity::WARP:    eh.type = 0; break;
      case sEntity::PERSON:  eh.type = 1; break;
      case sEntity::TRIGGER: eh.type = 2; break;
      case sEntity::DOODAD:  eh.type = 3; break;
    }
    fwrite(&eh, 1, sizeof(eh), file);

    // write the entity data
    switch (m_Entities[i]->GetEntityType())
    {
      case sEntity::WARP:
      {
        sWarpEntity* warp = (sWarpEntity*)m_Entities[i];
        WriteMapString(file, warp->GetDestinationMap());
        WriteMapWord(file, warp->GetDestinationX());
        WriteMapWord(file, warp->GetDestinationY());
        WriteMapWord(file, warp->GetDestinationLayer());
        WriteMapWord(file, warp->GetFade());
        break;
      }

      case sEntity::PERSON:
      {
        sPersonEntity* person = (sPersonEntity*)m_Entities[i];
        WriteMapByte(file,   person->GetDirection());
        WriteMapString(file, person->GetSpritesetFile());
        WriteMapByte(file,   person->GetDialogueType());
        WriteMapString(file, person->GetDialogue());
        WriteMapByte(file,   person->GetWalkSpeed());
        WriteMapWord(file,   person->GetWalkDuration());
        WriteMapWord(file,   person->GetWalkDelay());
        for (int i = 0; i < 32; i++)
          WriteMapByte(file, 0);
        break;
      }

      case sEntity::TRIGGER:
      {
        sTriggerEntity* trigger = (sTriggerEntity*)m_Entities[i];
        WriteMapString(file, trigger->GetFunction());
        break;
      }

      case sEntity::DOODAD:
      {
        sDoodadEntity* doodad = (sDoodadEntity*)m_Entities[i];
        WriteMapString(file, doodad->GetSpritesetFile());
        WriteMapString(file, doodad->GetFunction());
        WriteMapByte(file, doodad->IsObstructive() ? 1 : 0);
        WriteMapByte(file, doodad->GetActivationMethod());
        for (int i = 0; i < 14; i++)
          WriteMapByte(file, 0);
        break;
      }
    } // end switch entity type
  } // end for entity

  // save the tileset
  if (!m_Tileset.SaveToFile(file))
    return false;

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sMap::Create(int width, int height, int layers)
{
  // clear out the old map
  delete[] m_MusicFile;
  m_MusicFile = newstr("");
  delete[] m_ScriptFile;
  m_ScriptFile = newstr("");

  m_NumLayers = 0;
  delete[] m_Layers;
  m_Layers = NULL;

  m_NumEntities = 0;
  delete[] m_Entities;
  m_Entities = NULL;

  // allocate the new map
  m_NumLayers = layers;
  m_Layers = new sLayer[layers];
  for (int i = 0; i < layers; i++)
    m_Layers[i].SetSize(width, height);

  // put a default tile in the tileset
  m_Tileset.Create(1);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sMap::BuildFromImage(CImage32& image, int tile_width, int tile_height)
{
  if (image.GetWidth() % tile_width != 0 ||
      image.GetHeight() % tile_height != 0)
    return false;

  if (!m_Tileset.BuildFromImage(image, tile_width, tile_height))
    return false;

  // clear out the old map
  delete[] m_MusicFile;
  m_MusicFile = newstr("");
  delete[] m_ScriptFile;
  m_ScriptFile = newstr("");

  m_NumLayers = 0;
  delete[] m_Layers;
  m_Layers = NULL;

  m_NumEntities = 0;
  delete[] m_Entities;
  m_Entities = NULL;

  int num_tiles_x = image.GetWidth() / tile_width;
  int num_tiles_y = image.GetHeight() / tile_height;

  // create the single layer
  m_NumLayers = 1;
  m_Layers = new sLayer[1];
  m_Layers[0].SetSize(num_tiles_x, num_tiles_y);

  // fill it in
  for (int ty = 0; ty < num_tiles_y; ty++)
    for (int tx = 0; tx < num_tiles_x; tx++)
    {
      int src_x = tx * tile_width;
      int src_y = ty * tile_height;

      // grab a tile
      RGBA* tile = new RGBA[tile_width * tile_height];
      for (int iy = 0; iy < tile_height; iy++)
      {
        memcpy(tile + iy * tile_width,
               image.GetPixels() + (src_y + iy) * image.GetWidth() + src_x,
               tile_width * sizeof(RGBA));
      }

      // find which tile it is in the tileset
      int t = 0;
      for (int i = 0; i < m_Tileset.GetNumTiles(); i++)
        if (memcmp(tile, m_Tileset.GetTile(i).GetPixels(), tile_width * tile_height * sizeof(RGBA)) == 0)
        {
          t = i;
          break;
        }

      m_Layers[0].SetTile(tx, ty, t);

      delete[] tile;
    }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

PACKED_STRUCT(V1MAP_HEADER)
  byte version;
  byte vsp_fname[13];
  byte music_fname[13];
  byte parallax_mode;
  byte parallax_multiplier;
  byte parallax_divisor;
  byte level_name[30];
  byte level_showname;
  byte level_saveable;
  word x_start;
  word y_start;
  byte level_hideable;
  byte level_warpable;
  word layer_size_x;
  word layer_size_y;
  byte reservedC[28];
END_STRUCT(V1MAP_HEADER)

PACKED_STRUCT(V2MAP_HEADER)
  byte signature[6];
  dword empty;
  byte  vsp_name[60];
  byte  music_name[60];
  byte  renderstring[20];
  word  x_start;
  word  y_start;
  byte  reserved[51];
  byte  num_layers;
END_STRUCT(V2MAP_HEADER)

PACKED_STRUCT(V2MAP_LAYERINFO)
  byte  multx, pdivx;
  byte  multy, pdivy;
  word  size_x, size_y;
  byte  transparent, hline;
END_STRUCT(V2MAP_LAYERINFO)

////////////////////////////////////////////////////////////////////////////////

ASSERT_STRUCT_SIZE(V1MAP_HEADER, 100)
ASSERT_STRUCT_SIZE(V2MAP_HEADER, 206)

////////////////////////////////////////////////////////////////////////////////

bool
sMap::Import_VergeMAP(const char* filename, const char* tilesetFilename)
{
  delete[] m_MusicFile;
  delete[] m_ScriptFile;
  delete[] m_Layers;
  delete[] m_Entities;

  m_MusicFile = newstr("");
  m_ScriptFile = newstr("");
  m_NumLayers = 0;
  m_Layers = NULL;
  m_NumEntities = 0;
  m_Entities = NULL;

  FILE* file;
  bool  bSuccess = false;

  file = fopen(filename, "rb");
  if (file == NULL) 
    return false;

  // check for v1 maps (ver 4) and v2 maps (ver 5)
  char signature[6];
  fread(signature, 1, 6, file);
  rewind(file);

  if (signature[0] == 4)
  {
    sTileset tileset;
    V1MAP_HEADER header;
    word *layer_background, *layer_foreground;
    sLayer layer[2];

    fread(&header, 1, sizeof(header), file);
    layer_background = new word[header.layer_size_x * header.layer_size_y];
    layer_foreground = new word[header.layer_size_x * header.layer_size_y];
    fread(layer_background, 1, header.layer_size_x * header.layer_size_y * sizeof(word), file);
    fread(layer_foreground, 1, header.layer_size_x * header.layer_size_y * sizeof(word), file);

    if (stricmp(tilesetFilename + strlen(tilesetFilename) - 4, ".vsp") == 0)
      bSuccess = tileset.Import_VSP(tilesetFilename);
    else
      bSuccess = tileset.Load(tilesetFilename);

    if (bSuccess)
    {
      // process map and see if the map has tiles that are out of range
      int highestTileIndex = 0;
      for (int j=0; j<header.layer_size_y; j++)
        for (int i=0; i<header.layer_size_x; i++)
          if (layer_background[j * header.layer_size_x + i] >= highestTileIndex)
            highestTileIndex = layer_background[j * header.layer_size_x + i];
          else if (layer_foreground[j * header.layer_size_x + i] >= highestTileIndex)
            highestTileIndex = layer_foreground[j * header.layer_size_x + i];

      if (highestTileIndex >= tileset.GetNumTiles())
        bSuccess = false;

      // transfer data across into the sMap now...
      if (bSuccess)
      {
        layer[0].SetName("Background");
        layer[1].SetName("Foreground");
        layer[0].SetSize(header.layer_size_x, header.layer_size_y);
        layer[1].SetSize(header.layer_size_x, header.layer_size_y);

        for (int j=0; j<header.layer_size_y; j++)
          for (int i=0; i<header.layer_size_x; i++)
          {
            layer[0].SetTile(i,j, layer_background[j * header.layer_size_x + i]);

            if (layer_foreground[j * header.layer_size_x + i])
            layer[1].SetTile(i,j, layer_foreground[j * header.layer_size_x + i]);
            else
              layer[1].SetTile(i,j, tileset.GetNumTiles());
          }

        tileset.AppendTiles(1);
        memset(tileset.GetTile(tileset.GetNumTiles() - 1).GetPixels(), 0, 256 * sizeof(RGBA));
        m_Tileset = tileset;
        AppendLayer(layer[0]);
        AppendLayer(layer[1]);
        SetMusicFile((char*)header.music_fname);
        SetStartX(header.x_start);
        SetStartX(header.y_start);
        SetStartDirection(4);

        // calculate the parallax mode
        for (int i=0; i<2; i++)
        {
          GetLayer(i).SetParallaxX(1, 1);
          GetLayer(i).SetParallaxY(1, 1);
          GetLayer(i).SetScrollingX(1, 1);
          GetLayer(i).SetScrollingX(1, 1);
      }

        switch(header.parallax_mode)
        {
        case 0:
          SetStartLayer(0);
          break;

        case 1:
          SetStartLayer(1);
          break;

        case 2:
          SetStartLayer(1);
          GetLayer(0).SetParallaxX(header.parallax_multiplier, header.parallax_divisor);
          GetLayer(0).SetParallaxY(header.parallax_multiplier, header.parallax_divisor);
          break;

        case 3:
          SetStartLayer(0);
          GetLayer(1).SetParallaxX(header.parallax_multiplier, header.parallax_divisor);
          GetLayer(1).SetParallaxY(header.parallax_multiplier, header.parallax_divisor);
          break;
    }

      }
    }

    // cleanup
    delete[] layer_background;
    delete[] layer_foreground;
  }
  else if (strcmp(signature, "MAP5") == 0)
  {
    V2MAP_HEADER header;
    V2MAP_LAYERINFO LayerInfo[7];
    sTileset tileset;
    word *mapLayer[7];
    int i,j,k;
    int highestTileIndex = 0;

    fread(&header, 1, sizeof(header), file);
    for (i=0; i<header.num_layers; i++)
    {
      fread(&LayerInfo[i], 1, sizeof(V2MAP_LAYERINFO), file);
      //bug for v2's map: two bytes are added for no reason
      getc(file);
      getc(file);
    }

    // get info about map and uncompress it
    for (i=0; i<header.num_layers; i++)
      mapLayer[i] = new word[LayerInfo[i].size_x * LayerInfo[i].size_y];

    for (i=0; i<header.num_layers; i++)
    {
      // god, this is so dumb. It's supposed to be the buffersize, but do I look like I need it?
      fread(&j, 1, 4, file);
      for (j=0; j<LayerInfo[i].size_x*LayerInfo[i].size_y; j++)
      {
        word value;
        byte run;
        
        fread(&value, 1, sizeof(word), file);

        if ((value & 0xFF00) == 0xFF00)
        {
          run = (byte)value & 0x00FF;
          fread(&value, 1, sizeof(word), file);
          
          mapLayer[i][j] = value;
          for (k=1; k<run; k++)
          {
            j++;
            mapLayer[i][j] = value;
  }
        }
        else
  {
          mapLayer[i][j]  = value;
        }
      }
  }

    if (stricmp(tilesetFilename + strlen(tilesetFilename) - 4, ".vsp") == 0)
      bSuccess = tileset.Import_VSP(tilesetFilename);
  else
      bSuccess = tileset.Load(tilesetFilename);


    // transfer map array into the class
    if (bSuccess)
    {
      highestTileIndex = 0;
      // check for any tile index larger than the tilset's index
      for (i=0; i<header.num_layers; i++)
        for (j=0; j<LayerInfo[i].size_x*LayerInfo[i].size_y; j++)
          if (mapLayer[i][j] >= highestTileIndex)
            highestTileIndex = mapLayer[i][j];

      if (highestTileIndex >= tileset.GetNumTiles())
        bSuccess = false;

      //printf("Tiles in index: %i\n", tileset.GetNumTiles());
      //printf("Tiles in map: %i\n", highestTileIndex);

    if (bSuccess)
      {
        sLayer *layer;
        layer = new sLayer[header.num_layers];

        for (i=0; i<header.num_layers; i++)
        {
          char Name[7];
          memcpy(Name, "Layer A", 8);
          Name[6] += i;

          layer[i].SetName(Name);
          layer[i].SetSize(LayerInfo[i].size_x, LayerInfo[i].size_y);
        }

        for (i=0; i<header.num_layers; i++)
        {
          for (j=0; j<LayerInfo[i].size_y; j++)
            for (k=0; k<LayerInfo[i].size_x; k++)
              layer[i].SetTile(k, j, mapLayer[i][(j * LayerInfo[i].size_x) + k]);

          layer[i].SetParallaxX(LayerInfo[i].multx, LayerInfo[i].pdivx);
          layer[i].SetParallaxY(LayerInfo[i].multy, LayerInfo[i].pdivy);
          layer[i].SetScrollingX(1,1);
          layer[i].SetScrollingY(1,1);
        }

        for (i=0; i<(int)strlen((char*)header.renderstring); i++)
          switch(header.renderstring[i])
  {
          case '1': AppendLayer(layer[0]); j = 0; break;
          case '2': AppendLayer(layer[1]); j = 1; break;
          case '3': AppendLayer(layer[2]); j = 2; break;
          case '4': AppendLayer(layer[3]); j = 3; break;
          case '5': AppendLayer(layer[4]); j = 4; break;
          case '6': AppendLayer(layer[5]); j = 5; break;
          case 'E': SetStartLayer(j); break;
          }

        SetMusicFile((char*)header.music_name);
        SetStartX(header.x_start);
        SetStartY(header.y_start);
        m_Tileset = tileset;

        delete[] layer;
      } 
    }

    for (i=0; i<header.num_layers; i++)
      delete mapLayer[i];
  }

  fclose(file);
  return bSuccess;
} 

////////////////////////////////////////////////////////////////////////////////

int
sMap::GetStartX() const
{
  return m_StartX;
}

////////////////////////////////////////////////////////////////////////////////

int
sMap::GetStartY() const
{
  return m_StartY;
}

////////////////////////////////////////////////////////////////////////////////

int
sMap::GetStartLayer() const
{
  return m_StartLayer;
}

////////////////////////////////////////////////////////////////////////////////

int
sMap::GetStartDirection() const
{
  return m_StartDirection;
}

////////////////////////////////////////////////////////////////////////////////

const char*
sMap::GetMusicFile() const
{
  return m_MusicFile;
}

////////////////////////////////////////////////////////////////////////////////

const char*
sMap::GetScriptFile() const
{
  return m_ScriptFile;
}

////////////////////////////////////////////////////////////////////////////////

int
sMap::GetNumEntities() const
{
  return m_NumEntities;
}

////////////////////////////////////////////////////////////////////////////////

sEntity&
sMap::GetEntity(int i)
{
  return *m_Entities[i];
}

////////////////////////////////////////////////////////////////////////////////

const sEntity& sMap::GetEntity(int i) const
{
  return *m_Entities[i];
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::SetStartX(int x)
{
  m_StartX = x;
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::SetStartY(int y)
{
  m_StartY = y;
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::SetStartLayer(int layer)
{
  m_StartLayer = layer;
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::SetStartDirection(int direction)
{
  m_StartDirection = direction;
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::InsertLayer(int where, const sLayer& layer)
{
  // resize the layer array
  resize(m_Layers, m_NumLayers, m_NumLayers + 1);

  // move the layers past 'where' up one
  for (int i = m_NumLayers; i > where; i--)
    m_Layers[i] = m_Layers[i - 1];

  // stick in the new layer
  m_Layers[where] = layer;
  m_NumLayers++;
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::AppendLayer(const sLayer& layer)
{
  resize(m_Layers, m_NumLayers, m_NumLayers + 1);

  // stick the layer on the end
  m_Layers[m_NumLayers] = layer;
  m_NumLayers++;
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::DeleteLayer(int where)
{
  for (int i = where; i < m_NumLayers - 1; i++)
    m_Layers[i] = m_Layers[i + 1];

  resize(m_Layers, m_NumLayers, m_NumLayers - 1);
  m_NumLayers--;
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::SwapLayers(int layer1, int layer2)
{
  sLayer l = m_Layers[layer2];
  m_Layers[layer2] = m_Layers[layer1];
  m_Layers[layer1] = l;
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::AddEntity(sEntity* entity)
{
  resize(m_Entities, m_NumEntities, m_NumEntities + 1);
  m_Entities[m_NumEntities] = entity;
  m_NumEntities++;
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::DeleteEntity(int index)
{
  delete m_Entities[index];
  sEntity** new_entities = new sEntity*[m_NumEntities - 1];
  
  for (int i = 0; i < index; i++)
    new_entities[i] = m_Entities[i];
  for (int i = index; i < m_NumEntities - 1; i++)
    new_entities[i] = m_Entities[i + 1];
  
  m_NumEntities--;
  delete[] m_Entities;
  m_Entities = new_entities;
}

////////////////////////////////////////////////////////////////////////////////

int
sMap::FindEntity(int x, int y, int layer)
{
  for (int i = 0; i < m_NumEntities; i++)
    if (m_Entities[i]->GetX() == x &&
        m_Entities[i]->GetY() == y &&
        m_Entities[i]->GetLayer() == layer)
      return i;
  return -1;
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::SetMusicFile(const char* music)
{
  delete[] m_MusicFile;
  m_MusicFile = newstr(music);
}

////////////////////////////////////////////////////////////////////////////////

void
sMap::SetScriptFile(const char* script)
{
  delete[] m_ScriptFile;
  m_ScriptFile = newstr(script);
}

////////////////////////////////////////////////////////////////////////////////
