#include <stdio.h>
#include <string.h>
#include "Font.hpp"
#include "x++.hpp"
#include "packed.h"


////////////////////////////////////////////////////////////////////////////////

sFontCharacter::sFontCharacter(int width, int height)
{
  m_Width  = width;
  m_Height = height;
  m_Pixels = new byte[width * height];
  memset(m_Pixels, 0, width * height);
}

////////////////////////////////////////////////////////////////////////////////

sFontCharacter::sFontCharacter(const sFontCharacter& character)
{
  m_Width  = character.m_Width;
  m_Height = character.m_Height;
  m_Pixels = duplicate(character.m_Pixels, m_Width * m_Height);
}

////////////////////////////////////////////////////////////////////////////////

sFontCharacter::~sFontCharacter()
{
  delete[] m_Pixels;
}

////////////////////////////////////////////////////////////////////////////////

sFontCharacter&
sFontCharacter::operator=(const sFontCharacter& character)
{
  delete[] m_Pixels;

  m_Width  = character.m_Width;
  m_Height = character.m_Height;
  m_Pixels = duplicate(character.m_Pixels, m_Width * m_Height);

  return *this;
}

////////////////////////////////////////////////////////////////////////////////

void
sFontCharacter::Resize(int width, int height)
{
  if (width == 0 || height == 0)
  {
    m_Width  = width;
    m_Height = height;
    delete[] m_Pixels;
    m_Pixels = NULL;
  }
  else
  {
    byte* newPixels = new byte[width * height];
    for (int ix = 0; ix < width; ix++)
      for (int iy = 0; iy < height; iy++)
      {
        if (ix < m_Width && iy < m_Height)
          newPixels[iy * width + ix] = m_Pixels[iy * m_Width + ix];
        else
          newPixels[iy * width + ix] = 0;
      }

    m_Width  = width;
    m_Height = height;
    delete[] m_Pixels;
    m_Pixels = newPixels;
  }
}

////////////////////////////////////////////////////////////////////////////////

void
sFontCharacter::SlideLeft()
{
  sFontCharacter New(m_Width, m_Height);
  for (int ix = 0; ix < m_Width; ix++)
    for (int iy = 0; iy < m_Height; iy++)
    {
      int dx = ix - 1;
      if (dx < 0)
        dx = m_Width - 1;
      New.m_Pixels[iy * m_Width + dx] = m_Pixels[iy * m_Width + ix];
    }
  *this = New;
}

////////////////////////////////////////////////////////////////////////////////

void
sFontCharacter::SlideUp()
{
  sFontCharacter New(m_Width, m_Height);
  for (int ix = 0; ix < m_Width; ix++)
    for (int iy = 0; iy < m_Height; iy++)
    {
      int dy = iy - 1;
      if (dy < 0)
        dy = m_Height - 1;
      New.m_Pixels[dy * m_Width + ix] = m_Pixels[iy * m_Width + ix];
    }
  *this = New;
}

////////////////////////////////////////////////////////////////////////////////

void
sFontCharacter::SlideRight()
{
  sFontCharacter New(m_Width, m_Height);
  for (int ix = 0; ix < m_Width; ix++)
    for (int iy = 0; iy < m_Height; iy++)
    {
      int dx = ix + 1;
      if (dx > m_Width - 1)
        dx = 0;
      New.m_Pixels[iy * m_Width + dx] = m_Pixels[iy * m_Width + ix];
    }
  *this = New;
}

////////////////////////////////////////////////////////////////////////////////

void
sFontCharacter::SlideDown()
{
  sFontCharacter New(m_Width, m_Height);
  for (int ix = 0; ix < m_Width; ix++)
    for (int iy = 0; iy < m_Height; iy++)
    {
      int dy = iy + 1;
      if (dy > m_Height - 1)
        dy = 0;
      New.m_Pixels[dy * m_Width + ix] = m_Pixels[iy * m_Width + ix];
    }
  *this = New;
}

////////////////////////////////////////////////////////////////////////////////

sFont::sFont(int num_characters, int width, int height)
{
  m_NumCharacters = num_characters;
  m_Characters = new sFontCharacter[num_characters];
  for (int i = 0; i < num_characters; i++)
    m_Characters[i].Resize(width, height);
}

////////////////////////////////////////////////////////////////////////////////

sFont::~sFont()
{
  delete[] m_Characters;
}

////////////////////////////////////////////////////////////////////////////////

PACKED_STRUCT(FONT_HEADER)
  byte signature[4];
  word version;
  word num_characters;
  byte reserved[248];
END_STRUCT(FONT_HEADER)

PACKED_STRUCT(CHARACTER_HEADER)
  word width;
  word height;
  byte reserved[28];
END_STRUCT(CHARACTER_HEADER)

ASSERT_STRUCT_SIZE(FONT_HEADER, 256)
ASSERT_STRUCT_SIZE(CHARACTER_HEADER, 32);

////////////////////////////////////////////////////////////////////////////////

bool
sFont::Load(const char* filename)
{
  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  // read and check header
  FONT_HEADER header;
  fread(&header, 1, sizeof(header), file);
  if (memcmp(header.signature, ".rfn", 4) != 0 ||
      header.version != 1)
  {
    fclose(file);
    return false;
  }

  // allocate characters
  m_NumCharacters = header.num_characters;
  m_Characters    = new sFontCharacter[m_NumCharacters];

  // read them
  for (int i = 0; i < m_NumCharacters; i++)
  {
    CHARACTER_HEADER character_header;
    fread(&character_header, 1, sizeof(character_header), file);
    m_Characters[i].Resize(character_header.width, character_header.height);
    fread(m_Characters[i].GetPixels(), 1, character_header.width * character_header.height, file);
  }

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
sFont::Save(const char* filename) const
{
  FILE* file = fopen(filename, "wb");
  if (file == NULL)
    return false;

  // write header
  FONT_HEADER header;
  memset(&header, 0, sizeof(header));
  memcpy(header.signature, ".rfn", 4);
  header.version = 1;
  header.num_characters = m_NumCharacters;
  fwrite(&header, 1, sizeof(header), file);

  // write characters
  for (int i = 0; i < m_NumCharacters; i++)
  {
    CHARACTER_HEADER character_header;
    memset(&character_header, 0, sizeof(character_header));
    character_header.width  = m_Characters[i].GetWidth();
    character_header.height = m_Characters[i].GetHeight();
    fwrite(&character_header, 1, sizeof(character_header), file);
    fwrite(m_Characters[i].GetPixels(), character_header.width * character_header.height, 1, file);
  }

  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

void
sFont::SetNumCharacters(int num_characters)
{
  sFontCharacter* newCharacters = new sFontCharacter[num_characters];
  int copy_count = m_NumCharacters < num_characters ? m_NumCharacters : num_characters;
  for (int i = 0; i < copy_count; i++)
    newCharacters[i] = m_Characters[i];

  // free old characters
  delete[] m_Characters;
  m_Characters = newCharacters;
  m_NumCharacters = num_characters;
}

////////////////////////////////////////////////////////////////////////////////
