#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <png.h>
#include "Image32.hpp"
#include "packed.h"
#include "x++.hpp"



#define BEGIN_RESIZE_FUNCTION() int old_width = m_Width; int old_height = m_Height;
#define END_RESIZE_FUNCTION() OnResize(old_width, old_height);


////////////////////////////////////////////////////////////////////////////////

inline int sgn(int i)
{
  return (i < 0 ? -1 : (i > 0 ? 1 : 0));
}

////////////////////////////////////////////////////////////////////////////////

inline void max(double defVal, double newVal)
{
  defVal = (defVal < newVal ? defVal : newVal);
}

////////////////////////////////////////////////////////////////////////////////

CImage32::CImage32()
: m_Width(0)
, m_Height(0)
, m_Pixels(NULL)
{
}

////////////////////////////////////////////////////////////////////////////////

CImage32::CImage32(int width, int height)
: m_Width(width)
, m_Height(height)
, m_Pixels(new RGBA[width * height])
{
  memset(m_Pixels, 0, width * height * sizeof(RGBA));
}

////////////////////////////////////////////////////////////////////////////////

CImage32::CImage32(const CImage32& image)
{
  m_Width  = image.m_Width;
  m_Height = image.m_Height;
  m_Pixels = new RGBA[m_Width * m_Height];
  memcpy(m_Pixels, image.m_Pixels, m_Width * m_Height * sizeof(RGBA));
}

////////////////////////////////////////////////////////////////////////////////

CImage32::~CImage32()
{
  delete[] m_Pixels;
}

////////////////////////////////////////////////////////////////////////////////

CImage32&
CImage32::operator=(const CImage32& image)
{
  delete[] m_Pixels;
  m_Width  = image.m_Width;
  m_Height = image.m_Height;
  m_Pixels = new RGBA[m_Width * m_Height];
  memcpy(m_Pixels, image.m_Pixels, m_Width * m_Height * sizeof(RGBA));
  return *this;
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::Create(int width, int height)
{
  BEGIN_RESIZE_FUNCTION()

  delete[] m_Pixels;

  m_Width  = width;
  m_Height = height;
  m_Pixels = new RGBA[width * height];
  for (int i = 0; i < width * height; i++)
    m_Pixels[i] = rgbaBlack;

  END_RESIZE_FUNCTION()
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::Clear()
{
  for (int i = 0; i < m_Width * m_Height; i++)
    m_Pixels[i] = rgbaBlack;
}

////////////////////////////////////////////////////////////////////////////////

PACKED_STRUCT(I32_HEADER)
  byte  signature[4]; // ".i32"
  byte  version;      // 1
  dword width;
  dword height;
  byte  compression;
  byte  reserved[242];
END_STRUCT(I32_HEADER)

ASSERT_STRUCT_SIZE(I32_HEADER, 256)

////////////////////////////////////////////////////////////////////////////////

static int bytesleft(FILE* file)
{
  int currentlocation = ftell(file);
  int filesize;
  fseek(file, 0, SEEK_END);
  filesize = ftell(file);
  fseek(file, currentlocation, SEEK_SET);
  return filesize - currentlocation;
}

////////////////////////////////////////////////////////////////////////////////

bool
CImage32::Load(const char* filename)
{
  int old_width = m_Width;
  int old_height = m_Height;
  RGBA* old_pixels = m_Pixels;
  m_Width  = 0;
  m_Height = 0;
  m_Pixels = NULL;

  bool result;
  if (strcmp_ci(filename + strlen(filename) - 4, ".png") == 0)
    result = Import_PNG(filename);
  else if (strcmp_ci(filename + strlen(filename) - 4, ".pcx") == 0)
    result = Import_PCX(filename);
  else if (strcmp_ci(filename + strlen(filename) - 4, ".bmp") == 0 ||
           strcmp_ci(filename + strlen(filename) - 4, ".dib") == 0  )
    result = Import_BMP(filename);
  else
    result = false;

  if (result == false)
  {
    m_Width = old_width;
    m_Height = old_height;
    m_Pixels = old_pixels;
  }
  else
    delete[] old_pixels;

  return result;
}

////////////////////////////////////////////////////////////////////////////////

bool
CImage32::Save(const char* filename) const
{
  if (strcmp_ci(filename + strlen(filename) - 4, ".png") == 0)
    return Export_PNG(filename);
  else
    return false;
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::Resize(int width, int height)
{
  BEGIN_RESIZE_FUNCTION()

  RGBA* new_pixels = new RGBA[width * height];
  for (int ix = 0; ix < width; ix++)
    for (int iy = 0; iy < height; iy++)
    {
      if (ix < m_Width && iy < m_Height)
        new_pixels[iy * width + ix] = m_Pixels[iy * m_Width + ix];
      else
        new_pixels[iy * width + ix] = rgbaBlack;
    }

  m_Width  = width;
  m_Height = height;
  delete[] m_Pixels;
  m_Pixels = new_pixels;

  END_RESIZE_FUNCTION()
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::Rescale(int width, int height)
{
  BEGIN_RESIZE_FUNCTION()

  RGBA* NewPixels = new RGBA[width * height];
  if (NewPixels == NULL)
    return;

  int HorzAspectRatio, VertAspectRatio;
  int x,y;
  int ix, iy;
  HorzAspectRatio = (width * 100) / m_Width;   // (dstWidth / srcWidth) * 100
  VertAspectRatio = (height * 100) / m_Height; // (dstHeight / srcHeight) * 100

  for (iy=0; iy<height; iy++)
  {
    y = (iy * 100) / VertAspectRatio;

    for (ix=0; ix<width; ix++)
    {
      x = (ix * 100) / HorzAspectRatio;
      NewPixels[(iy * width) + ix] = m_Pixels[(y * m_Width) + x];
    }
  }

  m_Width  = width;
  m_Height = height;
  delete[] m_Pixels;
  m_Pixels = NewPixels;

  END_RESIZE_FUNCTION()
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::FlipHorizontal()
{
  for (int y = 0; y < m_Height; y++)
    for (int x = 0; x < m_Width/2; x++)
    {
      RGBA TempPixel = m_Pixels[y * m_Width + x];
      RGBA TempPixel2 = m_Pixels[y * m_Width + (m_Width - 1 -x)];
      m_Pixels[y * m_Width + (m_Width - 1 - x)] = TempPixel;
      m_Pixels[y * m_Width + x] = TempPixel2;
    }
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::FlipVertical()
{
  for (int ix = 0; ix < m_Width; ix++)
    for (int iy = 0; iy < m_Height / 2; iy++)
    {
      RGBA t1 = m_Pixels[iy * m_Width + ix];
      RGBA t2 = m_Pixels[(m_Height - 1 - iy) * m_Width + ix];
      m_Pixels[(m_Height - 1 - iy) * m_Width + ix] = t1;
      m_Pixels[iy * m_Width + ix] = t2;
    }
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::Translate(int dx, int dy)
{
  RGBA* result = new RGBA[m_Width * m_Height];

  for (int iy = 0; iy < m_Height; iy++)
    for (int ix = 0; ix < m_Width; ix++)
    {
      int sx = ix - dx;
      int sy = iy - dy;

      if (sx < 0)
        sx += m_Width;
      else if (sx > m_Width - 1)
        sx -= m_Width;

      if (sy < 0)
        sy += m_Height;
      else if (sy > m_Height - 1)
        sy -= m_Height;

      result[iy * m_Width + ix] = m_Pixels[sy * m_Width + sx];
    }

  memcpy(m_Pixels, result, m_Width * m_Height * sizeof(RGBA));
  delete[] result;
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::Rotate(double radians, bool autoSize)
{
  //double cx, cy;
  //double width, height;
  double    ix, iy;

  //cx = m_Width / 2; 
  //cy = m_Height / 2;
/*  
  if (autoSize)
  {
    double autocx, autocy;
    double xLeft, xRight, yTop, yBot;

    // finds the largest value and adjust it.
    autocx = autocy = 0;
    xLeft     = _RotateX(0, radians, cx);
    xRight    = _RotateX(m_Width, radians, cx);
    yTop      = _RotateY(0, radians, cy);
    yBot      = _RotateY(m_Height, radians, cy);
    max(autocx, xLeft);
    max(autocx, xRight);
    max(autocy, yTop);
    max(autocy, yBot);

    autocx -= m_Width;
    autocy -= m_Height;
    cx = autocx;
    cy = autocy;
  }

  width  = cx * 2;
  height = cy * 2;
  //cx += 0.5;
  //cy += 0.5;
*/
  //ix = _RotateX(x-(m_Width/2), y, radians);
  //iy = _RotateY(x, y-(m_Height/2), radians);

  //ix += m_Width/2; iy += m_Height/2;

  RGBA* NewPixels = new RGBA[m_Width * m_Height];
  if (NewPixels == NULL) return;
  memset(NewPixels, 0, m_Width * m_Height * sizeof(RGBA));

  for (int y=0; y<m_Height; y++)
    for (int x=0; x<m_Width; x++)
    {
      
      //ix = iy = 0;

      ix = RotateX(x-(m_Width/2), y, radians);
      iy = RotateY(x, y-(m_Height/2), radians);

      ix += m_Width/2; iy += m_Height/2;
      ////if ((ix>=0) && (ix<width))
      ////  if ((iy>=0) && (iy<height))
      ////    NewPixels[(iy * (int)width) + ix] = m_Pixels[(y * m_Width) + x];
      if ((ix>=0) && (ix<m_Width))
        if ((iy>=0) && (iy<m_Height))
          NewPixels[((int)iy * m_Width) + (int)ix] = m_Pixels[(y * m_Width) + x];
    }
/*
  if (autoSize)
  { 
    m_Width  = (int)width; 
    m_Height = (int)height; 
  }
*/
  delete[] m_Pixels;
  m_Pixels = NewPixels;
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::RotateCW()
{
  RGBA* new_pixels = new RGBA[m_Width * m_Height];

  for (int iy = 0; iy < m_Height; iy++)
    for (int ix = 0; ix < m_Width; ix++)
    {
      int dx = 15 - iy;
      int dy = ix;
      new_pixels[dy * m_Width + dx] = m_Pixels[iy * m_Width + ix];
    }

  delete[] m_Pixels;
  m_Pixels = new_pixels;
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::RotateCCW()
{
  RGBA* new_pixels = new RGBA[m_Width * m_Height];

  for (int iy = 0; iy < m_Height; iy++)
    for (int ix = 0; ix < m_Width; ix++)
    {
      int dx = iy;
      int dy = 15 - ix;
      new_pixels[dy * m_Width + dx] = m_Pixels[iy * m_Width + ix];
    }

  delete[] m_Pixels;
  m_Pixels = new_pixels;
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::SetPixel(int x, int y, RGBA color)
{
  if ((x < m_Width) && (x>=0))
    if ((y < m_Height) && (y>= 0))
        m_Pixels[(y * m_Width) + x] = color;
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::Line(int x1, int y1, int x2, int y2, RGBA color)
{
  int Dx, Dy;
  int absDx, absDy;
  int sgnDx, sgnDy;
  int posX, posY;
  int x,y,i;

  // set up the variables for use.
  Dx = x2 - x1;
  Dy = y2 - y1;
  absDx = abs(Dx) + 1;
  absDy = abs(Dy) + 1;
  sgnDx = sgn(Dx);
  sgnDy = sgn(Dy);
  x = absDx / 2;
  y = absDy / 2;
  posX = x1;
  posY = y1;

  if (absDx >= absDy) // if the line is more horizontal
  {
    for (i=0; i<absDx; i++)
    {
      SetPixel(posX, posY, color);
      y += absDy;
      if (y >= absDx)
      {
        y -= absDx;
        posY += sgnDy;
      }
      posX += sgnDx;
    }
  }
  else // the line is more vertical
  {
    for (i=0; i<absDy; i++)
    {
      SetPixel(posX, posY, color);
      x += absDx;
      if (x >= absDy)
      {
        x -= absDy;
        posX += sgnDy;
      }
      posY += sgnDy;
    }
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::Circle(int x, int y, int r, RGBA color, RGBA backcolor)
{
  int cx = 0;
  int cy = abs(r);
  int df = 1 - abs(r);
  int d_e = 3;
  int d_se = -2 * abs(r) + 5;

  do {

    SetPixel(x+cx, x+cy, color);
    Line(x-cy+1, y-cx, x+cy-1, y-cx, backcolor);
    if (cx) 
    {
      SetPixel(x-cx, y+cy, color);
      Line(x-cy+1, y+cx, x+cy-1, y+cx, backcolor);
    }
    if (cy) SetPixel(x+cx, y-cy, color);
    if (cx && cy) SetPixel(x-cx, y-cy, color);
    if (cx != cy)
    {
      SetPixel(x+cy, y+cx, color);
      if (cx) SetPixel(x+cy, y-cx, color);
      if (cy) SetPixel(x-cy, y+cx, color);
      if (cx && cy) SetPixel(x-cy, y-cx, color);
    }

    if (df < 0)  
    {
      df += d_e;
      d_e += 2;
      d_se += 2;
    }
    else 
    {
      if (cx != cy)
      {
        Line(x-cx, y-cy+1, x+cx, y-cy+1, backcolor);
        if (cy) Line(x-cx, y+cy-1, x+cx, y+cy-1, backcolor);
      }

      df += d_se;
      d_e += 2;
      d_se += 4;
      cy--;
    }

    cx++;
  } while (cx <= cy);
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::Rectangle(int x1, int y1, int x2, int y2, RGBA color, RGBA backcolor)
{
  int Dx, Dy;
  int absDx, absDy;
  int sgnDx, sgnDy;
  int x,y;
  int i,j;

  Dx = x2 - x1;
  Dy = y2 - y1;
  absDx = abs(Dx) + 1;
  absDy = abs(Dy) + 1;
  sgnDx = sgn(Dx);
  sgnDy = sgn(Dy);

  // draw the filler
  //y = y1 + 1;
  y = y1 + sgnDy;
  for (j=0; j<absDy - 1; j++)
  {
    //x = x1 + 1;
    x = x1 + sgnDx;
    for(i=0; i<absDx - 1; i++)
    {
      SetPixel(x,y, backcolor);
      x += sgnDx;
    }
    y += sgnDy;
  }

  // draw the outline
  x = x1;
  for (i=0; i<absDx; i++)
  {
    SetPixel(x, y1, color);
    SetPixel(x, y2, color);
    x += sgnDx;
  }

  y = y1;
  for (j=0; j<absDy; j++)
  {
    SetPixel(x1, y, color);
    SetPixel(x2, y, color);
    y+= sgnDy;
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::OnResize(int old_width, int old_height)
{
}

////////////////////////////////////////////////////////////////////////////////

bool
CImage32::Import_PNG(const char* filename)
{
  // open file
  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  // verify signature
  byte sig[8];
  fread(sig, 1, 8, file);
  if (png_sig_cmp(sig, 0, 8))
  {
    fclose(file);
    return false;
  }

  // read struct
  png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (!png_ptr)
  {
    fclose(file);
    return false;
  }

  // info struct
  png_infop info_ptr = png_create_info_struct(png_ptr);
  if (!info_ptr)
  {
    png_destroy_read_struct(&png_ptr, NULL, NULL);
    fclose(file);
    return false;
  }

  // read the image
  png_init_io(png_ptr, file);
  png_set_sig_bytes(png_ptr, 8);
  int png_transform = PNG_TRANSFORM_STRIP_16 | PNG_TRANSFORM_PACKING | PNG_TRANSFORM_PACKSWAP;
  png_read_png(png_ptr, info_ptr, png_transform, NULL);

  if (info_ptr->row_pointers == NULL)
  {
    png_destroy_read_struct(&png_ptr, NULL, NULL);
    fclose(file);
    return false;
  }

  // initialize CImage32 members
  m_Width = png_get_image_width(png_ptr, info_ptr);
  m_Height = png_get_image_height(png_ptr, info_ptr);
  m_Pixels = new RGBA[m_Width * m_Height];

  // decode based on pixel depth
  int pixel_depth = png_get_bit_depth(png_ptr, info_ptr) * png_get_channels(png_ptr, info_ptr);
  void** row_pointers = (void**)png_get_rows(png_ptr, info_ptr);

  if (pixel_depth == 32)
  {
    for (int i = 0; i < m_Height; i++)
    {
      RGBA* row = (RGBA*)(row_pointers[i]);
      for (int j = 0; j < m_Width; j++)
        m_Pixels[i * m_Width + j] = row[j];
    }
  }
  else if (pixel_depth == 24)
  {
    for (int i = 0; i < m_Height; i++)
    {
      RGB* row = (RGB*)(row_pointers[i]);
      for (int j = 0; j < m_Width; j++)
      {
        RGBA p = { row[j].red, row[j].green, row[j].blue, 255 };
        m_Pixels[i * m_Width + j] = p;
      }
    }
  }
  else if (pixel_depth == 8)
  {
    png_colorp palette;
    int num_palette;
    png_get_PLTE(png_ptr, info_ptr, &palette, &num_palette);

    for (int i = 0; i < m_Height; i++)
    {
      byte* row = (byte*)(row_pointers[i]);
      for (int j = 0; j < m_Width; j++)
      {
        m_Pixels[i * m_Width + j].red   = palette[row[j]].red;
        m_Pixels[i * m_Width + j].green = palette[row[j]].green;
        m_Pixels[i * m_Width + j].blue  = palette[row[j]].blue;
        m_Pixels[i * m_Width + j].alpha = 255;
      }
    }
  }
  else if (pixel_depth == 4)
  {
    png_colorp palette;
    int num_palette;
    png_get_PLTE(png_ptr, info_ptr, &palette, &num_palette);

    for (int i = 0; i < m_Height; i++)
    {
      RGBA* dst = m_Pixels + i * m_Width;

      byte* row = (byte*)(row_pointers[i]);
      for (int j = 0; j < m_Width / 2; j++)
      {
        byte p1 = *row >> 4;
        byte p2 = *row & 0xF;

        dst->red   = palette[p1].red;
        dst->green = palette[p1].green;
        dst->blue  = palette[p1].blue;
        dst->alpha = 255;
        dst++;

        dst->red   = palette[p2].red;
        dst->green = palette[p2].green;
        dst->blue  = palette[p2].blue;
        dst->alpha = 255;
        dst++;

        row++;
      }

      if (m_Width % 2)
      {
        byte p = *row >> 4;
        dst->red   = palette[p].red;
        dst->green = palette[p].green;
        dst->blue  = palette[p].blue;
        dst->alpha = 255;
        dst++;
      }
    }
  }
  else if (pixel_depth == 1)
  {
    png_colorp palette;
    int num_palette;
    png_get_PLTE(png_ptr, info_ptr, &palette, &num_palette);

    for (int i = 0; i < m_Height; i++)
    {
      RGBA* dst = m_Pixels + i * m_Width;

      int mask = 1;
      byte* p = (byte*)(row_pointers[i]);

      for (int j = 0; j < m_Width; j++)
      {
        dst->red   = palette[(*p & mask) > 0].red;
        dst->green = palette[(*p & mask) > 0].green;
        dst->blue  = palette[(*p & mask) > 0].blue;
        dst->alpha = 255;
        dst++;

        mask <<= 1;
        if (mask == 256)
        {
          p++;
          mask = 1;
        }
      }

    }
  }

  // we're done
  png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

PACKED_STRUCT(BMP_HEADER)
  word  bfType;
  dword bfSize;
  word  bfReserved1;
  word  bfReserved2;
  dword bfOffBits;

  dword biSize;
  long  biWidth;
  long  biHeight;
  word  biNumPlanes;
  word  biBitsPerPixel;
  dword biCompression;
  dword biSizeImage;
  long  biXPelsPerMeter;
  long  biYPelsPerMeter;
  dword biClrUsed;
  dword biClrImportant;
END_STRUCT(BMP_HEADER)

ASSERT_STRUCT_SIZE(BMP_HEADER, 54);

#define BMP_RGB    0
#define BMP_RLE8   1
#define BMP_RLE4   2

////////////////////////////////////////////////////////////////////////////////

bool
CImage32::Import_BMP(const char* filename)
{
  BMP_HEADER header;
  int   NumPal;
  BGRA* Pal;
  bool  result = true;

  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;
  fread(&header, 1, sizeof(header), file);
  
  // if the file ain't 'BM'
  if (header.bfType != 19778)
  {
    fclose(file);
    return false;
  }

  // my guess this is a OS/DIB bmp header
  if (header.biSize < 40)
  {
    word temp;
    // seek to BITMAPFILEHEADER + 4(the biSize of DIBINFOHEADER)
    fseek(file, 18, SEEK_SET);
    fread(&temp, 1, sizeof(word), file);
    header.biWidth = temp;
    fread(&temp, 1, sizeof(word), file);
    header.biHeight = temp;
    fread(&temp, 1, sizeof(word), file);
    fread(&temp, 1, sizeof(word), file);
    header.biBitsPerPixel = temp;
    header.biCompression = BMP_RGB;

    // calculate and grab palette
    NumPal = (header.bfOffBits - (header.biSize + 14)) / 3;
    Pal = new BGRA[NumPal];
    fseek(file, header.biSize + 14, SEEK_SET);
    for (int i=0; i<NumPal; i++)
    {
      byte Red,Green,Blue;
      Blue = getc(file); Green = getc(file); Red = getc(file);
      if (Red == EOF || Green == EOF || Blue == EOF)
        return false;

      Pal[i].red = Red;
      Pal[i].green = Green;
      Pal[i].blue = Blue;
    }
  }
  else // this is a normal windows bmp header
  { 
    // calculate and grab palette
    NumPal = (header.bfOffBits - (header.biSize + 14)) / 4;    
    Pal = new BGRA[NumPal];
    fseek(file, header.biSize + 14, SEEK_SET);
    fread(Pal, 4, NumPal, file); 
  }

  // set up the Image32 now...
  m_Width  = header.biWidth;
  m_Height = header.biHeight;
  m_Pixels = new RGBA[m_Width * m_Height * 4];
  if (m_Pixels == NULL)
    return false;
  
  // seek to data in case of > 16bit bitmaps
  fseek(file, header.bfOffBits, SEEK_SET);

  switch(header.biCompression)
  {
    case BMP_RGB: 
    { if (!BMP_ReadRGB(file, header.bfSize - header.bfOffBits, header.biBitsPerPixel, Pal))
        result = false;
  }
    break;

    case BMP_RLE8:
    {
      if (!BMP_ReadRLE8(file, Pal))
        result = false;
    }
    break;

    case BMP_RLE4:
  {
      result = false;
  }
    break;

    default: // I don't have that kind of decompression!
    result = false;
  }

  // clean up and exit this
  delete[] Pal;
  fclose(file);
  return result;
}

////////////////////////////////////////////////////////////////////////////////

bool
CImage32::BMP_ReadRGB(FILE* file, int imagesize, int bpp, BGRA* Pal)
{
  // calculates the number of bytes it has to realign
  int reAlignBytes;
  reAlignBytes = imagesize / m_Height;
  reAlignBytes -= m_Width * (bpp/8);

  for (int j=m_Height-1; j>=0; j--)
  {
    for (int i=0; i<m_Width; i++)
    {
      switch(bpp)
      {
        // 1 bit bitmap has its own little world. ~_~
        case 1:
        {
          byte pixel;
          char bit[8];
          long k=0;
          
          // grab the image upside down
          while (k < m_Width*m_Height)
          {
            pixel = getc(file);
            if (pixel == EOF)
              return false;
          
            for (int l=0; l<8; l++)
            {
              bit[l] = pixel & 1;
              pixel >>= 1;
            }

            for (int l=7; l>=0; l--)
            {
              if (bit[l])
                m_Pixels[k] = rgbaWhite;                
              else // if (!bit[k])
                m_Pixels[k] = rgbaBlack;

              k++;
            }          
          }

          // now flip it!
          FlipVertical();
          return true;
        }
        break;
        
        case 8:
        {
          byte pixel;
          
          pixel = getc(file);          
          if (pixel == EOF)
            return false;
          m_Pixels[m_Width * j + i].red   = Pal[pixel].red;
          m_Pixels[m_Width * j + i].green = Pal[pixel].green;
          m_Pixels[m_Width * j + i].blue  = Pal[pixel].blue;
          m_Pixels[m_Width * j + i].alpha = 255;
        }
        break;

        case 24:         
        { 
          byte Red, Green, Blue;          
          
          Blue = getc(file); Green = getc(file); Red = getc(file);          
          if (Red == EOF || Green == EOF || Blue == EOF)
            return false;
          
          m_Pixels[m_Width * j + i].red   = Red;
          m_Pixels[m_Width * j + i].green = Green;
          m_Pixels[m_Width * j + i].blue  = Blue;
          m_Pixels[m_Width * j + i].alpha = 255;
        }
        break;

        case 32:
        {          
          byte Red, Green, Blue;          
          
          Blue = getc(file); Green = getc(file); Red = getc(file);
          getc(file);
          if (Red == EOF || Green == EOF || Blue == EOF)
            return false;
          
          m_Pixels[m_Width * j + i].red   = Red;
          m_Pixels[m_Width * j + i].green = Green;
          m_Pixels[m_Width * j + i].blue  = Blue;
          m_Pixels[m_Width * j + i].alpha = 255;
        }
        break;

        // If I haven't done the resolution yet, it will return false
        default:
          return false;
      }
    }

    // this is to fix the alignment problem
    int w;
    for (w=0; w<reAlignBytes; w++)
        getc(file);      
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CImage32::BMP_ReadRLE8(FILE *file, BGRA* Pal)
{

// error handling
#define check(i)       if (i > m_Width*m_Height) { return false; }

  int i;

  i= 0;

  // decompress bitmap
  while (i < m_Width * m_Height)  
  {
    byte code;
    byte value;

    code = getc(file);
    value = getc(file);
    if (code == EOF || value == EOF)
      return false;

    if (code > 0)
    {
      for (int j=0; j<code; j++)
      {
        check(i);
        m_Pixels[i].red = Pal[value].red;
        m_Pixels[i].green = Pal[value].green;
        m_Pixels[i].blue = Pal[value].blue;
        m_Pixels[i].alpha = 255;
        i++; check(i);
      }
    }
    else
    {
      switch(value)
      {
        // end of line, useless.
        case 0: 
        {
        }
        break;

        // end of picture
        case 1:
          FlipVertical();
          return true;

        // displace picture
        case 2:
        {
          code = getc(file);
          value = getc(file);
          if (code == EOF || value == EOF)
            return false;
          
          // displace by x
          i += code; check(i);

          // displace by y
          i += (value * m_Width); check(i);          
        }
        break;

        default:
        {
          int j;

          for (j=0; j<value; j++)
          {
            byte ccode;
            ccode = getc(file);
            if (ccode == EOF)
              return false;
            
            check(i);
            m_Pixels[i].red = Pal[ccode].red;
            m_Pixels[i].green = Pal[ccode].green;
            m_Pixels[i].blue = Pal[ccode].blue;
            m_Pixels[i].alpha = 255;
            i++; check(i);
          }
          
          // word re-alignment
          if (j%2 == 1)
            getc(file);          
        }
      }        
    }
  }

  FlipVertical();
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CImage32::BMP_ReadRLE4(FILE *file, int bpp, int NumPal, RGBA* Pal)
{
  return true;
}

////////////////////////////////////////////////////////////////////////////////

PACKED_STRUCT(PCX_HEADER)
  byte manufacturer;
  byte version;
  byte encoding;
  byte bits_per_pixel;
  word xmin;
  word ymin;
  word xmax;
  word ymax;
  word hdpi;
  word vdpi;
  byte colormap[48];
  byte reserved;
  byte num_planes;
  word bytes_per_line;
  word palette_info;
  word h_screen_size;
  word v_screen_size;
  byte filler[54];
END_STRUCT(PCX_HEADER)

ASSERT_STRUCT_SIZE(PCX_HEADER, 128)

////////////////////////////////////////////////////////////////////////////////

bool
CImage32::Import_PCX(const char* filename)
{
  FILE* file = fopen(filename, "rb");
  if (file == NULL)
    return false;

  PCX_HEADER header;
  fread(&header, 1, sizeof(header), file);

  m_Width  = header.xmax - header.xmin + 1;
  m_Height = header.ymax - header.ymin + 1;
  m_Pixels = new RGBA[m_Width * m_Height];

  int   scansize = header.num_planes * header.bytes_per_line;
  byte* scanline = new byte[scansize];

  if (header.num_planes == 1)      // 256 colors
  {
    // read palette
    RGB palette[256];
    long position = ftell(file);
    fseek(file, -768, SEEK_END);
    fread(palette, 3, 256, file);
    fseek(file, position, SEEK_SET);

    for (int iy = 0; iy < m_Height; iy++)
    {
      PCX_ReadScanline(file, scansize, scanline);
      for (int ix = 0; ix < m_Width; ix++)
      {
        m_Pixels[iy * m_Width + ix].red   = palette[scanline[ix]].red;
        m_Pixels[iy * m_Width + ix].green = palette[scanline[ix]].green;
        m_Pixels[iy * m_Width + ix].blue  = palette[scanline[ix]].blue;
        m_Pixels[iy * m_Width + ix].alpha = 255;
      }
    }
  }
  else if (header.num_planes == 3) // 24-bit color
  {
    for (int iy = 0; iy < m_Height; iy++)
    {
      PCX_ReadScanline(file, scansize, scanline);
      for (int ix = 0; ix < m_Width; ix++)
        m_Pixels[iy * m_Width + ix].red   = scanline[ix + 0 * header.bytes_per_line];
      for (int ix = 0; ix < m_Width; ix++)
        m_Pixels[iy * m_Width + ix].green = scanline[ix + 1 * header.bytes_per_line];
      for (int ix = 0; ix < m_Width; ix++)
      {
        m_Pixels[iy * m_Width + ix].blue  = scanline[ix + 2 * header.bytes_per_line];
        m_Pixels[iy * m_Width + ix].alpha = 255;
      }
    }
  }

  delete[] scanline;

  fclose(file);
  
  return true;
}

////////////////////////////////////////////////////////////////////////////////

void
CImage32::PCX_ReadScanline(FILE* file, int scansize, byte* scanline)
{
  int bytesread = 0;
  while (bytesread < scansize)
  {
    byte data = fgetc(file);
    if (data == EOF)
      return;

    if (data > 192 && data < 256)
    {
      int numbytes = data - 192;
      data = fgetc(file);
      for (int i = 0; i < numbytes; i++)
      {
        scanline[bytesread] = data;
        bytesread++;
      }
    }
    else
    {
      scanline[bytesread] = data;
      bytesread++;
    }
  }
}

////////////////////////////////////////////////////////////////////////////////

bool
CImage32::Export_PNG(const char* filename) const
{
  FILE* file = fopen(filename, "wb");
  if (file == NULL)
    return false;

  // create png struct
  png_structp png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (!png_ptr)
  {
    fclose(file);
    return false;
  }

  // create info struct
  png_infop info_ptr = png_create_info_struct(png_ptr);
  if (!info_ptr)
  {
    png_destroy_write_struct(&png_ptr, NULL);
    return false;
  }

  png_init_io(png_ptr, file);
  png_set_IHDR(png_ptr, info_ptr, m_Width, m_Height, 8, PNG_COLOR_TYPE_RGB_ALPHA, PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

  // put the image data into the PNG
  void** rows = (void**)png_malloc(png_ptr, sizeof(void*) * m_Height);
  for (int i = 0; i < m_Height; i++)
  {
    rows[i] = png_malloc(png_ptr, sizeof(RGBA) * m_Width);
    memcpy(rows[i], m_Pixels + i * m_Width, m_Width * sizeof(RGBA));
  }
  png_set_rows(png_ptr, info_ptr, (png_bytepp)rows);
  info_ptr->valid |= PNG_INFO_IDAT;

  png_write_png(png_ptr, info_ptr, PNG_TRANSFORM_IDENTITY, NULL);

  png_destroy_write_struct(&png_ptr, &info_ptr);
  fclose(file);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

double
CImage32::RotateX(double x, double y, double radians)
{
  double value;
  value = (x * cos(radians)) - (y * sin(radians));
  return value;
  //return ((x - cx) * cos(radians) + cx);
}

////////////////////////////////////////////////////////////////////////////////

double
CImage32::RotateY(double x, double y, double radians)
{
  double value;
  value = (x * sin(radians)) + (y * cos(radians));
  return value;
  //return ((y - cy) * sin(radians) + cy);
}

////////////////////////////////////////////////////////////////////////////////
